/*
 * Copyright (c) 1998, 2012, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package jail.java.io;

import java.io.IOException;

import blues.Log;
import impl.java.io.BluesFileSystem;

class UnixFileSystem extends FileSystem {

    private final char slash;
    private final char colon;

    public UnixFileSystem() {
        slash = getProperty("file.separator").charAt(0);
        colon = getProperty("path.separator").charAt(0);
    }

    static private String getProperty(String name) {
        return glue.java.lang.System.properties().getProperty(name);
    }

    /* -- Normalization and construction -- */

    public char getSeparator() {
        return slash;
    }

    public char getPathSeparator() {
        return colon;
    }

    /*
     * A normal Unix pathname contains no duplicate slashes and does not end
     * with a slash. It may be the empty string.
     */

    /*
     * Normalize the given pathname, whose length is len, starting at the given
     * offset; everything before this offset is already normal.
     */
    private String normalize(String pathname, int len, int off) {
        if (len == 0)
            return pathname;
        int n = len;
        while ((n > 0) && (pathname.charAt(n - 1) == '/'))
            n--;
        if (n == 0)
            return "/";
        StringBuffer sb = new StringBuffer(pathname.length());
        if (off > 0)
            sb.append(pathname.substring(0, off));
        char prevChar = 0;
        for (int i = off; i < n; i++) {
            char c = pathname.charAt(i);
            if ((prevChar == '/') && (c == '/'))
                continue;
            sb.append(c);
            prevChar = c;
        }
        return sb.toString();
    }

    /*
     * Check that the given pathname is normal. If not, invoke the real
     * normalizer on the part of the pathname that requires normalization. This
     * way we iterate through the whole pathname string only once.
     */
    public String normalize(String pathname) {
        int n = pathname.length();
        char prevChar = 0;
        for (int i = 0; i < n; i++) {
            char c = pathname.charAt(i);
            if ((prevChar == '/') && (c == '/'))
                return normalize(pathname, n, i - 1);
            prevChar = c;
        }
        if (prevChar == '/')
            return normalize(pathname, n, n - 1);
        return pathname;
    }

    public int prefixLength(String pathname) {
        if (pathname.length() == 0)
            return 0;
        return (pathname.charAt(0) == '/') ? 1 : 0;
    }

    public String resolve(String parent, String child) {
        if (child.equals(""))
            return parent;
        if (child.charAt(0) == '/') {
            if (parent.equals("/"))
                return child;
            return parent + child;
        }
        if (parent.equals("/"))
            return parent + child;
        return parent + '/' + child;
    }

    public String getDefaultParent() {
        return "/";
    }

    /* -- Path operations -- */

    public boolean isAbsolute(File f) {
        return (f.getPrefixLength() != 0);
    }

    public String resolve(File f) {
        if (isAbsolute(f))
            return f.getPath();
        String baseDir = BluesFileSystem.getDMSCCRoot();
        return resolve(baseDir, f.getPath());
    }

    public String canonicalize(String path) throws IOException {
        String s = impl.java.io.BluesFileSystem.canonicalize(path);
        if (null == s)
            throw new IOException();
        return s;
    }

    private String cname(String path) {
        return impl.java.io.BluesFileSystem.canonicalize(path);
    }

    private String cname(File f) {
        return impl.java.io.BluesFileSystem.canonicalize(resolve(f));
    }

    /* -- Attribute accessors -- */

    public int getBooleanAttributes(File f) {
        int rv = BluesFileSystem.getBooleanAttributes0(cname(f));
        String name = f.getName();
        boolean hidden = (name.length() > 0) && (name.charAt(0) == '.');
        return rv | (hidden ? BA_HIDDEN : 0);
    }

    public boolean checkAccess(File f, int access) {
        return BluesFileSystem.checkAccess(cname(f), access);
    }

    public long getLastModifiedTime(File f) {
        return BluesFileSystem.getLastModifiedTime(cname(f));
    }

    public long getLength(File f) {
        return BluesFileSystem.getLength(cname(f));
    }

    /* -- File operations -- */

    public boolean createFileExclusively(String path, boolean restrictive) throws IOException {
        int r = BluesFileSystem.createFileExclusively(cname(path));
        if (r < 0)
            throw new IOException();
        return (r == 0);
    }

    public boolean delete(File f) {
        // Keep canonicalization caches in sync after file deletion
        // and renaming operations. Could be more clever than this
        // (i.e., only remove/update affected entries) but probably
        // not worth it since these entries expire after 30 seconds
        // anyway.
        return BluesFileSystem.delete0(cname(f));
    }

    public String[] list(File f) {
        return BluesFileSystem.list(cname(f));
    }

    public boolean createDirectory(File f) {
        return BluesFileSystem.createDirectory(cname(f));
    }

    public boolean rename(File f1, File f2) {
        // Keep canonicalization caches in sync after file deletion
        // and renaming operations. Could be more clever than this
        // (i.e., only remove/update affected entries) but probably
        // not worth it since these entries expire after 30 seconds
        // anyway.
        return BluesFileSystem.rename0(cname(f1), cname(f2));
    }

    public boolean setLastModifiedTime(File f, long time) {
        return BluesFileSystem.setLastModifiedTime(cname(f), time);
    }

    public boolean setReadOnly(File f) {
        return BluesFileSystem.setReadOnly(cname(f));
    }

    /* -- Filesystem interface -- */

    public File[] listRoots() {
        try {
            SecurityManager security = glue.java.lang.System.getSecurityManager();
            if (security != null) {
                glue.java.lang.SecurityManager.checkRead(security, "/");
            }
            return new File[] { new File("/") };
        } catch (SecurityException x) {
            return new File[0];
        }
    }

    /* -- Basic infrastructure -- */

    public int compare(File f1, File f2) {
        return f1.getPath().compareTo(f2.getPath());
    }

    public int hashCode(File f) {
        return f.getPath().hashCode() ^ 1234321;
    }

}
