#############################################################################
##
##  named-digraphs-main-database.g
##  Copyright (C) 2021                        Tom Conti-Leslie
##                                            Reinis Cirpons     
##                                            Murray Whyte    
##                                            Maria Tsalakou     
##                                            Marina Anagnostopoulou-Merkouri
##                                            Lea Racine         
##                                            James D. Mitchell  
##
##  Licensing information can be found in the README file of this package.
##
#############################################################################
##

#############################################################################
# This is the NAMED DIGRAPHS MAIN DATABASE.
#
# This file gets read when the global Digraphs function
# DIGRAPHS_LoadNamedDigraphs() is called. It binds the global variable
# DIGRAPHS_NamedDigraphs to a record where each component's name is
# the name of a "famous" digraph, and its value is the DiSparse6 string
# describing that digraph (i.e. can be obtained with DigraphByDiSparse6String).
#
# The digraphs in this database provide a useful collection of graph examples
# that can be easily loaded using Digraph("name"). Some basic properties of
# these digraphs are also recorded in the corresponding TEST database, at
# data/named-digraphs-test-database.g.
# These can be used as checks to ensure that Digraphs
# functions are working properly, and indeed every value in the test database
# is checked when running DigraphsTestExtreme(). Every named digraph added
# to this database must also have a corresponding entry in the named digraphs
# test database for the extreme tests to pass.
#
# N.B. this database also contains named UNDIRECTED graphs, for which each
# undirected edge is represented by two directed edges.
#############################################################################

DIGRAPHS_NamedDigraphs.("diamond") := ".Cg@_Qq_E@J";
DIGRAPHS_NamedDigraphs.("fork") := ".DkgC|kgC~";
DIGRAPHS_NamedDigraphs.("tetrahedral") := ".Ca@_QqGE@J";
DIGRAPHS_NamedDigraphs.("utility") := ".EaGcbcbyGaQMQN";
DIGRAPHS_NamedDigraphs.("twotriangles") := ".Ea@ozRg_[Ms";
DIGRAPHS_NamedDigraphs.("sixteencell") := ".G`?K?a_CS?bH?OqDaGXAepA?W@D?Gg@EQ?`cJCOqDL";
DIGRAPHS_NamedDigraphs.("banner") := ".DkQ_QupI@J";
DIGRAPHS_NamedDigraphs.("cross") := ".EowCczB_QR";
DIGRAPHS_NamedDigraphs.("octahedral") := ".Ea@_WGxGsyGE@_bcbR";
DIGRAPHS_NamedDigraphs.("threetriangles") := ".H`?L@rH[tbx`?L@rH[tb~";
DIGRAPHS_NamedDigraphs.("bull") := ".DkAcbuoIQN";
DIGRAPHS_NamedDigraphs.("hgraph") := ".EoBccz?MQR";
DIGRAPHS_NamedDigraphs.("rgraph") := ".Eob_QNkGwCb";
DIGRAPHS_NamedDigraphs.("pentatope") := ".Da@_Q_QNW_WCgCb";
DIGRAPHS_NamedDigraphs.("longhorn") := ".FkHgyP^kHgyP^";
DIGRAPHS_NamedDigraphs.("egraph") := ".EkY_sypiBR";
DIGRAPHS_NamedDigraphs.("claw") := ".Ck@JJ?Q";
DIGRAPHS_NamedDigraphs.("net") := ".EkHMbRjARgs";
DIGRAPHS_NamedDigraphs.("isobutane") := ".MsCOyGRSpwIUru_aFPAYeN@Qu^";
DIGRAPHS_NamedDigraphs.("ethane") := ".Gk?`WaEZD_CJCOr^";
DIGRAPHS_NamedDigraphs.("paw") := ".CgWCkgWCn";
DIGRAPHS_NamedDigraphs.("petersen") := ".I`ACWqHKhhccTEgO`EKaRIYXHDP";
DIGRAPHS_NamedDigraphs.("square") := ".CaHJGaQ";
DIGRAPHS_NamedDigraphs.("moserspindle") := ".Faq_YLGCb}JI@gs_QN";
DIGRAPHS_NamedDigraphs.("butterfly") := ".Daq_QNWkgCb";
DIGRAPHS_NamedDigraphs.("kite") := ".DgXIANYEQ_b";
DIGRAPHS_NamedDigraphs.("housex") := ".DgWCgCbu`_Q_QN";
DIGRAPHS_NamedDigraphs.("eiffeltower") := ".FkJdIP^kJdIP^";
DIGRAPHS_NamedDigraphs.("fish") := ".EkgE@G}kgE@G~";
DIGRAPHS_NamedDigraphs.("trianglesquare") := ".Fa@ozp^a@ozp^";
DIGRAPHS_NamedDigraphs.("agraph") := ".EkgIRRjIAcs";
DIGRAPHS_NamedDigraphs.("antenna") := ".EgWcyLMgWcyLN";
DIGRAPHS_NamedDigraphs.("balaban10cage") := ".~?@E_GA?a?QD`wYC`g]D`GUF`hoOCXGRbPAMCWoQbpMHDw_Ua`UJDGgYa@aHEWkZb`aKEw{YbPeSDXWVd@aTEX[Zd`iaKIKphRMcKigliqq]Hh{hgAa`HxsnfAy@Nb}ALBiALRmBLruBLbq@MBfE_GA?a?QD`wYC`g]D`GUF`hoOCXGRbPAMCWoQbpMHDw_Ua`UJDGgYa@aHEWkZb`aKEw{YbPeSDXWVd@aTEX[Zd`iaKIKphRMcKigliqq]Hh{hgAa`HxsnfAy@Nb}ALBiALRmBLruBLbq@MBf";
DIGRAPHS_NamedDigraphs.("balaban11cage") := ".~?@o_GA?`_UCc?_JaOiKBwsMc@EQCpQWDh][EhmeDhiWexeVfXqGFg_^aQEIGgkcaaUHGGkbbAYLIG{kbaeNIwwibQuKHwS]`qQCFw[a`AEDGGWd`aM]HiClgambIX{fhQqcIi?glb}{PJ\\@nSUwOjk}mSAyOxksdq}\\JhcoeBUYKXWrfBJMTk|VsDbPU\\HYtTvSVLLZpc^IR[lKqCe@Vd}AWEE@WeUAWuQBXeeBXubo_GA?`_UCc?_JaOiKBwsMc@EQCpQWDh][EhmeDhiWexeVfXqGFg_^aQEIGgkcaaUHGGkbbAYLIG{kbaeNIwwibQuKHwS]`qQCFw[a`AEDGGWd`aM]HiClgambIX{fhQqcIi?glb}{PJ\\@nSUwOjk}mSAyOxksdq}\\JhcoeBUYKXWrfBJMTk|VsDbPU\\HYtTvSVLLZpc^IR[lKqCe@Vd}AWEE@WeUAWuQBXeeBXub";
DIGRAPHS_NamedDigraphs.("barnettebosaklederberg") := ".e_OGC_GFa`gs]QIcAwcUMMbBwcUVIgdY@QWRecilOtRhRbk{xAKfWH_xBJfgXW~ALoRQRL?_OH?ONDBPg{cTGDpGk[\\EFpGkmTPIsAcofLHTY`ifQfFXxqCXMoR@qEVNOqp}CZ_ecf";
DIGRAPHS_NamedDigraphs.("biggssmith") := ".~?@e_GA?b?QDBGQC`WUE`gaFDGeFDgeGaWgWapiJEWkRa`IGCWWO`o}IBgqQHGuTHhiPgyEXIx]RfQu_dAaadh_merI[frY]IX?ibrAMKyQwNJYeiId?lJAtmSIrNymljcayjsiiOyqpPIUfPzHNlsuzNjo}nsBMSk|OoSJBPKdPnSjDPklRrCvFQGDSTWDUTwHWUWL[VwL\\VgHYU}Y?_GAK@GSK`GQD`WYEaG[SaW[UaWaHa`aJEgkXapMICg_P``AFBwgMbHGcbXSeehEbgXcjdxM\\JYASIIIUeAyZKhq^LhwhcAiNKGwrhJ_{liYgiSAskJUxOjK~iyumQJinQihBjJDChY\\Fkc}vRZk}nBy~OKxQrtB@OkLCqTE|QkTEqtNKR[\\G_TPT_TXV_d`X_tp^_tt]_dhZ";
DIGRAPHS_NamedDigraphs.("bislitcube") := ".Gg?Gg@F?OqpCKeI?AI?PoCKkPBH";
DIGRAPHS_NamedDigraphs.("brinkmann") := ".T____gEFbc`GHaGIbFcELaDJM`DKLaEH`FIPcDFHbDEIt____gEFbc`GHaGIbFcELaDJM`DKLaEH`FIPcDFHbDEI";
DIGRAPHS_NamedDigraphs.("brouwerhaemers") := ".~?@P_G?@_G?B_GA?_GA?_GA?@W?G_?Y?@w?J_?q?Ag?H_OKHA_kK_OKDB`KS_OKEC@GS_OKFBpCR_OKGBPCQ_OOD@_cP_OOD@ogQ_OOEA?oR_OOFA?kS_OOLB_{ODWCIBP?PD@q@AOsNC`K\\_OkMC@GREgCKB_{PD@m@@O_HB@?W_OSGA_kNDwCE@okKBPY@@_[HA_wX__KKBP?RE`saHAYA?okLBpOZFACbHgGBA_wNC`g\\FqKd__KHB`?PEpo_HAUA?oSE@o_TFq?`GgGCCPGRD@wbHASeIwGC@OkKB`[WFq?e__OEA_kOD`c_GaMA@?[HB?{UEP{`HGGCA?cIBP[WGQGd__SNC@CQDPW[FQWo___NC@KSDPcYEqSl__WLB`CRDP[ZFQKn__[LB`GSDP_YFAOm__SEA_oSE@c\\FaCi__SFAOkRDpc[FaGh__WGAOkQD`_ZF`{f__[GA_oPD`[YFa?g_oSEApGRFPw^GaKeIa{oKbOw_oSFB@CSF@w_GQOeIQwoKbKv_oWGA`CSEpw_GQKdHqsnKROu_o[GAPGRE`w^GaOdIAsmKRKt_oSHA_{OEPo\\GQGdJAsrLB[w_o_JB?{OD`gZFq?eJB?rLBSu_oWHB?sME@k\\FqCcJAwpKbWw_o[IAosMDpg[GAGbJA{pKbSv`?cJBP?QD`_[GaOeIQkqKrSwMwOIB?sNCPWVFQCbHagjKbOuLrqCA_oMC@OWEPg_HASgIrCrLb[x`?cJB_{RDpcZFqKdHqkpLBSwMgODA@?PCpSVGAGdHa_hJrOtLr}C@O_NC`OTE@{`HQWfIawrLb`?`?WFBPKSDPc`GaKcIQglKR[wNgOE@owPC`SUFq?bHA[gKBGtLbuDAOgMCPGVE@o\\FaSfIAkmJrG}Pw_JB?sRD@[WE`k]HaciIqwnKRtG`_cKC@CRD`cZFPwcIAcjJR?sOCYFA_kNC`OUEPg[FaKfIaklKBK~PWSEAowOD@SWEPk[GA[hJA{oMBpA`O[KB_{RDP[XE`s^IAgkJb?vMsEEA?gLC@GTD`_YFQGgIaolJrWyPG[GAOsNCPSUDpk[GQ[hJAsmLRdBsWA??WA??wA?_GA?_GA?_?U?AG?E_?]?Aw?K_?i?AWCBAOgJBGCB@OwRDGCB@`?QDGCB@o{PCwCBA?sPCgCC@OWHCWCC@O[ICgCC@__KCwCC@o_JDGCCBOwNC@U@A_sOCPO[_OcLBpGRFWCJB`?QCpi@B?wNCPOZ_OSGAOoOEGCDA?gJBp]@@_[JB?sU_OWFAOgMEWGBB?sOCpg\\GaOe__KJBO{SEpo`GqYA?ogMBpGYFP{bHWGBAOwOCPk[GAOd__KD@_[GDP{_GQIA@@CQCpO]GqOdHamA@?SJB?wVE@{_HgGC@_gJC@WXGAGb__OFAOoND`c^GQQA@?_HA_sVEACaHWGDBp?PC`SUF@seKGGGBp?RD@SXE`kdJWGEBOwPCpSVEpsbJwGFBOwQD@SWE`ocJgGD@_gKD@_XFPw`IgGD@ocJCp[XF@waIWGEA?cJC`WWEpw^HwGFA?gKCPWVE`w_IGKD@_kQCps]FqGbHagnKBGsMGKD@ooPD@o]GACcHacmKBGrLwKEA?gPD@k]GACbHQ[lJrCsLgKFA?cQCpg]FqGcHQ_lJbCrLWKDAOgNC@c[FQCaHQolKrOvMGKGAooNC@WYEp{_HaooKrOtLgKEAOoLB`_ZFP{`HAomKRGuMGKFA_kLB`[YFA?aGqonKRGtLwOHAosOC`WWFAGcHacjKbKtMBmCA_oLBpCUDps`GqWiIrGsLb[{`?gKB`?SE@cYGAOdIAkpKrWvMWOHAowNCp[XEp{bHQ[jKROtMBiC@O_OCPKTDq?aHQWgIQ{sLR[~`?SGBpGSDP_^GQSeHqgmKrWwOGOE@osRD@SXGQGbHAciJRCvMByC@_[MCPGTD`{_GqOfIB?qLRW|`OcIB`CQDp_[FPwdHq_jJa{qNc]GAooLCpOVE@gZFaWhIakmJrC|QGWHB@?PCpWXEps]HA_hIqsoLC@E`ogJBpGSD`cYF@wbHqgjJR?rNsUD@_kMC@OTE@cZFA?fIQonKB_{OgSFB?wNCpSVEPg\\Fq_iJAwoLrl@`__IBP?QDPWWE`saIAgkJQ{uMcQFA?cLBpCTD`[ZFACfIQolJbSxO~";
DIGRAPHS_NamedDigraphs.("butane") := ".Ms?GhcJ`IjFUru_@DK`[HTWyu^";
DIGRAPHS_NamedDigraphs.("celminsswartsnark1") := ".Y___eDgCDefHfcdKcEfniOgHjbNRaLMbJO`PQaR`KWy___eDgCDefHfcdKcEfniOgHjbNRaLMbJO`PQaR`KW";
DIGRAPHS_NamedDigraphs.("celminsswartsnark2") := ".Y___dCcgFchfIeGdhLkLdIeFjgQbNP`OQaRaMbV`UWy___dCcgFchfIeGdhLkLdIeFjgQbNP`OQaRaMbV`UW";
DIGRAPHS_NamedDigraphs.("chvatal") := ".K`ACIQJCkqENGXwcKeG`dxACGScUHXcK]OrpGXKPBJ";
DIGRAPHS_NamedDigraphs.("clebsch") := ".O_____`A`B`C`DaBGHaCFHaDFGbCEHKbDEGJcDEFIo_____`A`B`C`DaBGHaCFHaDFGbCEHKbDEGJcDEFI";
DIGRAPHS_NamedDigraphs.("contiguoususa") := ".pb_G{gRM@PgsuIPaQywk`LIrAkCXEFrqDmEIMp?hGfXeERQAOHFO_w_[a`crhyQILfRqI_VMRQHag_jHdDSk_SIYhWofTQq`\\KgSoQ`h?`SRrXoycjFCq\\QjVSQPK}cRjd~EM?bq`KwDEbRWhEHJjaqCsjKIoPcW^NGUwWgzCAca\\eWXLGH?c\\ABa@qIERMfhGgu\\NGi@[xLGeIaAkeSTQq@OhidbA\\TJIDsq`R@IEcADPNLfBiQk[RHtIl\\PHCryPMmV~";
DIGRAPHS_NamedDigraphs.("conwaysmith") := ".~??~_O?K?B??o?S?D?@`g?KJ?A@O{?GECR@[mVOFrx|A`PISQ\\GeThDAsqXLLRRKws[fER[kWLFERPkw\\NRq@CcRIDQp[pDCap`KlEDAp`OjFCaphKm`cA`WsgWPP_wk[TJgc[SOJFDbADH@@Cqxc|@@DB@gz@@cqpk{f__q`Sw\\SOO`SmXMHtSCGUKEbY`RDCBq@cxEBbq@gvFCbqHgymbA`xGu]VQ@O{aXNJUCSQNHFBit^AAbA@Kq`RjWOWWOIEcQ`[r``ph?eZPiTsKOLGDBaPSlYoowwaTMhTQ}EGFCQpxKhWlwOS[QKFSq`dCCbQP[wbTkg_gWPIeSQlbCDBQP_ucUKFSOQKGdbQDWn\\o_owcVNHSz@r~_O?K?B??o?S?D?@`g?KJ?A@O{?GECR@[mVOFrx|A`PISQ\\GeThDAsqXLLRRKws[fER[kWLFERPkw\\NRq@CcRIDQp[pDCap`KlEDAp`OjFCaphKm`cA`WsgWPP_wk[TJgc[SOJFDbADH@@Cqxc|@@DB@gz@@cqpk{f__q`Sw\\SOO`SmXMHtSCGUKEbY`RDCBq@cxEBbq@gvFCbqHgymbA`xGu]VQ@O{aXNJUCSQNHFBit^AAbA@Kq`RjWOWWOIEcQ`[r``ph?eZPiTsKOLGDBaPSlYoowwaTMhTQ}EGFCQpxKhWlwOS[QKFSq`dCCbQP[wbTkg_gWPIeSQlbCDBQP_ucUKFSOQKGdbQDWn\\o_owcVNHSz@r";
DIGRAPHS_NamedDigraphs.("coxeter") := ".[___fCdechIdeKfgMgfchOePdQ`IKaLMbJN`OPbSTaQR{___fCdechIdeKfgMgfchOePdQ`IKaLMbJN`OPbSTaQR";
DIGRAPHS_NamedDigraphs.("cremonarichmondconfiguration") := ".NiAPYDe]HcOGa?bpCTSqHUqGrIV}gHDgUXweP?aGANCPTRGdZGbKh^";
DIGRAPHS_NamedDigraphs.("cricket") := ".DkgCbuq_QN";
DIGRAPHS_NamedDigraphs.("cubical") := ".G`ADOaHCXHBHaCGTAGcPccKf";
DIGRAPHS_NamedDigraphs.("cuboctahedral") := ".K`?KGBkCkqqIY`RpGXK`fPqC?o_MoQrJGhiDNC`cqE\\F";
DIGRAPHS_NamedDigraphs.("dart") := ".DkQ_QNZCgCb";
DIGRAPHS_NamedDigraphs.("decane") := "._jCq`U_PHO_[?AWa@RCWMXdArS_QZfBqxuWLJFKDWeSIsAHICB_?RCOIWb@rKgUYcAR[w]VMr@hWx";
DIGRAPHS_NamedDigraphs.("deltoidalhexecontahedral") := ".}i??KKNABOokGHA@p@CWRFCpPC_^HFP@SWVFEpPcoZIDrqCwbNHRa[g`LHrASkb@HOQ[?`?GtCyd]piHdAx`UKGCIUiVPGsd[oXLFYhWqZMUbq@KilfsI\\Z\\NGtAqw^PIT]TSlWK[alWmV~i_?BBBo_sKJAAO_[OPECppKSPGFqPsOTEDppkSXKEq`[{`MGrqSwfIGRQ[odJGoQSCf?GOALPMhVkYaXOmWTbA@AdidsQLHVKERPuiUKerdg{_RIjX|AfUvRqLOkmFsQdVdTJUBFGjUJd~";
DIGRAPHS_NamedDigraphs.("deltoidalicositetrahedral") := ".Ym?@GHaCIKbFJLaEILbDJKaBKLeFHLcDGK`FHJ_EHI`DGJ_CGIym?@GHaCIKbFJLaEILbDJKaBKLeFHLcDGK`FHJ_EHI`DGJ_CGI";
DIGRAPHS_NamedDigraphs.("desarguesconfiguration") := ".Ig?SPqF?XBPCSs?cI\\Ei?DC[`oEOsPDL?HAfP";
DIGRAPHS_NamedDigraphs.("desargues") := ".S___fCDEccededaGLbGKaHJbHI`IJ`KLs___fCDEccededaGLbGKaHJbHI`IJ`KL";
DIGRAPHS_NamedDigraphs.("disdyakisdodecahedral") := ".Yk?HI_GJcDFbEF`BGaDH`CIaEJ_@GIKLOQ_AHJKLPR`BCFMNOQaDEFMNPRbEGJLNORcDHIKMPQyk?HI_GJcDFbEF`BGaDH`CIaEJ_@GIKLOQ_AHJKLPR`BCFMNOQaDEFMNPRbEGJLNORcDHIKMPQ";
DIGRAPHS_NamedDigraphs.("disdyakistriacontahedral") := ".}n?BXq?YMoOgYACBo`?mEHDPp?qGHEp`WyIIFrAHY[OJrq@aYPKoaPQERIsapigVMTR@meXMoAPOs[NFtax^?HdRXs{^UkEKCGDCacADEdRoOo[OJOGSQPLCBb@hDAcQiD[SKMFcCAXMhTO`?ocUPHDAp\\BCbQXdEdTjUKKSNIeCY\\SlWO`WwgVPHdIp^PJERPs}gTjeL?mWLfBqdSnW^jo?u[_EbkCIE_`?{GOJ`aPS[OKaAPkWUMaa`{oaUfCA{{_WecRKGcS`cqlGkYiDrdSoZheRk?cSLFBp|WmVoAXSu\\NFtj@b@A@PGh?`PhS{CKFCAsADCcRP?woYPOhCi`VDBB`x@?eRiTSGOKHDcQPOkVOpGseXPhTYtbBDBqh`EfTJUCGUMIDsQXQkVsQpcs\\NiDYxbOJeBXo{hTJuF";
DIGRAPHS_NamedDigraphs.("dodecahedral") := ".S___fCDEceGcdIdeK`GaIMaJbKObL`HQs___fCDEceGcdIdeK`GaIMaJbKObL`HQ";
DIGRAPHS_NamedDigraphs.("domino") := ".Eaq_xHMaq_xHN";
DIGRAPHS_NamedDigraphs.("doro") := ".~?@C_GA?_GA?_?Q?@W?E_?I??w?@c?uMCG}NbPMMCgsNC@IMBpCRbOwSDX?Pc`SXcpOXEgCA?osMBwOIB?wOC`qEA_kND@S[`OkKBPCRFGGC@`?SDpq@@?SQCp_[_oSECPSUFGOD@`WVE@s]Fw[GAPWVEA?`Gg[HA`?SDqMGAOkPDPWb`o_KC`KWGwOHB@?WEaYCA?gQDpce`O[JCpWXHQeEA?gTDpkf`_[JD@WYHqaDAOoPE@kdIwgJB?sMBqSeHwCE@p?WE`saJWGDAPSVEpw`IWCEA@CWEp{_IBAB@?_RD`c^GAqA@O[QDpc\\Gakq_oOHD@WYFaCiJwGEB?{OEp{dJBM@@?kMCpg]Hqcp_oWKBpCYFQWjKbYB@OgLDPc\\Hass_OSIBPGZFa[iJrUA@?kMD@c^HQ_oMGCHA_{OEq?hLB]AA?kLDPcaJBCx_OcJBpCYGagoLWKFA_wSEQ?jJrW{_o[KB`KYGQsrMbuAA?oLC`k`IBGwMwCA@owOCaOgIQw{NWCBA?sPCqOiJQwzNgGBAO{SDQOjJAw~OKQ?_GA?_GA?@G?D_?Y??g?B_?EOBWwObw}LCwwQbO{OCgwNCPMLB`OTc@EQDPeRD@cY_OGBBOwN`?gKB`?QFGWIAo{SDPqDAooLCPK[__OEC@OVFGCC@PGRE@qB@OWPDPW[`?SED`[WFPw^`o_HD`[WGACa`ocIC@OVGw_HApCTDaMFA?oQCp_b`?cKC@_YHgOGA`GVEQYD@okRD`cdIWWGA`SVEq]E@okSD`gfIGSHB@CWEqSja_kKBOwNHQWf_OWFC@_YFQGl__SHDP[ZFaCh_OWGCP_ZFq?gKGKCA@KUEP{_JGGD@pGVEPsaIrIB@?cSD`g]GQgn__WKBp?ZFqSkKwCCAowRE`wfIREB@_oNCPg\\HakqLgKDA_sTEPseJRQ@@OgLC`k]HqgnLWGCAowSEP{dIB?w_OcIBp?ZGAcsLwGGAosTEQGkKRe@AOkNCPgaIb?t_o[IB`OXGAknLbqB@ooMCpg`JRKyNWGGB?sQEqCgKb_z_OGFB`?QHA_hJbo|_OKGBPCRHAglJbk}__KHBpOTHAkkJb|?";
DIGRAPHS_NamedDigraphs.("doublestarsnark") := ".]___eDccfedgIhJdFcEgJhIsKRlQrq`UVbNVaMU`OPbRTaQS}___eDccfedgIhJdFcEgJhIsKRlQrq`UVbNVaMU`OPbRTaQS";
DIGRAPHS_NamedDigraphs.("doyle") := ".Z____eDdjGHgIeHKfIJbGcGcFbE`KQaJP`LOSaMNRbDHOcDINaFHQ`EIPz____eDdjGHgIeHKfIJbGcGcFbE`KQaJP`LOSaMNRbDHOcDINaFHQ`EIP";
DIGRAPHS_NamedDigraphs.("dyck") := ".__OGC[IEa@g_UUIdQhCeNJFR@{k`KGRPss^CbPWHCnAHDwPOpBJEWHKrBIeKCA@?b`OsOLCAqpSiTGcpxWyWNdcH`AYMebwcYJ@GdwPGnAIEGXWr@HeWXSp";
DIGRAPHS_NamedDigraphs.("edgeexcisedcoxeter") := ".Y___dCgEcfdGlEHgKfdNfKeIcbQRaPR`HK`IJbMOaLNy___dCgEcfdGlEHgKfdNfKeIcbQRaPR`HK`IJbMOaLN";
DIGRAPHS_NamedDigraphs.("ellinghamhortongraph54") := ".u_OGC[LEdAXL?XIFCHW{`HcCwxMWTeDGsOkWadrCOJLFRP{k\\IFp@CSNZmFzhz?`pH[YQZasy\\EbNe`Id{LUn_itkDSk`JTsLWmzOGCAMEbQ`Ke_KdBaCk]OcqA[[fKIrAcYGUKPQxaGDebhg}UMdBw_aIFlvB|s|_OwcmLHLpY\\MbPfrOdQ}EjVoTYuAiUOdiyEkV^";
DIGRAPHS_NamedDigraphs.("ellinghamhortongraph78") := ".~?@M_GA?`OQCa_eHb_gLaouKbGiJbWqQDXKTc@QPDG{Qb`MHcHE[FXo]ePuWFhWZdpibHH{agAEGiIaFiQmiIy{higSmJwWmKGWlJwSkKGaG`JSvlb]mljUyMzg{mRqwMzx?nR}F`yUfhgDDPwHDPgLCPwLBPgHAPGD@O{y?_GAD@GQIAWeMA_uJBWqKagmLbHGTcpUODHCSbpIMCweOcXo\\f@yXFX_]d`mVEiKcfqI_GWagiG]hIygjjqei`Qwn`awo`asn`QooaGaClR]uLyyulZgzmbqxNJ_zncA|Nw]FhY]e_STF_cTE_sPF_sLE_cHC_SDB";
DIGRAPHS_NamedDigraphs.("errera") := ".Pa?_A_A``D`D`E`FG_B_CIaBFHaCGHKdEIJbDFIKMcEGJLMpa?_A_A``D`D`E`FG_B_CIaBFHaCGHKdEIJbDFIKMcEGJLM";
DIGRAPHS_NamedDigraphs.("faulkneryoungergraph42") := ".i_OGCSHCcaGwaKHaa`[wPFFQWpAIFaSQ[w`Redh|GpLetAtSpYlUr|mm`kfgPtF@MGgPyE^ROSKL?gtOGCAICaQPC[PECpPOm[GbbhKW`DBpIHM[OhrQs}cWerY`YiWlUjX}vVOuRsGyb_fCSG|BNhgIEE_S^";
DIGRAPHS_NamedDigraphs.("faulkneryoungergraph44") := ".k_OGCWJGBpx_mTEEQxO}QMFqQC_YQapG`QZIDqA{SXEBtAkQs[jeJluvXPTSLX@BNgwHxHAQosQaA``GDUq@?_P_k_NFeAxSWXJdBxGw^HGQ@hIJCaDHkgVGJpPcWNSIpJPqmWmvZ\\dDTOtcCK}b_fccHJBPIGIEC_T^";
DIGRAPHS_NamedDigraphs.("fivesquares") := ".S__cBbfEeiHhlKklMiJfGcD`As__cBbfEeiHhlKklMiJfGcD`A";
DIGRAPHS_NamedDigraphs.("fivetriangles") := ".N`?L@rH[tbyRdVU[Z}C?sFKdrUNhMT\\Xpn";
DIGRAPHS_NamedDigraphs.("folkman") := ".S____j@DEF`DEFaDGHbEGIcFHIcFHIbEGIaDGH`ABCs____j@DEF`DEFaDGHbEGIcFHIcFHIbEGIaDGH`ABC";
DIGRAPHS_NamedDigraphs.("foursquares") := ".O__cBbfEeiHhiJfGcD`Ao__cBbfEeiHhiJfGcD`A";
DIGRAPHS_NamedDigraphs.("fourtriangles") := ".K`?L@rH[tbyRdVGOBO[qVLW}cxT";
DIGRAPHS_NamedDigraphs.("franklin") := ".K`ADpcJK`gdLGhgaHqCGVEOkqEaSsaeaGf";
DIGRAPHS_NamedDigraphs.("fritsch") := ".He?K?b_G[@CJCOqpCOp{r?E?PoCM?aDaGXWaGW~";
DIGRAPHS_NamedDigraphs.("frucht") := ".K`A?icgUtwdPKyGaHqCGAiQ`ZVaTCrgaGf";
DIGRAPHS_NamedDigraphs.("gardner") := ".~?B]_WA_gC_cG?{F`WI`SM@kLaGOaC[A[[bk\\cG_cCcC[bcwecsmDkmdsqEKpegsecxFCUFKvFKYew{fl?DG~f|BDXAgTFGtJGhHhLFHTIiHNiWWihShxSj@DILYIPUJKWIlVd`GiHQiCXI\\WiXSh|MKPbhh`K\\MKDJJp^h`^h`[JljKpihPidGigp`h@HglBLxog`mMDCLlEM\\iL\\kgxugHtMt@M|?M`xbOyF[tE{wNcUFCrNkVEX}awqe`~eKoOQBagnO[ZFc~dIFcyGdAEbWgcyFP[tFsiGS}PSiPKeciOQKcPyPciMcaLe?udXGeqTdYSc[aRIYcIWRSaQ{`QsSDm`DclRskRk_bybSc]SQcby_SK]R{YB[[SD[SM_tYkteitqnaQnaIlaQoaImU\\VLCRTDVUSRUKO`ywVKMUyx`yu`qti@Qa_niA|aa{`kKWJA`Z?WSKV{JVtTJKTETXWsTWkI`JIX[GXJJ`JG`BFoLZb?XaoVxzPxrO_{EYjU_jSYsEYBR_jPYS??GC_GBTK??OXzsB?SD?cC`?F_{J@SI`oL`kPACOb_Rbc\\bk`CC_c_bc[fCsedoldsmeOpeKtEcsfGwaoxewxbSvFc|g?hF{~gWjGTAgxEhXDHLHgxIhTPH|RBDTIdNIdWGhPjPQIpXb@Ti{kHDPITObHRjDRIdNhpaK\\LKHbhp_hX]J|KJ|KJ`\\lXeLTILShDTEKLGHLDgXnMDCLpog`lgprlPjldFMt@MhugHvg@sNKYFOzegvf@{aoweX|awrNsVESsN{peAAO[TDyBbW{f{hO{fPCgOsZDCfOyJeg}dPAfqIdQHcsdQAPcaNQKdPscPkoEsjHCuQkjQcbcQXRS`RAYcQVcIUa_lsGkdi]da\\cC^SYcbqaSc^SA`bq^bOZba_ja`sEjTektUmT{QT{PTkQUCPTqrixgaYgiyqaYpaCNVAx`qvVKNUsMUlOISSD|OVkSVcL`b@WSJWBA`a~`Y}ihXagqjJEajD`SHXRJ`BHX[HXCGW}@j[WBKUA~NYNMYCF_rTYsDYbU_rOY[DYJQ_?@?c@?Yh_?ABN";
DIGRAPHS_NamedDigraphs.("gem") := ".DgHI@G|gHI@G~";
DIGRAPHS_NamedDigraphs.("georges") := ".q_OGCSHCc@xWaPHDq`cgVHEPw{sbNHRaL?dRJTAkWpDJqRSgrCmFZXvCDbVs\\xD`PwCUF@`oOtAeEiTocieAcSOSqmCfT[gCA@DAPH?]UGcQP[gXIDqPc]NLGrqSwbOHSqtOjEKPQ{ctIKpJ`uu\\p@Wt|F]PWS]@D`oW[CLOh`id[HIh_hDCDKj`HtV";
DIGRAPHS_NamedDigraphs.("gewirtzbipartitedouble") := ".~?@o_GA?_GA?_GA?_HSJB?sMBp?PC`KSaoqMBwwObPALBxKScPIPCxGSbOyNCh?Pc@INCWwQb`EMCwwSc@QNCw{Sc@MLCgsPbpALCwsSc`MPDGkLb?yJBgoLb@EKCgoSb@MKCGoNapEJCgkSapMJBwkO_oOUFq?`GaKrLGSED`{gIQgjLRYGAPW`IAsoKR`?a?gUGAckJrGvNw[HDaGcHakqMRuFA`WbHQ[iKRg}_oSUHQWlJa{zOWOEDaOfJAwoNCI@?`[WEPgZF@s]__SVGq[jJrGrOgGEEAGeIb?pKsE@@Pc`HacoKbO{_OWYGA[gJrCsMwGBEaCdIQopLSIA@@c_HA_lKbT@_OKWGqOjJRCuNGCCDqGdIaoqLbmAA@kiIqolJbs}__[[GACeHqw~OGGIF`{bHAcoNSAAAPs^GaSgJrw~_O_[GaKmJr?xMgCFEqOdIAcmLra@APw^GQ[jJB[y_Og\\Fq?eIaswMWcIFPwpKbKsLRYEA@c]GAkrMRp?`O_YFQCiKrgzNwKIDpoeJBKyNBuCAP_[HqsrMRk}`O[WFQKgLBd?OWWFDpwaIROyNsIBAPc[HA{sLs@A`?gYFASoLB_~OWSIE`kfJRSvNCAEAPcZHaotMBk~_o[VFQ?jLR[zNgOFE@w`IbSwNBuDAP_ZHR?uMbtA`_gVEqOnLbc}OWKGEPsaIRWwNSECA@g]Gq_uLrxA`o_ZFBCqMrp@OgKEE@g^Jb_yNcAC@P[XFqwvMRs~{GA?_GA?_GA?_GATAooLB_{OCPGRDGkKb_}MCGsObO}RDHCQcPMQDGsMbpIOCX?QbpEMCgwPb`MMDH?SbpMNDH?RbPILCW{ObPMLDHGRcPQJBWoMaoyKBWoPb@IKDGoRb@AKBwkPapIJDGkRao}JCGKCD`{_GQGbKrQD@`W^IAciIrSua?cUGQ_lKBCwOG_IDa?hJA{qLr}FAPWaHAWjKbc|`ogUGqSfIbCyNgKDDaSeJQwnMsEC@`WcHqomKBpA_OGVE@cYEpo\\FgGDDqKfIq{qKsIA@`_aHagoKRL@_OSXGQWhKBGsNGCEEa?fIA{pLBmA?pg`HQckKRTA__OXGAOgJRGtOWCBEAKcIqspLbq@@@[aHQgkKbWz___ZIakkJQw|NgGFFA?`Ha[mNsAAA`w^GqOhKBt?__c\\FqGdIA{}NwCGFAGbJa{oMRi@@pkcHQ_hJb[w_Oc]FqCfIqovMgCIFP{_HaglMBeHA`s]KRGrLBSu`__XFa?jKrc{OGSGE`s`IbKyMr}BA`[[HaorMbo|`?cWFA[lKrczNgSFE@sbIBOxOCEE@p[]GacsMb|A_ocXFAOnLB\\?OgOIE`odKBOwNsEDA`gZHqstLrp?`_cXEqWkLR_zNwKFDps_IrSvMryC@p_]GQgtMBo|`OcWEqSoLbg|OgWIDpkcJrWxNcEBA@c\\GacuMBt@`?_YFaKgLb[}Og[GEpopKbk{OSIB@`_YFqwwMbx?`?SVEP{mLrc|N~";
DIGRAPHS_NamedDigraphs.("gewirtzcone") := ".x_OGCA@?_OGCAAA__wGaAF_`gGQAD_agGeCBEbgO_]R`@pxICEGCWOgYP`APpACCECgOSWR`aPXClBDAq@UGDDbqcWSJFEwocULLPp?kWVa@pXKpDCAqPeEEHCr@c{```px?oZOGwW_]PKebyIECEcqp[}b``PpGiVOGhGgWNIDsQMKFGcq`gybbAA@GgZMGg_caQIDRyDJDDCAXOk_Ohh?g[RJeb`xJFCbQP[u\\NHgwgWPIeRq@MjcAP`?kWNFtAiGIEbrHkw`SIggc[NKEbiDMja@`_wiWMGti}IEEBQpcyaUJg__[PJEri@Gha`ph?iYMFsae?@@?o_SKFCAPOkWLFBq@CcRIDQp[oXLEr`s{^OGSQLGdRHtAdSjUJTq|_pXKubTkv{oGCA@?_OGCA@@@OO[CP@BoOsCH@AoPSCRA@bPsGONHo_w{dABCAKGSLGo`Gw`AABASGIKHopGkaU`a`X?jCAapxQKIDbb[WQJEegw_UKJp?wkeWaa@XGrBBCaX_q]Ooow{_WLgC[KONGeRP|DBABQXWm^PoogwcTJgCScSKFdAyHFEBcQXOs\\Pp`@?cSLfCSOQPHDAh|Adaaa@KgUOGSs_SMHdrPo{dbaPhGmZMfcs[SKGdRHx?fTq@Go_UKFby`TCDBPxcu[OiDSSQMFeBPtAfTp?oo[TKFCYt^DBB@hWq\\PJDsOOMGdbXt?cSpOws_TLFByPR??__WOIEBa@GgUKEb`x?aQHdAhWmWKebXoy]NgCIHEcQhcy`QiTjDix]oWkeZPiuZ~";
DIGRAPHS_NamedDigraphs.("gewirtz") := ".w_OGCA@?_OGCAAA__wGaAF_`gGQAD_agGeCBEbgO_]R`@pxICEGCWOgYP`APpACCECgOSWR`aPXClBDAq@UGDDbqcWSJFEwocULLPp?kWVa@pXKpDCAqPeEEHCr@c{```px?oZOGwW_]PKebyIECEcqp[}b``PpGiVOGhGgWNIDsQMKFGcq`gybbAA@GgZMGg_caQIDRyDJDDCAXOk_Ohh?g[RJeb`xJFCbQP[u\\NHgwgWPIeRq@MjcAP`?kWNFtAiGIEbrHkw`SIggc[NKEbiDMja@`_wiWMGti}IEEBQpcyaUJg__[PJEri@Gha`ph?iYMFsafp?_OGCA@?_OGCCD@@oPCCN@BOOcCJ@DOPKGELFO`?{fABbqSGKOGo`OsbACbaCGGKHO_gofBCaqHYEIDcAkOIJFdGogUMLp`GkYYba@WonCBaqXaIGDcbKKKQHeBHxBBBbq@_u_Pop?{aXLFsSKGLHdax|FBAbaPSm_PQPOo]SJgc[WMPHdBPtFECCAPOu[PP@HCcSIfsIUIIGCq`X?`RQ@OweVLFBqUMHEcaxky]RPpOoaTKfcA\\VGCbA@Wo]NiDSOSLFeRXpAgTPPGw]WLFSI\\VCBB@pSo[PjT{SKKEdbHtCkVP@?waULfSAPRDBbQ@Ss[NhDN";
DIGRAPHS_NamedDigraphs.("goldnerharary") := ".Jk?XGaJ?GqeN?PAeMaG`cK\\A\\u?KcPD_CXRF_G`RFPCOqEM`N";
DIGRAPHS_NamedDigraphs.("golomb") := ".IgMXG@F?PGaJ?GbFPi_xc_C[@CaGk?aK\\F";
DIGRAPHS_NamedDigraphs.("gosset") := ".w_O?K?AA_?OOM?@@?oc?AA@`G?CCBA@g?CCBA`g?CCBA`pC?AAA@Ooa?@@@?g[QI_?OOWMGCa`[?ABA@Oo_T??_o_SMHDbW?CEEBa@GoYM_?O_WMIDb@gw^??`P?cSJEBPo{`?@?o_SKGDBW?GECA`pGk[Q_?_WWMGCb@xGf?@@?o[SJECAPKh?@@P?cSJECQPKgT_?o_WMLFBq@GeSIo?WSOHEb`xCcRIDa{?GDDApgw_PHCqhWmW_@`?gWLFcAHGgTJDr@e?FCap_w]OGcq`SkVKERTsAA@`?gWOIEcbcCCBA@OwcUMHfBkCCBB@p?cWNIFBhyAAA@_wgUKGDR`s{^___g_QIDbAHWw\\NFsCCECB@pgw]OJfBhw}___og_QLFBqH_w\\NFsIIACAa`Ws[OGeR`s{_Ogc[CKGDB@g{_PLFBh|?`PGscCMHDb@o{_PLfBp|?`PGsaUCBA@_xGeSIdr`s{^OGgOKIGCcaXOkWMFRp|AbRo__SSJHCqhWq[MfcADGfSO_o_SKHDAhWs[MfsADIfSIWO[QJECq`SkZMFby@AeRiDIiECAbPpGeVKER`s{aPhCy`RBBA@g{cSJeBPoy^PGsi\\OiUOowc[NHdAx_u[NFsQLKfSItauGEDBQ@GiVKeb`t?aQHSydSkUp?wk[OHdQxcu[NGCQPKfSitax^DCappCeUKERXo{`PhCq`QjUJeCSOIEcQPWoXLFBiDEcQiDIhWlVkWo[WNGDAh[sZMFsAHIeRidYt[nWQ@Go]PIDb@gu[NgSYTKgTItixaqXq`Wo_PIdbHgu[OGSaTKhTItz@aqXlHgw]OGdr@csZMGcYPIeUJTq|_pXKubUcRIDQp[oXLEra\\OhTItat[nWKURLgtZ]GC?B??_g?CCB_?OOKH??__WQ?@@?o_Y?@@?ogY?@@?og[P??___SKG_?OOOIFCag?CCEBa@GgV??_o_SKGDO?GKGDBaPWu?@@`_w_QKEbg?CGEBa`WoYMFo?GSOHDAp_s[NGO?OKGDBA@Ou?A@`?g[QJFCg?GEEBa@Go]QHo?OOKFDAp`?cRIO?OSOHDAp`CcRIDW?KGEBbPo{_QHdAk?EDCAPgw]PHCq`Wn?A@POkYMGCQPKiUJeG?WOIEBPx?aQIDQp[oX_@pGkWMFcAHKgTJDr@ct\\?__WOIECA`hGx@@?o_SMHDbaXoz@@?oo[OHEBq`oy]____WMIDbA@Sw\\NFwGGIGCa`WoaUMFRp|@@@`?o[YMFcAxoy]NgGGKIGCbPo{aWMFRp|Aa_`?ggULFCAHcw\\NGCIHF@BA@OoYNGCRPoy^OGSQLH@BaPWo[NGCRXo{^OGSQLGd`?o_WMQHdAh[w\\NFsAICBAa@HGeSJEB`s{^Ogs{GGDDAqPKiUKfBhx?`QHtCGKGDBAPOiULFBh|?`QhtAeCFCap`KgTJEr`w}_Ohcy`Qi``?gs[QHdr@cw\\NGcYPMgSooo_YNHDAx_s[MfsQLIfSIdcKMHFBqXOmWLfBp|CbRHtAlWla@`Os_QIdrHgw\\OGcaTMhTJDkOMJFCAXSmXLfBq@CcRHtIlWmVpPGk[PHdb@cu[NGSYPKgSitax`DCA`hCcUKERPoy`PhCi`QiUJTzEKFEBq@OiVLEr`|?aQhcyhUlVJuC_QKFcQ`WoYLfByDEdRIDQlYmWke[gUKGCQhWqYLfCADGdRITQl]oWkeZQYMFcAH[oXLEraHEcQhdat[nWKURLgthCq`SkVKERPkwfSITQlWlVJuBDcrYLUv";
DIGRAPHS_NamedDigraphs.("graphcartesianproductofk33andk3") := ".Q_____CgEf`CEF`DGHIaDGHbDGHaCEFKbCEFL`ABEG`ABFHq_____CgEf`CEF`DGHIaDGHbDGHaCEFKbCEFL`ABEG`ABFH";
DIGRAPHS_NamedDigraphs.("grayconfigurationmengerdual") := ".Z____@_B_Aj@FGcGIJcFHJbFGaFG`HIKLbHIaHIdFHMPeGINQdGIMPeFHNQ`ABJMN`ABOPQcDELRUcDEKSTz____@_B_Aj@FGcGIJcFHJbFGaFG`HIKLbHIaHIdFHMPeGINQdGIMPeFHNQ`ABJMN`ABOPQcDELRUcDEKST";
DIGRAPHS_NamedDigraphs.("gray") := ".u_OGCoGDb@w_aIJaag_eMGbAwoiMHcAxGiXMfsYDUhRhRY|YobkGjXqqYlgJPyw^kfZyAy]og{IIAgToTIiElVodayEcRociZl?_OH_OJEBp@CSVDDP@K[PEDp`S[RGDqPSq\\NgsIlQfQetyt`FWPUrdctZOUbto~XMvsDs|`PwSSDOj_iTSLYn`JDsLGf`HSv";
DIGRAPHS_NamedDigraphs.("greatrhombicosidodecahedral") := ".~?@w_GA?`OQCa?]FaoiIb_uLcPAOd@MRdpYUe`eXfpy]gaE`hQQgHY{mjjGpjBEilaixHjewMj[zn`ugIQmlnRx?i{IhOytBnSPDlRg{hbSukbSukbKslyqvkZ@KRj@LRi|JRYMaIAeaGqQ_HtI^NRy^GAEwQza\\FhlYed`Xe`k[eA}VFTmVE@eTOHOmVhOTDhHKcQt`cPGRbsIMJEQMBpAKLGkjXwkKBWcdaAhia?cI`aPl`QYD@_]BFDeBFA]AMVE@HtI@?_Rw_GA?`OQCa?]FaoiIb_uLcPAOd@MRdpYUe`eXfpy]gaE`hQQgHY{mjjGpjBEilaixHjewMj[zn`ugIQmlnRx?i{IhOytBnSPDlRg{hbSukbSukbKslyqvkZ@KRj@LRi|JRYMaIAeaGqQ_HtI^NRy^GAEwQza\\FhlYed`Xe`k[eA}VFTmVE@eTOHOmVhOTDhHKcQt`cPGRbsIMJEQMBpAKLGkjXwkKBWcdaAhia?cI`aPl`QYD@_]BFDeBFA]AMVE@HtI@?_R";
DIGRAPHS_NamedDigraphs.("greatrhombicuboctahedral") := ".o_OGCSHCc@w{kTIfBXlKdQgtqtYpXhdZT[tPHRrsww\\fBqConJGfxWoZHIqAqIOHDPbCSuda`_{K_[`cAcGqg_cqcCCCwOGCAICaQ?{]UIdR`kueQhSYxYlWksqlimYgchxy[[Mr`xAWVdcR{kWLcdX@XDGCagpaIZQpOo]EOMOq@QCXSOQXQAAA^";
DIGRAPHS_NamedDigraphs.("grinberggraph42") := ".i_OGC_NDbAW_aGFaaGo]XJecHo}THHpGkY_TfdYX]gUhDyHYrWNVZdhBZOOR{C|APosQYE`QocART?_OH?]JECp@CONDCP_{qVLGR`{iQRaPWt?jNIsq|OlQJsQteo]mvJQEu__fwHyCb`gcsLAd`GCf";
DIGRAPHS_NamedDigraphs.("grinberggraph44") := ".k_OGCSHCdAXKWVDDBH?e_GDppSW^DBpIXQcSfsA|E[XfqG|_xTMtb\\gx\\lts\\dDBOhWHxHANosQaAb`GDUq@?_POcQSHcp`[SSKcAY?_VFDP`{SNChdIPQ^OJsXpe^GbuBdSzULubduvVPuSSLAd_fccG~BPIGIMC_T^";
DIGRAPHS_NamedDigraphs.("grinberggraph46") := ".m_OGCSHCdAWwiYGbAgo_RDBbg{mOKcbX_uIPaTa|CdVheilOxKhVBtk|^nujAQybkggYDH@OHgQ]EdTOS[HCkvOGCAICaQ`K[TLCP`SWOHa`ps]VGEQPkoZDGpIp]aQjsrTUg[eSj`yu]nvzT`H\\PuSSLAc_gCsHNBQigIMCaU^";
DIGRAPHS_NamedDigraphs.("groetzsch") := ".Ji?c@pEUPBFaGhg@CKey_H?[PdcOpwaIY?PBH";
DIGRAPHS_NamedDigraphs.("gruenbaumgraph121") := ".~?@x`oMB@wUDAWQCAwYE`g]GageKawsObpENCwwQbPUMDH?UcP]QEXKWd@mTEhq\\fyE_fiIbhAedIIYfJGMD`IGggqecIiSjhqueJIymLy}nMZAoMzEqkzUslj[|mByxNzh?nCEzOjtBncQ~P[@Eocb@P{fIq{rMr[~OT\\FQT|NSU[|TsDZPT|HWtDjRU\\n[v\\z_v}FawubcxejdX}DfwebbY]PixUreYw@l_Exr_U}@[FUA[gHp\\~eF?wKF`WSH`GOJ`gYE`waIaWqJbPANCW{Rb`ILDWwSc@YPDxGXcpaSExSYfHu^gYA]giMcIYSghi[k_wUCgaabIYOihQmfJYWkjiwvjy{xkJ?zkZIrlZQulruwNjc~mcA{OZlAnSM}PJ|DoCZAQKDFq[jJrKzLr|@Ts\\HVs|PXrtVOTlDVsdbSUlLXu|r\\vmB^w]JbYMReYmTfwU^aYMLhxEjdZMXj_Eu?ZfM@ZwDo\\WHq_fDv";
DIGRAPHS_NamedDigraphs.("gruenbaumgraph124") := ".~?@{_GA?`OQC@W_F`oaKAgmIBGkLc_{Pb`AOCxCQe@SVd@YUEX[Wexi\\f@y\\Fho^hACbgAIHGaUGGqQPcIGhgqaFiQigIyqkJyylkBInKZGrkRQulZavMZ_xlri~Mry{NWW}NwS|OGiJpouKqsPEosVGQk\\HdciVQZxDnSYCpdJDS[hNqTBCQ{LKqCvFRlNRVLVSvTz[V|nYu\\bUt}ZfYMVcwunaYm\\hxebdYmPjwuraZ\\|`veAB[VUB[FQ@ZwHm_fLx_VHw~GA?_GSC`?UG@w[Gb?iJa_qJBXGNCWwOc@MPCh_TDxOUd`eVEHmYfXo]fPy[FyO`Gy?aaQGdaAKccXAaIYKg`yciiAmkjA}mjZ?qjrEqKzCsljUwlrewMZ[ynrk}nBuENb}DNSAIa{[LbKlCPkLDqCjFQXXIdse}PZtE`KXQpTFIR{dOpCnBRK`LpszRstrTtLt]vD~ZulfWtl^exubdxMLjwejfY]XgxUjcY}Lkwev^W\\x__vDt_v@s_U}AZgHr]WDq]N";
DIGRAPHS_NamedDigraphs.("gruenbaumgraph25") := ".X____hGfed`DEaFGLaIKaDHbHIcJKP`GKbFJR`HJcEIOcDFTbEGNx____hGfed`DEaFGLaIKaDHbHIcJKP`GKbFJR`HJcEIOcDFTbEGN";
DIGRAPHS_NamedDigraphs.("halljanko") := ".~?@c_G?@_?CA_?CA_?CA_?CA_?E??W?@_?CB@?[G_?CD@_[G_?CB@O[H_?CC@_[H_?CC@O_H_?CB@__H_?GFA?cIAw?A@o_HB?u??_[GAOwN_?GD@_cIApCQ_?GB@?cIApCQ_?GB@O_KBP?Q_?GC@__KBP?Q_?GC@O[MBp?P_?GB@_[MBp?P_?SEAOgKB`?SD`a?@OWHA_sNC@OTDw?C@__IB?wPCpSWEg?B@_[IB?wQCpWVEg?C@O[IBO{QCpSWEW?B@O_IBO{PCpWVEW?B@?cJB?{OCpWVEpu??oOHAosMC@KTE@o]_?OEA?kKBpCSDP[XF@s__?OD@okKBpGSD`_YEpw__?KE@okLB`GSDP[XEpw^_?KDA?kLB`CSD`_YF@s^_OGFA?cIAp?PC`KS_OGFA?cKBP?PC`SU_OGFA?cMBp?PC`[W_OGB@?cIAp?RD@{_Ha]@?_SEAOgJC@KSEPgeHwCA@?WGB?sPDPWZGQSf_OGB@O_KBPCTD`wcHQ]@?_OD@owNC`[WFQGdHgCA?oWFB_{QDp_[GqSe_OSEAOgKB`KXE`k[HQ_iJWCD@_cIBO{REPg\\FaSgIqq@?oWFA_oME@cZFAKfIQgkJwCC@__IB?wUEPk[GQWhIqsn_OOD@ogLBp[YFPwaHqcjJQy@?oSGA_sNDPg\\FaOeIQgkJgCB@?cJB?{SFq?`GaShIaopKgCB@?cJBOwSFq?bHAShIqsoKwCC@__JB?{UEp{`GaWgIqomKBGt_OOD@okKBp[\\FqCaHq_iJQ{pKrU@?oSGAosMDPw_GqOeIAglJr?qLGCB@_[JBOwWFA?bHA[gIqomKRKs__KCA`?RDP_ZFP{_HQciJB?rLBUA?oOIC@KUDpo]Fq?dIQklKRGsLWGD@_kOD@WWEPgaHASgIqomJrWx__SEAp?SDP[XEaCbHQ_iJQwnLraA?oSKCPKUDpo]FqOeIAglJbKvMBg{__OEBPCRDP_ZFQ?`Ha_jJA{pLbczNWGB@_{QCpWVF@w^Gq[gIqonKBWxMbuA@?SMC`KTE@k\\GAGfIAglJbGvMBk{__KDB@CSD`_XFaGcHaciJB?rLB_zNS@@__KEBpGSDP[YFACbHqciJB?rLBczNBw~__OEBPCSDP[YEqCbHacjJRCqLRWyNC@@__ODB`GSD`_XFQGcHqcjJRCqLR[yNRw~`o_IApCQCpO[FQCcHa[gIRCrLrc}NsLD`o_IApCQCpOZFaGbHa[gIR?qLb`?OSHC`ocKBP?QDPWYFA?aHQ[iIqwqLBczNCDB`ocKBP?QDPWXFP{bHQ[iIq{oLR[yNS@Da?cMBp?PDp_XFa?`HQWkJQ{pLB_zNR|Aa?cMBp?PDp_YEp{cHQWkJQwrLRWyNBxC_o[IB@CSE@cZFaGbHQklJbCqLBcyNBxCPSXHQgOGA_wQD@WXF@s`HASiJAwoKrSuMrp@OcLFQSiBA?gNC`OTE`o\\GQOdIqsnKRGsMBg|OCPDPs`J`?[IBPCSDpgZFaGbHQgkJr?rLR[zNR|AOsXGQwKFA_oODP_XEps^GqWgJQwpKr[xMrs~OSHFQClMRwKGA_{ODP_YEps^HA[gIq{oKbWwMro~OSLEQShKRWOGA_wOD`[XF@w_GQ[gIawoKbWwMbs}OCTEQClMRwOFA_sOD`[YF@w_GaWgJA{pKr[xMbo}OCPFQShKRWSFAooPCp[[FP{`HASjJAwqLBWxMrs~OCHFQSlLRdADA?kMC`KTEpw_GaKdIqomKrSuMRg|Ns@DPc`IRC|Q`_[JBPCRE@o\\GACcHQglJr?tLr_yNBx@PC\\GQctMSwWGAo{QCpWZF`{aGqSiJQ{pLB[wMro}OSLEQSlKRtED@okKC@SVEPs^GQKeIQooKbOuMBg{NcLCPc`IRDDQTTYEA?kNC@WWE`k^GaOfIQgpKrOvMRg|OCHDPs`IRd@RTTYE@okLC@WWE`o_GaOeIQsoKbSuMBk|NsHDPcdJRtDQTD]DA?kMC@SVEPw_GQKfIQkpKrSvMRk{OSLCPsdJRT@RTD]DAOoMCPKVE`k[FqOfIQknKBCtLrl?OSHCPc`IRtDTTd`X`OcKB`GRDPgZFA?aHackJr?pLB_yNb|BPS\\HQsxRTD\\YUwWHBO{QCpWXFPw^GqWhJQwqKrSuMrw~OsTFQChLSDTUUDeEAOsNCPKWEPs]GACfIQgmKbKsMRh?OSHCPcdJRDHSTthZ_ocKBpCSE@c]GACaHq_jJr?tLb[{Nb|BPS\\GQsxNSDHUUtt]_ocKBpGSDPg[GACaHa_lJbKtLb[|OCDAPCXHQcpLSTLTUdp^`?cLB`GSD`c\\FqKcHa_kJrCsMBc{OCDAPCXGQsxNSDHVUDp^`?cLB`CSDpgZFqKcHq_iJbGsMBc|Nb|BPS\\HQcpLSTLSUTt]xGA??W?@?g?@?g?@?g?@?g?@_?E??W?@?oOFAG?@@OWFAG?@?oSFAW?@@?WFAW?@@?SGAW?@?oWGAW?A@o_HA_m??_[GAOoL_?GFA?cMBw?A@OWHA_kPCg?A?oOHA_kPCg?A?oSGB?sOCg?A@?WGB?sOCg?A@?SFB_{OCW?A?oWFB_{OCW?D@_cIB?wOD@WW_?SEAOgLBp?SDP]?@?WGA_oMCPKTE@i??oWFA_oMC`KUDpi?@?SFA_sNC`KTE@e??oSGA_sNCPKUDpe??oOHAooNC@KUDpk\\_?KCAOkLB`?RDP_[Fg?C@__JB?{PD@SVEPo\\GG?C@O[JB?{QD@WWE`k]GG?B@_[JBOwQD@SVEPk]Fw?B@O_JBOwPD@WWE`o\\FwCA@o_HA_kOCPGRDGCA@o_HB?sOCPGTDgCA@o_HB_{OCPGVEGCA?oOHA_kOCpO^GAWf_OGD@_cIAp?RD@cYHa]@?_OEA?oLCPSUEqCdHwCA?oSGB?sPDPW]HASf_OGC@O[MBpGVE@saHQY@?_KE@owNC`[WFAKdHgCD@_cIB?wREPgZFASgIau@@OWHA_sNCpcYFPwdIAkk_OKE@ogKB`_XEpobHqciJA}@@?WGA_oMD`cZFACeIQklJwCC@O[IBO{VE`s]Ga[hIqsm_OKDA?gLBpSYFPwcHaciJAy@?oOHAooND@{_GQGdIQgkKRI@?oOHAosMD@{_GqOdIQklKBM@@?WGAooND`k^GQGeIAkkJb?qLWCC@O[JB?{VFP{`Ga[gIasnKRKt_OKDA?kLB`S]GAKcHa_iJQ{oKbQ@?oWFAosME@o_GqOfIAkkJbCrLGGB@?gOCpSWEps^GAShIaooKrOt__KCA`?RD`[[F`{_HQcjJRCqLBUA@OWJC@OUE@cYGaOdIAkkJa{uMWGD@_kOD@SVEPg`GqSgIasmJr[w__KDB@CRD`[[F`{cHa_iJQwrLr_yNGGC@_sPCpSWEps_GQWgIqonKRWxMruA?oWNC`KUDpo]FqKfIAkkJr?uMRg|__ODB`GRDP_ZFQ?aHq_iJQwqLr_zNGGB@OoPD@WWEPwaHAWhIaooKrOwMrt?OWGB@_{QD@SVE`o`Gq[hIaooKrOxMro}NwGC@_sPD@SVE`k`GqWhIqspKbSuMbp?OWGC@OwQD@WWEPsaHA[hIqspKbSvMbs}Nw[GA_kPC`KSF@s`HAWfIAcpKr[xNb|BPW[GA_kPC`KSEpwaGqWfIAcoKbWwOCDAPG[HB?sOC`SUE`o_GaSfIakmKbOxMrp@Ow[HB?sOC`SUEPs^GqSfIaknKBSvMbt?PW_HB_{OCP[WEPw_GQSeJAsnKROwMrs~Og_HB_{OCP[WE`k^HASeJAsmKrSuMbo}PGKFA_oPD@_XEpwaGqSjJQwpKbOxMbo}PCTEQSiCA?gMC`OUEPo\\GQOdIaomKBKtLbk{OSHBPsdI_o_IBpGSDPg[FQCcHQklJrCqLB_yNS@CPS\\GQwOFA_sPD@[YEpwaGqSiJA{oKrSvMrs~OcLEQCmB@ogKC@SWEPk\\FqKeIAsmKRKvMRk|NsDAPs`JRc}BA?gNC@SWE`k\\FqOfIAknKBGuMBk{NsDBPcdIRCuCA?gMC@WVEPo]GACfIAgmKBGuMBg|Nc@DPc`JRc}C@ogLC@WVE`o]GAGeIAonKRKvMRg{Nc@CPsdIRCuD@okKCPKVF@s^GQOdIqomKbOuMRk|Ns@APsdJRSxO`O_JB`GRDPk]GAGbHQkkJbKtLbcyNR|?PSXGQcpNSgWFAosPCp_[FQ?`HASiJQ{oLR[wMbo}OSPFQChLRdMEA?kNC`KUEpw^GaKdIasnKROvMBk{NcDBPcdJRC|P`O[JB@?TDpc\\FqCbHackKBGsLb_yNBxBPCXGQcpPSdTU`__JBp?UE@gZFqGcHqciKRKsLrcyNS@APS\\GQcxOStTU`_[JBP?UE@g[GAGcHaclKBGtLb_zNR|APSXHQs|PSdPV`O_JB`?TDpc]GACbHqcjKRKtLrczNCDBPC\\HQstOStPV`OcKB`CRDpgZF@{cHqcjJr?pLR[zOCDAPCXGQc|PTTXWUWSHB?wQCpSYEpo_GaWhJA{oKROwMbw~OsTFQSlMStPVUdmEAOsNC`KUEPs]FqKeIQsmKbKtLbk}NsLDPs`IRT@TTd`X`_cLBpCRE@c\\Fa?`HqciJbGrLBcyOCDAPCXHQspQTD\\YUwKHB?{PD@_XFa?`Ga[gIq{oLRWvNBw~OsTFQClMRt@QTdl\\VgKHB?{QD@SYFA?`GaWgJQwrLRWvNS@@OcPEQShKRTDRTTh[VwOHBOwQD@WXFP{bHAWgJA{pLB_xNC@@OcPEQClMRt@QTt`[VwOHBOwPD@[YEp{bHA[gIawqLB_xNRw~OsTFQShKRTDRTDd\\Vn";
DIGRAPHS_NamedDigraphs.("halvedfoster") := ".l_O?KA?B_O?k_NICr@[wZPGRww_]OGchGgUKEbgW[XDBbg_[UUb@phUEGDp@?qIGEeWo_[Y_bPxmAJGEWGwc[NOP`Cs\\`aQPeIHGew_c]YPP`H?w``a`xSzDDCApxJEDCRAAGIHDry]CLHECKGUPJgdKGWOIhGOw]UPi|kA?@_O?[A?Dc@xOeWJfBYHA^FCBq@CdHDAp_s\\BBbGg[\\CBaqsWMLIop?mGGEPP?srECBbSCYNLoPX?r@FCb`yAKGebkKQQKpPHCvCCbrQIKHGFCKKSNIfWgg_UNHWogaWOP@PGm^Ro`hGo``AqH\\Ch`BA@THAFBqqLV";
DIGRAPHS_NamedDigraphs.("harborth") := ".s_O?KA?Bb@XGSLEBaWwiMJdbIHAOOgCQTCfSiDZ@WnTJEH@InVLtr@miXLVJ{{s\\NRrXo~MORryA[_PhHqNLLGCX?oYgaa[}XMj`IgIlXCFBwa\\Un``_waEDTJWGHYmyOG?E@?@p_kcIEb@pK[TFDqpdC`GGSAHIaRiTAl_kVidbC_dVjeyx_vTKejd}]YMfhxkw^fGHx|@MOGscxFeecAK_WLSPPM^KfTodSDUkaB`{PMjVooo[PBAidkCClV^";
DIGRAPHS_NamedDigraphs.("harries") := ".~?@E_GA?a?]E`GUC`gUD`GYF`xoKCg{Rc@ELBgoSbP]ODWwWcPiNDhKXc`mJEg_Ta`]GEGkZaPQHDggXdPYVExOWePioKzCqgAa`IYOjhQiaJIKnhqyeJXo^fPy@Lb]AMBeAMbu@MrqBLR}BLBzE_GA?a?]E`GUC`gUD`GYF`xoKCg{Rc@ELBgoSbP]ODWwWcPiNDhKXc`mJEg_Ta`]GEGkZaPQHDggXdPYVExOWePioKzCqgAa`IYOjhQiaJIKnhqyeJXo^fPy@Lb]AMBeAMbu@MrqBLR}BLBz";
DIGRAPHS_NamedDigraphs.("harrieswong") := ".~?@E_GA?a?YF`WQC`W]E`GUF`hoKCx?QbpELBgoSbP]QDWwWc@mPDhKXbpiHEg_Tap]GEGcXa`QJDggZdPYVEXOWe`moKzCqgAa`IYOjhQqfIiWngqyaJXo^fPy@Lb]AMBeAMbu@MrqBLR}BLBzE_GA?a?YF`WQC`W]E`GUF`hoKCx?QbpELBgoSbP]QDWwWc@mPDhKXbpiHEg_Tap]GEGcXa`QJDggZdPYVEXOWe`moKzCqgAa`IYOjhQqfIiWngqyaJXo^fPy@Lb]AMBeAMbu@MrqBLR}BLBz";
DIGRAPHS_NamedDigraphs.("hatzel") := ".xa`GwYQGdag?Q?D`awGYCH`AGG]EIbQac]IUaQBCmQYeSRcuc]hriAE]__OW[Iwcmh{YV?enHKQ^AenhY`YmWkeyhj_pXK|AfWnYzTrVSjZL~KSHFBQPCkT?AO?kKV@BO`KGP@BopSYSSbpQsQOWdqRSqa[esbtM\\OOrsCABB`VCdtNbQwCtxHaRwSt|JSJTrDcvTL[EJHfgSzDzVYmYydZXn";
DIGRAPHS_NamedDigraphs.("heawoodfourcolor") := ".Xa?@_@AdCccEcDF_B_AdGeFfGJK`BH`AIpINOjKLOhMcDERdJOPQSeKOQRS`MNOPRU_HIPRSTxa?@_@AdCccEcDF_B_AdGeFfGJK`BH`AIpINOjKLOhMcDERdJOPQSeKOQRS`MNOPRU_HIPRST";
DIGRAPHS_NamedDigraphs.("heawood") := ".M`AE?cJCpxCLK`xDNKhnO`B?QDaW{aEeO{afeSv";
DIGRAPHS_NamedDigraphs.("heptane") := ".VoLMNiJK_@cDQaBQeFORgHPSvoLMNiJK_@cDQaBQeFORgHPS";
DIGRAPHS_NamedDigraphs.("herschel") := ".JcAKWObGXRqGW{qDLmOGp`AK`dNG`brGSv";
DIGRAPHS_NamedDigraphs.("hexane") := ".SmJKLgHIaB_@OcDMOeFNPsmJKLgHIaB_@OcDMOeFNP";
DIGRAPHS_NamedDigraphs.("higmansimsbipartitedouble") := ".~?BG_C?_C?_C?_C?_C?_C?_C?_C?_C?_C?_C?dgVB?XBOZB_\\Bo^C?`CObC_dCofD?hDOjDcVB?XBOZBccCgeCwgDK`CWeDGiDc`COfD?iD[aCWcCgjDc^C?gDGjDc]C?dCwiDc]BwcCoiD[\\BwbCggDS\\C?aC_hDS\\C?`CgeD[\\Bw`C_fDc\\BobCwhD[\\BoaCogDc^C?aCWeC{]C?`CWcDC]Bw`COdDKVB?\\Bo^CCZB_eCwjDcYBWdDGiD[YB_cD?iDcXBObCogD[XBOaCwhDcXBWbC_fDSXB_aCgeDSXBW`CggDcXB_`C_hD[ZB_aCWgDKYB_`CWdC{YBW`COcCsXBO^C?`DSXB_]BwbDcXBW]C?aD[YBW\\C?bDcZB_\\Bo`DSYB_\\BwaD[ZB_^C?cCkYB_]C?eDKYBW]BwfDCXBW\\BweDKXB_\\C?fDCXBO\\BocCkVBG\\CGaC[WBG\\DOjDcVB__CWiD[VBW^COiDcVBO]CGjDcWBW^CGbD[WB__CGaDcWBO]CObDSWBW_CogDSWB_^CwhDSWBO^CgeDcWBO_C_fD[WBW]C_hDcWB_]CggD[VBG_C_eDcVBG^CgfD[VBG]D?hDSVBO\\CofDSVB_\\CghDcVBW\\C_gD[VBW_CGfDKVB_^CGeDCVBO^CWcDKVBO_COdDCVBW]CWdCsVB_]COcC{WBG_CWdDKWBG^COcDCWBG]CGeC{WB_\\CWcCsWBW\\COdC{WBO\\CGgDKVB?bCwgDcVB?aCohD[VB?`C_dDSJ@_L@oNA?mFxKHhMHxOIHWJHYJX[Jh]J{J@_QAWTAomFxKI@RI`TIp_KHgLHkLhsM{MA?PAWSAomFxMIHQI`UIxbKhfLXnMHrMsL@wPAOSAgmFxLHxQIXTIxaK`eLPmM@qMkG@OJ@gSAomGXGHHJIHTJ@\\LXlLprMhvNKH@OM@wTAomGhEHPJI@VJ@^LHoMHsMhuNSG@GKA?SAgmG`FHHIHxUJ@]LPkLxqMpvNCF@GJ@wRA_mG@CH@JHpRJ`]KXcKpkMHsNKF@OJA?PAgmGPCGhHHhSJX^KHcL@mMHrNCF@?L@oRAgmGPEGxJH`QJX]K@aKplLxrNSF@?NA?QAomGHDH@IH`QJP\\KHbKxkLpqNSF@OK@oQA_mG@BGxIHhSJ`\\KPdKxlM@sNCF@GK@gPAomGHBGpHHpRJP^K@dL@nM@qNKG@GJ@oPAOmG@AGXDHxUJHYK@bKphLXtNCH@OLA?QAWmGHAGxGI@VJH[K`dL@iLXvNSG@OK@wPAWmG@@G`EIHTJHZKHaKxhLPuNKC?oJ@oSAgxFW{Fg~HhUJ@]KplMHrM`tNCD?oLA?TAouEw|Fo~H`VJ@^L@mLxqMXvNSC?gK@wSAowFO{Fo~HpTJ@\\KxkM@qM`uNKB?gJA?QA_rFGyFg~HxSJ`\\KXcLXkLpvNCB?oJ@wPAotEwyF_~IHRJP^KHbLHmMHtNKB?_NA?RAgsEwxFo~I@QJX]KHcLPkMHuNSB?_L@oQAotEowFg~I@QJP\\K@dLXlM@tNSB?oK@gRA_rF?zFo~IHRJ`]KPdLPlLxvNKB?gK@oPAgsEozF_~HxSJX^K@aLHnM@uNCC?gJ@gPAWrEgvFW~HpTJHZK@cKpgLXrNKD?oM@wQAWsEgwFG~H`VJH[KPbKpfLHsNSC?oKA?PAOrE_uFO~HhUJHYKHdKxgLPqNC@?OJ@_PA_rFOzF`?GXCHHLHpNIHRI`TIs@?OMA?QAgsEoxFhAGhFHPKHhNI@QI`UI{@?OL@wRAotEwwFp@GpGHXKHpOIHQIXTI{A?gG@WNAgoEwxF`CGhJH`NIhWJXeLHkLsA?_H@WOAopEwyFhDH@HHpOIpWJPgLXkMKA?_I@_LAgoEozFpEGxHHhOIhWJXgLPlMCA?oI@wOA_qFGyFpCH@IHhPIxWJ`fLPmMKA?gH@gMA_qF?zFhBGxJHpNIxWJ`eLXnMCA?oG@_MAopEowF`BGpIH`PIpWJPfLHlL{A?WFA_TAonF_|FpHHPJIPRI`kLhmLxoMKA?WI@WMAWpEgxFX?GPJI@PI`ZJ`hLXlMKA?WG@gOAGqEWuEx@GPHHxPIPYJXiLXmL{A?WH@_NAOoE_wFP?GHIHxOIXYJ`hLPkMCA?oF@WLAOoEWtFhAGXGH`LIXYJ`eL@lLsA?gF@_OAWpEWsFp@G`FH`MI`ZJ`fL@kL{A?_F@oNAGqE_tF`?GhEHhMIPYJXeKxoMKA?gIAGQAonEguFP@GXDI`TIxfL@hLXmMCA?_GAORA_nEWwFH?GxGIPTIpeKxiLXkLkA?oHAGRAgnE_vFXAG`EIXUIxeL@hLPnMK@?WFAGQAWnEWsEh?GHAIPRI`_KHaKXcKk@?WH@WLAgoEwzFhAHHJHxOIX]Jx_K`tM{@?WI@_OAopEoyFp@HHII@PI`\\Jx`KhuM{@?WG@oNA_qF?xF`?HPJHxPIP\\JpaKXtMs@?gF@WMAopEg{FhBGhJH`MI`\\Jx_KXrMc@?oF@_NAgoE_{FpCGpIH`LIX]Jx`KPqMc@?_F@gOA_qEW|FpFH@HHhMIP\\JpcKhqM[@?_I@WNAOoEgxFP?GhGHhOIhXJh`K`sMk@?oG@WOAWpEWvFHAG`GH`PIpXJp`KXrM{@?_H@_MAWpE_wFX?GpFHpOIpXJp_KhsMs@?gG@_LAOoEWuF@@GXFH`NIhXJh_KPqM{@?oI@gMAGqEguFXAGXEHhPIxXJxaKhrMk@?gH@wOAGqE_vFP@G`DHpNIxXJxbK`qMs@?gIAWSAgnFGzFpCGxJI`TIxaK`rM`uM{@?oHAOSAonF?yFhBH@IIXUIxbKhqM`tM{@?_GAGTAonEovF`DGpHIPTIp_KHqMXtMsD?oH@OJ@_oEGyFXBG`WJH[JxgLHsMxwNKC?gG@OMA?pEOuFHDGxWJHZJhfLXrMpwNSC?oG@GL@woEOvF@EH@WJHYJpeLPqMhxNSB?_F@?J@_oEGrF`?HHYJX\\Jp_KHkLhwNKB?gF@OL@woEOtFp@HXZJ`\\JxaK`mM@xNSB?oF@GMA?pEOsFhAHPYJ`]JxbKhnMHwNSB?gG@GRAonEGvF@@HX_KXhLXkLxuMxxNSB?oG@OQAgnE?uFHAHP`KPhLPlLptMxwNSB?_H@OPA_nEOyFX?HHcKhiLXoMHtMpwNKC?oF@ORAonEGtFpEH@`KhfL@lMHrM`xNSC?gF@GQAgnE?sFhDGx_K`eL@kM@qM`wNSD?oF@?PA_nEOrF`BG`aKXeKxmLxqMXwNK@?OF@?H@OnE?pEP?GHAGXCGhEGxGHHIH[@?OB?_D?onE?pEOrE_tEovF?xFOzF_|FvG_C?_C?_C?_C?_C?_C?_C?_C?_C?_C?_C?dgVB?XBOZB_\\Bo^C?`CObC_dCofD?hDOjDcVB?XBOZBccCgeCwgDK`CWeDGiDc`COfD?iD[aCWcCgjDc^C?gDGjDc]C?dCwiDc]BwcCoiD[\\BwbCggDS\\C?aC_hDS\\C?`CgeD[\\Bw`C_fDc\\BobCwhD[\\BoaCogDc^C?aCWeC{]C?`CWcDC]Bw`COdDKVB?\\Bo^CCZB_eCwjDcYBWdDGiD[YB_cD?iDcXBObCogD[XBOaCwhDcXBWbC_fDSXB_aCgeDSXBW`CggDcXB_`C_hD[ZB_aCWgDKYB_`CWdC{YBW`COcCsXBO^C?`DSXB_]BwbDcXBW]C?aD[YBW\\C?bDcZB_\\Bo`DSYB_\\BwaD[ZB_^C?cCkYB_]C?eDKYBW]BwfDCXBW\\BweDKXB_\\C?fDCXBO\\BocCkVBG\\CGaC[WBG\\DOjDcVB__CWiD[VBW^COiDcVBO]CGjDcWBW^CGbD[WB__CGaDcWBO]CObDSWBW_CogDSWB_^CwhDSWBO^CgeDcWBO_C_fD[WBW]C_hDcWB_]CggD[VBG_C_eDcVBG^CgfD[VBG]D?hDSVBO\\CofDSVB_\\CghDcVBW\\C_gD[VBW_CGfDKVB_^CGeDCVBO^CWcDKVBO_COdDCVBW]CWdCsVB_]COcC{WBG_CWdDKWBG^COcDCWBG]CGeC{WB_\\CWcCsWBW\\COdC{WBO\\CGgDKVB?bCwgDcVB?aCohD[VB?`C_dDSJ@_L@oNA?mFxKHhMHxOIHWJHYJX[Jh]J{J@_QAWTAomFxKI@RI`TIp_KHgLHkLhsM{MA?PAWSAomFxMIHQI`UIxbKhfLXnMHrMsL@wPAOSAgmFxLHxQIXTIxaK`eLPmM@qMkG@OJ@gSAomGXGHHJIHTJ@\\LXlLprMhvNKH@OM@wTAomGhEHPJI@VJ@^LHoMHsMhuNSG@GKA?SAgmG`FHHIHxUJ@]LPkLxqMpvNCF@GJ@wRA_mG@CH@JHpRJ`]KXcKpkMHsNKF@OJA?PAgmGPCGhHHhSJX^KHcL@mMHrNCF@?L@oRAgmGPEGxJH`QJX]K@aKplLxrNSF@?NA?QAomGHDH@IH`QJP\\KHbKxkLpqNSF@OK@oQA_mG@BGxIHhSJ`\\KPdKxlM@sNCF@GK@gPAomGHBGpHHpRJP^K@dL@nM@qNKG@GJ@oPAOmG@AGXDHxUJHYK@bKphLXtNCH@OLA?QAWmGHAGxGI@VJH[K`dL@iLXvNSG@OK@wPAWmG@@G`EIHTJHZKHaKxhLPuNKC?oJ@oSAgxFW{Fg~HhUJ@]KplMHrM`tNCD?oLA?TAouEw|Fo~H`VJ@^L@mLxqMXvNSC?gK@wSAowFO{Fo~HpTJ@\\KxkM@qM`uNKB?gJA?QA_rFGyFg~HxSJ`\\KXcLXkLpvNCB?oJ@wPAotEwyF_~IHRJP^KHbLHmMHtNKB?_NA?RAgsEwxFo~I@QJX]KHcLPkMHuNSB?_L@oQAotEowFg~I@QJP\\K@dLXlM@tNSB?oK@gRA_rF?zFo~IHRJ`]KPdLPlLxvNKB?gK@oPAgsEozF_~HxSJX^K@aLHnM@uNCC?gJ@gPAWrEgvFW~HpTJHZK@cKpgLXrNKD?oM@wQAWsEgwFG~H`VJH[KPbKpfLHsNSC?oKA?PAOrE_uFO~HhUJHYKHdKxgLPqNC@?OJ@_PA_rFOzF`?GXCHHLHpNIHRI`TIs@?OMA?QAgsEoxFhAGhFHPKHhNI@QI`UI{@?OL@wRAotEwwFp@GpGHXKHpOIHQIXTI{A?gG@WNAgoEwxF`CGhJH`NIhWJXeLHkLsA?_H@WOAopEwyFhDH@HHpOIpWJPgLXkMKA?_I@_LAgoEozFpEGxHHhOIhWJXgLPlMCA?oI@wOA_qFGyFpCH@IHhPIxWJ`fLPmMKA?gH@gMA_qF?zFhBGxJHpNIxWJ`eLXnMCA?oG@_MAopEowF`BGpIH`PIpWJPfLHlL{A?WFA_TAonF_|FpHHPJIPRI`kLhmLxoMKA?WI@WMAWpEgxFX?GPJI@PI`ZJ`hLXlMKA?WG@gOAGqEWuEx@GPHHxPIPYJXiLXmL{A?WH@_NAOoE_wFP?GHIHxOIXYJ`hLPkMCA?oF@WLAOoEWtFhAGXGH`LIXYJ`eL@lLsA?gF@_OAWpEWsFp@G`FH`MI`ZJ`fL@kL{A?_F@oNAGqE_tF`?GhEHhMIPYJXeKxoMKA?gIAGQAonEguFP@GXDI`TIxfL@hLXmMCA?_GAORA_nEWwFH?GxGIPTIpeKxiLXkLkA?oHAGRAgnE_vFXAG`EIXUIxeL@hLPnMK@?WFAGQAWnEWsEh?GHAIPRI`_KHaKXcKk@?WH@WLAgoEwzFhAHHJHxOIX]Jx_K`tM{@?WI@_OAopEoyFp@HHII@PI`\\Jx`KhuM{@?WG@oNA_qF?xF`?HPJHxPIP\\JpaKXtMs@?gF@WMAopEg{FhBGhJH`MI`\\Jx_KXrMc@?oF@_NAgoE_{FpCGpIH`LIX]Jx`KPqMc@?_F@gOA_qEW|FpFH@HHhMIP\\JpcKhqM[@?_I@WNAOoEgxFP?GhGHhOIhXJh`K`sMk@?oG@WOAWpEWvFHAG`GH`PIpXJp`KXrM{@?_H@_MAWpE_wFX?GpFHpOIpXJp_KhsMs@?gG@_LAOoEWuF@@GXFH`NIhXJh_KPqM{@?oI@gMAGqEguFXAGXEHhPIxXJxaKhrMk@?gH@wOAGqE_vFP@G`DHpNIxXJxbK`qMs@?gIAWSAgnFGzFpCGxJI`TIxaK`rM`uM{@?oHAOSAonF?yFhBH@IIXUIxbKhqM`tM{@?_GAGTAonEovF`DGpHIPTIp_KHqMXtMsD?oH@OJ@_oEGyFXBG`WJH[JxgLHsMxwNKC?gG@OMA?pEOuFHDGxWJHZJhfLXrMpwNSC?oG@GL@woEOvF@EH@WJHYJpeLPqMhxNSB?_F@?J@_oEGrF`?HHYJX\\Jp_KHkLhwNKB?gF@OL@woEOtFp@HXZJ`\\JxaK`mM@xNSB?oF@GMA?pEOsFhAHPYJ`]JxbKhnMHwNSB?gG@GRAonEGvF@@HX_KXhLXkLxuMxxNSB?oG@OQAgnE?uFHAHP`KPhLPlLptMxwNSB?_H@OPA_nEOyFX?HHcKhiLXoMHtMpwNKC?oF@ORAonEGtFpEH@`KhfL@lMHrM`xNSC?gF@GQAgnE?sFhDGx_K`eL@kM@qM`wNSD?oF@?PA_nEOrF`BG`aKXeKxmLxqMXwNK@?OF@?H@OnE?pEP?GHAGXCGhEGxGHHIH[@?OB?_D?onE?pEOrE_tEovF?xFOzF_|Fv";
DIGRAPHS_NamedDigraphs.("higmansims") := ".~?@c_GA?_GA?_GA?_GA?_GA?_GA?_GA?_GCA?oOD@gCABOwNCWCAAooOCgCA@ogSDgCAA?cRDWCBAowTDgCBB?sRDGCBAOgNCGCB@o_PCgCEBpGSDWCEC@CRDgCEA?gJBWCE@ocKBgCC@okNCwCDA_wQCwCCAOsQDgCDAOkPDGCD@osODWCDA?oNDgCCA?wODGCCA_oPDWGBC@CSDQGbHASeIWGBBpGRDaGbHq_iIwGBA?gKBa?`HAWfIGGB@ocJBQ?`HQciIwGEB?sTD`w^HASfIgGEAowRD@w^Ha_hIwGEAOgPC`o\\HA_hIgGE@o_NC@o\\HQWfIwGDAOwOD`s^GAGcIwGDA?sQD@o]GQKcIwGDA_kNDPs^GQKeIgGC@owQDPs]GQGfIWGCAOoND@o^GQGdIGGCA?kPD`s]GAKdIGGCA_sOCpo^GAKfIWGD@ooPCpo]GAGeIgKEBOwOC`gZHA[hIrWwMRmB@_kKBpCYEqSeIAgsLR[y_oWGAPOUE@ccHQ_jLb[yMwKE@ogRDP_XHa[hIbOtMBeB@OgLCPWXEq?bHAgpKr[w_oSHB@GTE@g`GaOiKRKxMgKD@owND@cZGQGeIr?qMRiB@?gJC`OWEqCbIAcoKrOz_oOGBO{TEPg`GqSfKRGsMwKC@ooOD`_ZGAGdHrCqLRYB@?cMCPKXEa?aIAcoKrSu_oSGAp?RE@g_GqWjKBGvMGoMCPGSD`[ZFaGcIA{rLbg~PC]JBO{OD@WVEp{bHQkmKb[zNsDEaosPC`KTDpg]GqciJbKsMBxAPWoMBp?RDP[YFqGeHq{qLRc}OCMGA`?QDPWVE@sbHA[nKR_zNSHE`ocNCPSUDpc\\GaSiJbCtMbpBPw_IBpCRD@[XFAKeIA{oLB[{OSUFAP?QCpOVE@oaIQkmKBWxNS@CaOgLB`OTDpc^GQOhJRKxMrtDPw[GAooSDP[WFaCdHasqLBg{OCYFA?sMCpWVEPw_HqkkKbWwNSDBaOgJB@KUDp_^GA_iJBKtLrpAPG_HB?sOCP[YFA?cHQspLb[~OcMFA_kMC@CVEps_HaclKBSwNcDCa?cJB_{QDpg\\GQ_jJB?yMr|?PW[IB?sNC`[ZFACfIaopLBc}Pc]C@O[GAOgWEPo\\GACkJR?pNBtGQShJ`OWGB`CTEPg\\FaOeJQ{wMcLDQS|RTwSEA_oOD@_ZF@{cHasnLrdCPchLSdYC@_gMBpWXEps^Hq_kJrSzOS\\IRtDS`OWFApGUE@k\\FagjJAwwMcPEQsxOTGOE@osPD@cZF@wdIQsmLBX@PslKStYC@_cJC@SWE`s^HQclJbSzOCHGRdHV`OWHBO{REPg[FqgjJAwvMSLDQCpPTWOEA?oQCp_YF@wfIAonLBX?OcdLSDUC@PKSDPWWEPw^GaKmJrGrNBtSTTXV`?SNC@CQE`k[FQGbJa{oKRw~SDDQSwODAooLB`gZF`{_GQolKbK}NspLRc~c_GA?_GA?_GA?_GA?_GA?_GA?_GA?_GCA?oOD@gCABOwNCWCAAooOCgCA@ogSDgCAA?cRDWCBAowTDgCBB?sRDGCBAOgNCGCB@o_PCgCEBpGSDWCEC@CRDgCEA?gJBWCE@ocKBgCC@okNCwCDA_wQCwCCAOsQDgCDAOkPDGCD@osODWCDA?oNDgCCA?wODGCCA_oPDWGBC@CSDQGbHASeIWGBBpGRDaGbHq_iIwGBA?gKBa?`HAWfIGGB@ocJBQ?`HQciIwGEB?sTD`w^HASfIgGEAowRD@w^Ha_hIwGEAOgPC`o\\HA_hIgGE@o_NC@o\\HQWfIwGDAOwOD`s^GAGcIwGDA?sQD@o]GQKcIwGDA_kNDPs^GQKeIgGC@owQDPs]GQGfIWGCAOoND@o^GQGdIGGCA?kPD`s]GAKdIGGCA_sOCpo^GAKfIWGD@ooPCpo]GAGeIgKEBOwOC`gZHA[hIrWwMRmB@_kKBpCYEqSeIAgsLR[y_oWGAPOUE@ccHQ_jLb[yMwKE@ogRDP_XHa[hIbOtMBeB@OgLCPWXEq?bHAgpKr[w_oSHB@GTE@g`GaOiKRKxMgKD@owND@cZGQGeIr?qMRiB@?gJC`OWEqCbIAcoKrOz_oOGBO{TEPg`GqSfKRGsMwKC@ooOD`_ZGAGdHrCqLRYB@?cMCPKXEa?aIAcoKrSu_oSGAp?RE@g_GqWjKBGvMGoMCPGSD`[ZFaGcIA{rLbg~PC]JBO{OD@WVEp{bHQkmKb[zNsDEaosPC`KTDpg]GqciJbKsMBxAPWoMBp?RDP[YFqGeHq{qLRc}OCMGA`?QDPWVE@sbHA[nKR_zNSHE`ocNCPSUDpc\\GaSiJbCtMbpBPw_IBpCRD@[XFAKeIA{oLB[{OSUFAP?QCpOVE@oaIQkmKBWxNS@CaOgLB`OTDpc^GQOhJRKxMrtDPw[GAooSDP[WFaCdHasqLBg{OCYFA?sMCpWVEPw_HqkkKbWwNSDBaOgJB@KUDp_^GA_iJBKtLrpAPG_HB?sOCP[YFA?cHQspLb[~OcMFA_kMC@CVEps_HaclKBSwNcDCa?cJB_{QDpg\\GQ_jJB?yMr|?PW[IB?sNC`[ZFACfIaopLBc}Pc]C@O[GAOgWEPo\\GACkJR?pNBtGQShJ`OWGB`CTEPg\\FaOeJQ{wMcLDQS|RTwSEA_oOD@_ZF@{cHasnLrdCPchLSdYC@_gMBpWXEps^Hq_kJrSzOS\\IRtDS`OWFApGUE@k\\FagjJAwwMcPEQsxOTGOE@osPD@cZF@wdIQsmLBX@PslKStYC@_cJC@SWE`s^HQclJbSzOCHGRdHV`OWHBO{REPg[FqgjJAwvMSLDQCpPTWOEA?oQCp_YF@wfIAonLBX?OcdLSDUC@PKSDPWWEPw^GaKmJrGrNBtSTTXV`?SNC@CQE`k[FQGbJa{oKRw~SDDQSwODAooLB`gZF`{_GQolKbK}NspLRc~";
DIGRAPHS_NamedDigraphs.("hoffman") := ".O____h@DEFaDFGbDEG`CEFaCFGbCEG`ABDo____h@DEFaDFGbDEG`CEFaCFGbCEG`ABD";
DIGRAPHS_NamedDigraphs.("hoffmansingletonbipartitedouble") := ".~?@c_GA?_GA?_G{GAOgJB?sMaGaGaGaGaWiMbWmKaWiMbGmLaWeHaWqLbgmJbgqLbGuJbgiIagiLbgmK_p?[Ha[gIWOOFqOjLBaECACdIbGv`p?_GqotLgSOFaGlKreAC@smJr?p_Po\\F`{_GWCcIAgnLRe@HQcjJbKu_QKeJRCqMGCaHqooLB]@D`[WEPgZ`@SYFAwtLw[QEPooKbeDD@k[KROu``KWFA{rMGORDq?dKReFD@[`Gaww``SVFaOoLgSQDp{bJr]CCPkbIB?r`PCWHA[mKgWPEQGhKRUFCPgdHa{s`@OUFackKg[TD`{fIbMDCpW`Hakt``GUGA_lLGOQE@saHaiDDPc\\HQ_k`pKZFQOhJWWSE`sbHqmBCPWuLr_x_pKXFaKiJgKSE@{dJRABC`g`HAop_pSZGAGjJwGPDqgjJAuACpg^IBGu_`OXGAWrLwGTEAChLBeAC`k]HrSwxGA?_GA?_GANA?cIAooLBgaGaGaGaGeIbguJbGeIbgqJbWeHaWeKbWyJawyKbWqLawyIagiIbWyJbGKOFAWfIAeCC@{cIrOw``?`HQgqLw[OGAKkLRYDC@waJRKx_`?\\Ja{oKWC[FPw^GAE@HA_iJrSx_QShIqwrLgCbHaspKba@Ga[kKBOv_PWVE@cYEwOTE`omLR]FC`c[KBGx`POZFBCsLgWRE@onKraCCp[_HRCx`pOVGQGmMGWTDpwcKBYDC`[^Gq{v`@CZGq_oKwSPEAOfJbIECPcaIRCt`pCYHQWnLGOSD`whJBIFDPW^Hqgr`PKUGQWjLWWQDa?gJRQCC`_\\GaWi`PSXFQSgJG[REpscIQuED@g\\Gq[j_pCULb[wMWKREPwbIayBD@_^HQso_pGYGQOkKWKTEq?aIq}ACP[iIqol_`KYFq_qLgGSEQ?eKr]ADP_`IROx_`GZFa[tMN";
DIGRAPHS_NamedDigraphs.("hoffmansingletoncomplement") := ".q_O?K?AA_?OOM?@@?oc?AA@`?k?AA@`?gY?@@?o_SKF_?OOKGDB@pC?AA@`?gWMGCo?GGECA`p?cT??__WSKFCAPOm?@@?o_WMGCa`Wq?@@@?gWMGCa`WoZ??__WOIGCa`WoYM_?OOKGEBa@GkWLFBw?CCBA`_wcSJEBPo{`??___SKFCA`WoYMFcAK?AA@`Oo[OHDAp_w]OGo?GGECA`p?cSJEBPx?cR_?OOKGEBa@GgULFCAHGeS_?OOOIEBa@GgWLFBqHGeSIo?GGECA``WoYMFcAHGeSIdg?CCBA@_w_SKEb`x?cRIDQp]?@@?o_SMGCapgw]OGcq`SkVKO?GGEDBA@GgUKEbqHGeSIdax_r??___SKFCa`Wo[OGcaXOiUJeBHi?@@`?gWMGCap_w]OGcaXSkVKERPm?@@`?gWMGDAp_s]OGcaXOkVKERPkx??_o_SKFCa`Ws[NGCQPKgTJeBHgu[Mo?GKGDBA@GgWLFBq@CcSIdax_qYLfBhy?@@?o_SKFCa`_s[NGCQXOkVKERXs{^_?OOKGDB@p?gULFBqHGeTJDr@cs[MfcC?AA@`?gWOHDAp_w]OHCq`SmWLEr`w}_Oo?GGECA`_w_QJEBQ@CcSIdaxcsZMFRy@Aa_?Ow_QIDb@gw]OGcaXOiUKERPkw\\NFsADCb_?_WOKFCa`WoYNGCQPKgTJDr@gu[MfcADCbQO?OKIEBa@GoYMFcAHGeSIdax_qYMFby@AaPhCk?CBA@Ow_SJEB`x?aQHdAhWmWKer`s}_OgcYPIe_?__SKGCa`Ws[NGCQPKgTJDrHgu\\NFsADCbQHSq]?A@`?gWMGCa`WoYMFcAHGoXLEr`s{^OGSQLGdRHtC?ECA`_w_QIDb@gweSIdax_qYLfBhw}_OgcYPIeRiDKCCBA`_w_SJEBPp?aQHdAh[oXLErhw}_OgcYPKfSITSCCBA@OwcSJEBPo{aQIDQp[oXLEr`w}_OgcYPIeRiTQmAA@`?o_QIDb@gw]OGcqhWmWKebXoy^OGSQLGdRiDIhUk____SKFCAPWoYMFcAPKgUJeBHgu[MfcADCbQHSq`QiTjDkCCBA@Oo[OHDBq@CcRIDQp_qYLfBhw}_OgcYPIeRiDIlWlVOOWOIEBa@GgUKEb`x?aQHdAhWm[Mfby@AaPhCiXMgTItat[n`?o_SKFCAPOkWLFBq@CcRIDQp[oXLEr`s{^Qhcy`QiTjDix]oxOG?E?@@O?GGF??__WQ?@@?o_U?@@?o_SL??__WOIEBo?GGECA`_wa?@@?o_SKFCAW?CCBA@Ow_QI_?OOKIEBa@GgV??__WOKFCAPOkX??___SKFCAPOkWL_?OOKGDCAPOkWLFO?GGECB@p?cUKEb`{?AA@`Oo[QIDb@gw]O_?OOOIEBa@OkWLFBq@E?@@?ogWMGCa`Wo[NGCW?CCBA@Ow_QIDb@g{_QHo?GGECB@p?cSJEba@CcRIO?GGGDB@p?cSKEb`xCcRIDW?CCBA@OokWLFBq@CcRIDQs?AA@`?o[OIEBPo{_QHdAhWn??__WOIFCAPWs[NGCQXOiUJeG?CCBA``?cSJEBPxCcRIDQp[oX_?OOOIEBaPOkWMGCQPKgTJDr@ct??_o_SKFCAPWo[NGCQPKiUJeBHgv??_o_SKFCA`WoYNGCQPKgUJeBHgu[_?OWOIEBaPOkYMFcAHGeSIdr@csZMFW?CECA``?cSKEb`x?aQIDQp[oXLEr`s|??__WOIEBaPOoYMFcAHKgUJeBHky]No?GGECA`_w_SJEb`xCcRIdax_qYMFRqA?@@?o_SKGCa`Wo[NGCaXOiVKEbXo{^OGW?CCBA@Oo[OHDb@h?aQIDQp[qYLfBh|?`PO?G[OHDAp_s[NGCQPKgTJEBHgu[Mfby@AaPo?OKGEBaPOkWLFcAHGeSIdax_sZMFRq@AaPhG?GEDB@p?cWLFBq@CcRIDQp[oXLFBp|?`PGsaU?A@`?g[OIDb@o{_PHCq`SkVKERXoy^OGSQLGdRO?OOIECAPOkYMFcAHGeSIdaxcsZMfby@AaPhCiXN?@?o_SKFCAPOkWLFBq@CcWKebXoy]NgCIHEcQhcya?BA@Oo[OHDAp_s[RIDQp[oXLEr`s{^OGSQLGdRHtAeAA@`Oo[OIDb@gw_PHCq`SmWKebXs{^OGSQLGeRiDIiAA@`?g[QIDb@gw]PHDAhWmWKebXo{^OGSQLGdRHtIhV@@?o_WOHDAp_s[NGCQXSkVKERPkw\\NgCIHEcQhtAdSjUOOOOIEBa@GkWLFBq@GeSJDr@csZMFRq@AaPhCiXOhTItauAA@`?gWMGCa`x?aQHdAhWoXLEr`s{^OGSQLGdRHtAdUkUjgGKGDB@p?cSJEBPo{_PHCq`SkVMFRp|?`PGsaTKfSIdYpYmVo_WOIEBa@GgUKEb`x?aQHdAhWmWKebXoy]NhSq\\OhTItat[nW^";
DIGRAPHS_NamedDigraphs.("hoffmansingleton") := ".q_OGCA@?_OWKEB@_o`C[YMaA`sKWMbApsSQMbAG_aMG`aGgaCIIo``]CHKO`hYCJIOpG{gXaBAP_w]aaq@Ws]bBAXWrDEcaxeMJGeBKWYPIesC[QQIegWkeTMPP_{i\\aBQXOsbaAQHWyaaApx[u`bA`x_seSpPPCg[QigwgeVMfscKYOKFSIUEIHDbY\\PECcAxpEfbbA@OuaR[gCA@?_OGCEB@_oWKGPFEbg_g\\BEBgok\\DCbgoaGGbaGWaIG`AakGWV`ARCGYU`AqcKQNIEW_ocWMFggk_ULFgooeUKpPhGmXbaqH_rEEcQhl@FCcahiEJHdRcSWNIfW_seSLGw_caUMgg_k]VLgWog]WLHdKSSPIFCiiMIHdrh|HBEcB@tAd`aaPWufSP`H?m[Phwwo_SLgcv";
DIGRAPHS_NamedDigraphs.("hoffmansingletonline") := ".~?An_C??K??GA_?@?OB_?@?OB?c??GA?WC?k?_?F_?F@C??wG@K??wG@GI_?F@?H@OJ`oL`gM`gM@{L@oNACVASQA{QAwWaOVB?Xa[RB[RBW[aWZB_\\acSB{SBw_a_^C?`akUagbaocaocCsTCWdagbCggaocCofagVaWWaW^DcRCgkDkRCwkDgmaO[aO_ECQCooEKQDGoEGqa_ba_cEcSBosEkSBOsEguagZD[UBKUBgxag`DWwaogFGyaoaFGyFcTDOjF?zawfDWnawaDW|F{[CWoEcWC_kEkZCoqFCXCgmFK^ColEPBc?dDopGcbCwnE_~GKWCGkFXAbG]EoxGc^D?lF`Dbg`FOzHCcDGrEhAbghEWyHXKb_aE?|G@@bOZEwwG[_DOpFpEbOgEw{HPNboiEo}HHOaGVDW~GCLCWsGHF`gXFHCHHS`geEPBGhSIkLCGzH@JI`TIsMBWwGXN`ocEhAH`W`odDpCGpWJKMCO|G@MJ@XJSNB?kGPG`w[E@@Hp[`wgF`IIH[JkNDO}I@QJ`\\JsPBwlGhII[OC?pGpOa?\\FPJHh`a?fDw~Gx`KSPBouHHQIX_aGhEXKHhRK@ca?YExNIH`KPb_GL@oNA?P_GRD_lDonK{@AOoEGqEXfLC@A_sEguExfL@h_GTDWwFW}KxgLHi_GUFGyF_|KxgLHiL[F@gM@wOAHf_wQAwWBGYLkFAWZB_\\BplLsFA_^C?`CPlLpn_wTCWdD?hLhmLxo_wUC_eCwiLhmLxoMKBAOVB?XBPm_ORBW[Bg]L{CA_^C?`CPo_gUC_eCwiMSEAgbCggDHp`?RD_lDonLCHAOoEGqEXh`OSE_tEovLSJAoxFO{Fhk`_TDWwFW}L[IAwjFx?IXy`?hEXKHhdNCGCO|G@MJXwNsGDO}I@QJxwNp~`?YExNIHeN@}Ny?`GfDw~GxbNKHD?{HPPJpxOSHCGzH@JIxxOQB`G]EpHIPcNIAOYC`O\\FPJHhaNP|`OdDpCGpYNP|OsICoqGXDIpyNiEO{JCWsGHFI`z`WZF@BHxWNYH`WWD`AH@[NYHPSKBGxG`HIh{`_^DhDHP_NaK`_[E@@Hp\\NaKPkJC?pGpOKHzPIIP[KC_tGPKJH{PaLPsAAwjFx?IXsNkAC?pGpOKHsPyP_OcEhAH`XMaOQIQ_OgF`IIH]MaBQIQQ[BCWsGHFI`rPKBBwlGhIK@rPiT_W\\FPJHhaMYEQiU_WiFpOIP^MY?QiUQ{CBWwGXNJ@tPSCBGxG`HIhtPaX__fDw~GxbMiARIY__hEXKHhdMh}RIYR[DB_oGHMJhuPsEB?kGPGJ`vP[DCgmG`EJPuOy\\_oeEPBGhUMyGRsDCGzH@JIxuOa\\R{EBouHHQK`vOi]SCDBOvHxPKpuOI\\Ry`_oaFh?HpZMx~Rq_SSG@hSIhUIxwNp~OA@`GMJ@XJPZNIAOYCOkI@x[Jh]JxyNiEOyG`_OKHaKXeNaKPiMQCJAHRK@cKhzPIIPYN_OLI`TIpVMaPQQRQad_WMJ@XJPZMYTQqVRAe__NJ`\\Jp^MiXRQZRaf_oOKHaKXeMy]SAaSag_gPIX_K`dMq\\Ry`SYht{?_?@_?@?S??GA?[??GA?WC_?@?OB?_D_C??{??wG_?F@?H_?F@?H@S??wG@GI@[M@kL@sL@oN`gM@wOawQaOVaOVBCQAwWBKRaWZaWZBcRBW[BkSa_^a_^CCSBw_CKTasTC[UCcUC_eagbCkTCWdDCUC_eC{TA{RBCRBwkaWdD_laWfD_lDsQBcQC?oaOeE?paOhE?pESSC[SC_sa_]E_ta_YE_tEsTBWjaoXao\\FKTCGjFCUD?xFSUCOxFO{agiDWwF[VCwjD{VCOjFg~b_bE?sb?cD_tbWeEOwbGdDoxbweDgqG[_CgmEHCcWfDwsFx@b?`D_zGSXBouFHCbwgDg{Gk\\CGyFXGc_hEWtGS\\DGrFPJHc[COoFh?GKYBWvF@Bc?iEG}GsYD?vF`IH{]DOuFpHICPAwjFx?`gbE`@G{LBGxG`HIcLCoqGXDI`T`g`FXGHXSIhU`oZF@BH{MC_tGPKJCMCgmG`EJ@X`oaFh?HpWJHY`wWD`AHCNB_oGHMJcND?{HPPJ`\\`wiFpOIP[Jh]aG^DhDHPRa?_EHEICOBgyHXLKKOCwnFxFKHaaG]EpHIPRKCPDGrH`LIX_KcOBOvHxPKHaK[@@gM@wOAK@AWkDgmDxf_GQE?pEOrKxg_GSE_tEovKxgLK@AgjF?zFpfL@hLS@AoxFO{FhfL@hLPj_wL@oNA?PK{FAOVB?XBPl_wRBW[Bg]Lhm_wSBw_CGaLhmL{FAgbCggDHlLpnMCFAocCofDPlLpnM@p_WQAwWBGYLsAAWZB_\\Bpn__SBw_CGaMCDAocCofDPq_oTCWdD?hMKGAWkDgmDxg`GQE?pEOrLKIA_sEguExi`WUFGyF_|LcKAgjF?zFpj`OVDW~G@RNSGDGrH`LKhw`?aFh?HpZN@}`?iFpOIP^N@}N{GBOvHxPKpwNp~OCHCwnFxFKXx`GgF`IIH]NIA`G`FXGHXVNIAO[HBouHHQK`xOQBOcIBgyHXLKPyNkICgmG`EJPyNiE`OeEPBGhUNP|OqF`WbE`@GxSN[JBWwGXNJ@zPKJB?kGPGJ`zPII`_XFHCHHTNcKBwlGhIK@{PcKB_oGHMJh{PaL`W_EHEI@`NYHPQJ`_cEhAH`XNaKPiM_OVDW~G@RM`|_O_EHEI@`MaNQKAC_tGPKJHsQAPQSAD?{HPPJpsOYPQQR_WbE`@GxSMYH_W^DhDHP_MYLQkBBgyHXLKPrOqTQsBDO}I@QJxrOATQqV__ZF@BHxWMiI__XFHCHHTMiKRKCCwnFxFKXtOQXRSCDGrH`LKhtNqXRQZ_g[E@@Hp\\MqM_oWD`AH@[MyJ_gdDpCGpYMqFRkECoqGXDIpvPA]_g`FXGHXVMqCRi^_o]EpHIPcMyDRq__gYExNIHeMq@Ri^SKECO|G@MJXvNy]SAa`?LI`TIpVN@}Ny?OKH@pWJHYJXxOQBOaD`ONJ`\\Jp^NP|OqFPCKA@`KPbKp{PaLPqO`WPIX_K`dNYHPQJP{A@hSIhUIxsQIQQYSSkB@pWJHYJXrQiUQyWSsC@x[Jh]JxtRIYRY[S{EA@`KPbKpvRq_SQcTCDAHRK@cKhuRi^SIbTN";
DIGRAPHS_NamedDigraphs.("hoffmansingletonminusstar") := ".i_OGCA@?e?O{WOKaapcKQKaA`cGNEBoo{ONDBoP@IAIIOPXUAHHpP@OrCCcbCKSTJP`XKnACCAXaCJFdAsGQLIdwOg[QKp@?{i\\OpPOseZNopX?cYOP`Gwg[NOp?waVLp@WsaXMP`O{aWLPPH?aUMygCA@?_OH_CNECBGgkXBCbG_gXABp_{KNCBpO{COQ_aacCUT_aQ[SOSKp@HGpBDDQsWURJo`@?eW`ApxOlACbQh]CIFCbKOONIfSKSSLHer{KUOHEcCWQMIFBsKOMGdr[OULGeRcWSNGeBSSQOGdbn";
DIGRAPHS_NamedDigraphs.("hortongraph92") := ".~?@[_GA?aOQHaWwLbXCOcH_QDPYPDP]PCpYQD@]SDhKVdXKSgPw_fqAEGHo\\`hyMHAUMGqYNGaYNGQULiymKJGoljIuokY{ojbEqLZKscGqqKzoyMzczmBmuLzaxnsA{OZt?ncFAP[LCaC]FPkfG`ohJa?lI`?UFAGlJaciD@gUCt_LMTWHNTWDRTWLQTGHRTGDOS\\q?_GAH@GeHb_uLcPAOe@GTDhCTDxCRDhGSDxOUcp]TcpQ`FaA^GGW_f@uEfgwcHWwbHg{aHg{`HWujiwokbAukjZApjrAmKZGtkrQObJGrnBgzmRmwMzWvmJe~OJp@nSA}O[HDosQGPw\\Eq[aFAcmGAsiC@W[GasmIQgSE`WRU?sxT_c|T_TLT_tHS_dLS_T@P";
DIGRAPHS_NamedDigraphs.("hortongraph96") := ".~?@__GA?`oUD`gYL@oqGBGcJa_mFaGeI`GQPDXGTcpYSDgwPbPIOCw{Se@mXFhg\\dpq^Gi?`hIMehiUd`WYfIy_jiQqiJGqJ`I]giierkzCskRUqLjGvjbQlLZ?vjrZCO{MwOjdAmb}zNjt@nCBBojaxrCzKR{\\Mpc~CR[TLqSnGQkVC_t\\X_dXW_dTX_tPW_T@R_TDQwGA?_G[D`WYEbO[Ka?qHAwgJ`waHagQCcPUQDXKUd@YMCWsQc@MNDH_ZePyYFX[[fqI_GYQbhiYdhWUEhqmgIyckiaqKawQfiIihkzMpLJCtkbYqLywsjRUoLy{upCNBmCIxOjg~mry|OZp?o{Iwm[pMrC~FRkXNpCvDR[dJqCjDpGLVUWHUUGHTUWLSUGDOSwDPSn";
DIGRAPHS_NamedDigraphs.("house") := ".DaoIQNWkAcb";
DIGRAPHS_NamedDigraphs.("icosahedral") := ".K`?K?oD?XJ`CYGrFcOpxdKa`RHTqC?oB?S@cmCPgbK]PBFeSqIDKdV";
DIGRAPHS_NamedDigraphs.("icosahedralline") := ".]___@_A_@B_ACD_@BD_ACEFiHhiJhIJK`HIaHIMbJKcJKO`IKLMaHJLNcHJLPRbIKLOQ`BMOQTaCNPRSdEHIMNfGJKOP`DEMQUWaDENRVWYbFGOTUXcFGPSVX[}___@_A_@B_ACD_@BD_ACEFiHhiJhIJK`HIaHIMbJKcJKO`IKLMaHJLNcHJLPRbIKLOQ`BMOQTaCNPRSdEHIMNfGJKOP`DEMQUWaDENRVWYbFGOTUXcFGPSVX[";
DIGRAPHS_NamedDigraphs.("icosidodecahedral") := ".]___@_AeDddEdFefIhgKeIfJhKgLiJkLmPnO`MSaNTcOTbPS`QUaQVYbRXcRW[}___@_AeDddEdFefIhgKeIfJhKgLiJkLmPnO`MSaNTcOTbPS`QUaQVYbRXcRW[";
DIGRAPHS_NamedDigraphs.("kittell") := ".V__@`A_@eDe`BCbGdeFijKkLdEIJKLfJLMcGHiKMPhMOPQ_ADEFaBFHORS_CDIPQSv__@`A_@eDe`BCbGdeFijKkLdEIJKLfJLMcGHiKMPhMOPQ_ADEFaBFHORS_CDIPQS";
DIGRAPHS_NamedDigraphs.("kleindistance2") := ".W__@_@A_@AB_@AB_@ABC_@ABD_@CF_@DEG_CDGH_ACEGHI_BDFGHIJ_ACEFGIJ_BDEFHIKL`BCDEHIM`ACDFGILN`BCEGHJKN`ADFGHJKOPaBEFJKLMPQbCFGIKLMNOPRaDEHIJLMNOQR`EFGHLMNOPQRSTaBCDIJKNOPQRSTw__@_@A_@AB_@AB_@ABC_@ABD_@CF_@DEG_CDGH_ACEGHI_BDFGHIJ_ACEFGIJ_BDEFHIKL`BCDEHIM`ACDFGILN`BCEGHJKN`ADFGHJKOPaBEFJKLMPQbCFGIKLMNOPRaDEHIJLMNOQR`EFGHLMNOPQRSTaBCDIJKNOPQRST";
DIGRAPHS_NamedDigraphs.("klein") := ".W__@__@_A_BC_BDi@AGaEHI`FHIcEGIJdFGIKcDHaBHJ`BHKOcFHKLNdEHJMN`CGNPaDGNOSbFGLOQTbEGMPRSw__@__@_A_BC_BDi@AGaEHI`FHIcEGIJdFGIKcDHaBHJ`BHKOcFHKLNdEHJMN`CGNPaDGNOSbFGLOQTbEGMPRS";
DIGRAPHS_NamedDigraphs.("krackhardtkite") := ".I`OKhdcOkqENGXRFPiD?qeUPArGW{`dK\\F";
DIGRAPHS_NamedDigraphs.("kroneckerproductoficosahedralgraphcomplementandonesmatrixj2") := ".W__@_@_@_@_AB_CDE_CDE_ABFG_ABCD__`ABCDEFGHIJK`EFGHIJK`EFGHIJK`CDEHIJK`CDEHIJK`ABFGIJKOP`ABFGIJKOPaBCDEHJKMNQRaBCDEHJKMNQRaBCDFGJKMNOPaBCDFGJKMNOPw__@_@_@_@_AB_CDE_CDE_ABFG_ABCD__`ABCDEFGHIJK`EFGHIJK`EFGHIJK`CDEHIJK`CDEHIJK`ABFGIJKOP`ABFGIJKOPaBCDEHJKMNQRaBCDEHJKMNQRaBCDFGJKMNOPaBCDFGJKMNOP";
DIGRAPHS_NamedDigraphs.("kroneckerproductofpetersenlinegraphcomplementandonesmatrixj2") := ".]__@_@_@AB_@AB_@AB_@AB_@CDEF_@CDEF_@ABCD_@ABCD_ABCDEFGHIJ_CDEFGHIJ_CDEFGHIJ_ABEFGHIJLM_ABEFGHIJLM_@EFIJKLM_@EFIJKLM_ABCDGHNOPQ_ABCDGHNOPQ`ABCDEFGHIJKLMNOPQRS`CDEFGHIJKNOPQRS`CDEFGHIJKNOPQRS`ABGHIJKLMNOPQRS`ABGHIJKLMNOPQRS`ABCDEFKLMNOPQRSUVWX`ABCDEFKLMNOPQRSUVWXaBCDEFGHIJLMPQRSUVWXaBCDEFGHIJLMPQRSUVWX}__@_@_@AB_@AB_@AB_@AB_@CDEF_@CDEF_@ABCD_@ABCD_ABCDEFGHIJ_CDEFGHIJ_CDEFGHIJ_ABEFGHIJLM_ABEFGHIJLM_@EFIJKLM_@EFIJKLM_ABCDGHNOPQ_ABCDGHNOPQ`ABCDEFGHIJKLMNOPQRS`CDEFGHIJKNOPQRS`CDEFGHIJKNOPQRS`ABGHIJKLMNOPQRS`ABGHIJKLMNOPQRS`ABCDEFKLMNOPQRSUVWX`ABCDEFKLMNOPQRSUVWXaBCDEFGHIJLMPQRSUVWXaBCDEFGHIJLMPQRSUVWX";
DIGRAPHS_NamedDigraphs.("kummer") := ".__OGCA@?g_OO[OHDP?g[QKFp?o[SLFPOo[OJGOo_cSJGOog_QLFOoo_SKFoOW[UKEo_W[[NGO___WLGO_ggUKFO_ocULFoO__UMFoOggYNGOOocWMGWGCA@?_OIGCCFCAPSOIFCb@{OKFDBPsSKFCAqCKGHDAqCKIGCbPsKKGDB@{CEFDb@kGEFFBqCGGGEBQCGIIDb@sGKHDbP{CGGDb`{CIIEbqCCKHEBaF";
DIGRAPHS_NamedDigraphs.("levi") := ".]___kCDeFgJhIfGeHfIeJcGdHdJcI`KL`MNaPSbOTbRVaQU}___kCDeFgJhIfGeHfIeJcGdHdJcI`KL`MNaPSbOTbRVaQU";
DIGRAPHS_NamedDigraphs.("ljubljana") := ".~?@o_GA?b_mKbWUF`hSRDHCSc`QZBpEMCx?QbhANh@caeAEYGxo_fPyZFx[bdQIUGWgJa?uHBGSJ`_uFBGSG`oiEAX_]eaAXFxiXeIEbggQC`GeIaKWuPZ`DlsUzPJhAmSMcNIW}hRugMYczhriiNyp@isAlNyx@jsApNJG}kBurOjTBlCRKR{xPrTBHSklSqdNTUL\\YtdfEU{`\\ptqAVeEBWEM@VuI@XE]AXeeBXUbo_GA?b_mKbWUF`hSRDHCSc`QZBpEMCx?QbhANh@caeAEYGxo_fPyZFx[bdQIUGWgJa?uHBGSJ`_uFBGSG`oiEAX_]eaAXFxiXeIEbggQC`GeIaKWuPZ`DlsUzPJhAmSMcNIW}hRugMYczhriiNyp@isAlNyx@jsApNJG}kBurOjTBlCRKR{xPrTBHSklSqdNTUL\\YtdfEU{`\\ptqAVeEBWEM@VuI@XE]AXeeBXUb";
DIGRAPHS_NamedDigraphs.("localmclaughlin") := ".~?Aa_C??K??K??K??K??K??K??K??K??K??K??OB_?A?c??OG_?A@K??OJ_?A@S??OF_?A?k??OE_?B?_RAc??WFA?P_?B@GOAc??WGAGR_?B@OL@{??WJ@gM_?B?gNAS??WE@oQ_?C?gPAOVBS??_EA?QB?X_?C@WNAWUBc??_I@oSAoZ_?C?wM@wVBC?@GIAOSB?YB{?@?JAORAwXCC??wJ@gSB?Z_?F@OLAWVBc??gJAGSBG[CK??oIA?RBOZCK??gH@gOB?[CC??oG@gPAwZB{??gG@wSAoYBod_?E@GMAWUBG\\Cc?@?H@gQAgUCof_?H@WMAGTBW]Ck?@?I@wOAg[Bgc_?D?wMA?TBGaDK??oF@wPAgYCWg_?C@GKAG[CWcCwiD{??_G@_OBWaCgeDWo_?F@?KA_XBg^CwgDk??wH@_RBO]C?eDGm_?D?oK@gUCGaCWlDs??gI@_MAw]BwcD_o_?E@WK@wWBg_CgkD{?@OJ@_QAg`D?hDOj_GA?WKBg]Bw_CGaCWcCgeCwgDGiDWkDgmDwo_GA?_LAoVB?ZB_aCWeCwiDWlDonE?rE_uEww_GA?gRAoVBGYB_]C?`COcDGkDgmE?pEOrEww_GA?oSAoWBGYBW\\Bw`CWdD?kDgmDwpEOsEow_GA@WOAgWBGZB_\\C?`COdDGiDWkE?pEWsEgu_GA@OPAgVBOZB_]Bw`CWcD?iDWkDwqEWsEgv_GA?wQAgVB?XBO^C?eCwgDGiDWlDopEOtEov_GA@GNAgUB?YB_\\Bo_CWcCgeCwhDwqEWtEow_GA@?MAgUAwXBW\\Bo^COcCgeCwgE?pE_tEww_GB?_M@wOAGQAgaCWcCgeCwgDGiDWrE_tEovFW{_GB@WNAGQAWSBO]C?`CggDGjD_mDwpEOrEguFP@_GB@OMA?QAWSBG\\Bw`C_gDGiD_lE?pEOsEgvFP?_GB@?L@wOAOSB?\\Bw_CgeCwjDgnE?pE_tEowFW}_GB@GL@oPAORAw]Bw_C_eCwiDonE?qEWtEwwF_|_GB?gL@oOAGSBW]Bw`CObCgjD_mE?pEWsEwwFx?_GB?oL@wOAGRB_\\C?`CObC_iD_lDwqEWsEowFx@_GB?wL@oNAWSAo\\BoaCWeCwgDGlDopEOuEwwFg}_GC@?NAGQAWSAoVBGYB_cCwgDgnEOsEguEwwFG|G{@?_H@oOAORA_UB?XBOZCgeDGmE?pEWtEovF?xFpG_GI@WL@oNAWSAoVB?ZB_\\BokDwoEGqEWsEgwFG~GS@?gI@gNA?QA_TAoWBO[CGbDGjDgpEOrE_uEwxGHE_GE@WL@oPAORAgUAwXBW`COgDOmEGqEWsEovFH?Gk@?wH@gMA?PA_TB?XBW[Bg^CWfDOqE_tEovF?xFXB_GF@?L@wOAGRAgVBOZB_]C?aCojEGrEguEwwFG{Gc@?gE@oNA?PAOTAwWBGYBw_C_dD_pEOrE_tF?xFPH_GC?wKA?PAWSBGYBW[CGaCWgDGiDWkDgmEovG@@GhEH`P_GC@WK@oPAOSAwWBGZBwdCwgDOjD_mDwoE_tFX?H@HHhO_GC@OK@wOAORAwWBO[C?cCohDOjD_lDwoEWtF`@GxHHpN_GC?gK@oNA?SAoWBG[BgaCWcCgfDGkDgoE_wFo~GXEHpO_GC?oK@oNAGRAoVBOZBoaCWcCgeD?kDonEWwFg~G`DHhN_GH@WK@gQAWSAoWBG[Bg_CGfDGiDgmDwoEOuFp@GPFI@PIs@@?I@_LAORA_UAwYBW]Bw`CogDWlDonE?pEw|G@AH@NIHT_GG@GKA?PAWSBGYBW[Bg]Bw_CGcCgkDwoEgwFO~GPHHhM_GD?wK@gNAGSAwWBO[Bo^C?bCwjD_lDonEOwFh@GPCHXM_GE?wK@gMA?RAwWBGZBg^C?aCoiD_lDooEGwFp?GPBHPL_GD@?K@gOAGQAgVBG[Bw_CGaC_fDOjDgoE_vF`?GXHHXKIs@?oH@_LA?PAOTB?YBW^C?`CWdCoiDWmDwrEozGHCHHIH`T_GG@WK@gNAGSAgUBW[Bg]CGaCWdCwgDWnE_uFW~G`EHXPI`Z_GH@OK@gMA?RAgUBW[Bg]CGaCWcCohDOoEWvF_~GXDHPPIXY_GD@GK@oPAOSAgUBGYBo^CGbC_dCwgDGmEOvFO|GhGH`OI`Z_GE@?K@wOAORAgUBGYBg_CGaC_dCogDGlEGuFO}GpFH`NIXY_GF@OK@oNA?SAgWBOZBg]BwbCgeD?hDWkEGtFOzGpGHPMIx[_GF@WK@oNAGRAgVBG[Bg]C?aC_fD?hDOkEOtFO{GhFHXLJ@\\_GD@WK@gM@wQAgUAwWBo_COdCofDGjDooEGrF_}G`GHPNIPX_GE@OK@gM@wQAgUAwWBg^CWcCofD?iDgnEOsFW|GXFHXOIPX_OB?_D@?JAOSAoXCGaCgfD?hDWlDooEGsEovFp?GpGH`NI@PI`UJHYJX\\JxaKXd_OB?_E@GIAORAoYCGbC_eD?hDOlDonEOrEovFh@GhFH`NI@PIXTJHYJX[JpaKXc_OB?_F@OJ@wSB?[BgbCwgDGiDWkDgnEOsEguFX@GpFHXMI@PIpWJHZJ`\\Jx_KHc_OB?_F@OJ@oRAwZBoaCogDGiDWkDooEGrEgvF`?GhGHPLHxPIhVJHYJ`\\Jp_KHd_OB?_F@?HAGSBOZBo^CWdCofD?jDonEguEwwFW|G`GH`LHpPI`TIxZJ`^KHbK`d_OB?_F@?HA?RBG[Bg_COcCofDGiDgoEguEwwF_}GXFH`LHpPIXUJ@YJh]K@aK`d_OB?_D@?J@wPAw[Bo_COcCgfDWkDwoEWsEgwF_~G`HHXLHpNIPVJ@ZJh^K@`KPdK{A?WC?oH@OMA?WBW\\BwbC_dCoiD_nE?rE_tF?zFxBHHIHhMI@QIxWJP[Jp_KHbK`e_OB?_D?oF@wOB?YC?aCWdCohDWkDgmEGrEowFp@G`EHPKHpNIXVJ@XJ`]Jx_KXd_OB?_D?oF@oPAwXBwaCWcCwgDOkDgmEOsEwwFh?GXDHXKHhOI`VJ@XJh]Jx`KPc_OB?gE@OJ@gQAwWBw_CGiDWkDgmDwoEGqEWsG@@GPHHPJHxOIPTIpXJp^K@`KPbLPj_OB?gG@GI@gSAo[Bg]Bw`CWcCwlDwoEOsEwwFg~GPBHXMI@PIPRI`UJX\\KHaKXcLHm_OB?oG@GJ@gRAoZBg]C?`COdComDwoEGrEowFo~GPCHPLHxPIPRI`TJP[K@aKXdL@n_OB?gF@GJ@oSB?XBg]Bw_CgfDGkDooEGqEgwFO}GPGHPLHpOIPSIpWJ`\\Jp^KHdK{A?WE?wG@ONAWVBO\\Bo^C?cCogD_lDwpEOtF?yFhAGxJHhMHxQIXTIx[Jh]Jx_K`e_OB?gG@GIA?QAgYBo^C?`C_dCohDWoEGrEgvFO{H@HHPKHpNIPRIhUIxYJX]KXdL@n_OB?oG@GJAGQAgXBg^C?`C_dCwgDOnEOsEguFOzGxHHXKHhOIPSIhUJ@YJX^KPcLHm_OB?gF@GJ@gPAg[Bo_CGaCWfDGiDWmEOrEovF`@G`DHPJH`PIXSIhUJ@XJX`KPdLhs_OB?oF@?I@gOAgZBg^CGaCWeD?iDWlEGsEovFX?GXEHPJH`PIXSIhUIxXJP_KXcL`r_OB?gE@OJ@oNAgUBg]CGaCWcCggDGkEGqEWsFO~GhEHPJHxOIPRI`VJ@XJPZJ`\\LPj_OC?gF@?IA?SAwWBGYBW[BwjD_lE?pE_tEwwFH?GPBGpGHHMIpVJh]Jx_KHbK`dKxqMpvNKA?_E?wH@WPAWVB?XBOZB__DOkDonEOrEguF?xGHAG`DGxHHhTJ@[Jp^K@`KPcKheMHtN@x_OC?gH@OJ@oQAgUAwWBG[C_fDGiE?qEWsEgvFG{GXDGxGHHOIPUJ@XJPZJh]KHaLPmMPsNCA?_E@?I@WNAOTAoVB?YBWdCogDWnEGrE_tEoxFXCGpFH@HHxQIhVJHYJX[Jx_KXjLxpMXv_OC?gE@?HA?PAgUBGYBW[CGaCWcCgrE_uEwwFG~GXCGhEHHKIXSIxWJPZKPbK`dL@hM@rMcA?wG@GI@WRA_TAoXBOZB_\\Bo`D?hEGqEguEwxFPAGhEGxGIHRI`TIpYJX[JhcKhkLhmLxo_OD?oF@?J@gNAgUAwWBG[Bg_COfDgpEOsEowFG}GPBG`EGxJIPRI`UJ@XJh^K@aKxhLPlMkA?gE?wH@OL@oTAoVB?YBW]BwbComEGqEWvF?xFhAGXCGhGHPQIXSIhVJH[Jp`KXeL@jL`u_OC?gH@OJ@_SAoWBOZB_]CGbCghDWkDonE?rFx@GPCGhEH@HHPMHxOIHZJ``KXdLXnM`uNA?_OC?oG@OJ@_RAoVBGZB_\\CGaC_gDOkDgnE?sFx?GPBGhEGxHHXLHxOIHYJh_KPcLPmMXtMy@_OC?gE@?H@_QAoVB?XBO^C?cCgeCwlDonE?wFg}GPBG`FH@HH`LHpNI@QJp^KPbL@hMxwNH~_OC?gF@?I@_NAgUAwYB_]CObC_eCwgDGjDgvF_|GXCGhEGxGHXKHpNIHRIxXJX\\LhoMPrMpz_OC?oF@GJ@_MAgUB?XBW\\CObCgeCwgDGiDouFW}GXCGhEGxGHPKHhOIHSJ@XJP[L`oMHsMhy_OF@?H@OJ@_LAgVB?ZB_\\Bo^C?eCwiDWnE?tFW{GPBG`FH@HHPJHhMIHQIhUK@`KpfLpnNH}_OD?oF@?J@_PAgVBGYBW]Bw_CGaCggDWkDopFP?GPCGhEH@HHPJH`LHxSIhVJxdKxgLXlMH{_OD?oF@GI@_OAgWBGYB_\\Bw_CGbC_hDOkDgqFP@GPBGhEGxHHPJH`MI@RIpWJpcKphLPkMP|_WC@?H@OJ@gOAGQAWSBW[CGiDWnE?rE_tEovFGzF_~G@@HHPIhUJPZK@`KPbK`dLpnMXsNI?OICOiEPAH_WC?gF@GI@oNA?PAWSBO[BobC_hD_qEWtEwwFGyF_|Fx@GhMIXVJ@ZJ`\\Jp`K`dKpoMPuN@|OABOaEOyG_WC?oF@?J@oNA?PAWSBGZBgaCggD_pE_tEowFGyFW}Fx?GpLI`VJ@YJ`\\Jx_K`dKxoMHtMx{OIAOaDOyH_WC?gE@GJ@gM@wPAOSAoWCWdCwmDwqEWsEowFGzFg}Fx@G`OIPSJ@XJX[Jx`KPbLHjM`tN@yNyBOiFPAH_WC?gE@?I@gM@wOAORAoVCOcColE?pEWsEwwFG{Fg}Fx?GXNIPRIxXJP\\Jp_KPbL@iMXuMxzNyAOqFPAH_WD?oF@?H@gOAGQAWSBGYBw_CGlDopEOuEwwFGyFg}G@@GPKIXSIhUJp^KPbK`dL@hL`lNH{NiAOYDOqF_WD?wG@OJ@gM@wPAOSAgVBo^CwgDWpEOsEgvFGyFW{Fh?H@JIPSIhUIxXJX\\Jx`KxjLhmMPzNqAOYCOqH_WE?wH@OJ@gM@wOAORAgWBg_CohDOpEOrEguFGyFW{Fp@GxIIPRIhUJ@XJP[Jp_KpiL`nMHyNqAOYCOiG_WC?gF@GI@_L@oOAOSB?^CWeCwhDOjDgmE?vFW{Fg}G@@GXGHPKHpOIHUJH]KHbL`qM`uNHzNh}Ny?OYGPcB?_E?wG@WK@gNAGQAWVC?aCofD?iDWlDonEozF_|Fp?GHCGxJH`LHxPIhXJx_KPlMHrMhxNP{Np~OIAPIJ_WC@?H@OJ@_M@wQAWSAo\\BocCgeCwgDGnE?tFOzF_|Fo~GxGHhMHxOIHQJPZJ`\\LpnM@vN@yNX}OA@PAHP{B?_D?oG@OK@wOAGQA_YBw`CWcCggDWkDgnE_yFW|Fx?GHEHHJH`MHxOIxZJxbK`hLXqMXvNX{Ny?OIEOyP_WC?gE@GJ@_MA?PAORBG_CGaC_dDGiD_mE?rFO{Fo~G@@GhHHPKHhNI@WJP]KPdL@iMHsN@yNh~OA@OiFQCB?gF@?I@WK@gNA?RA_[Bg]C?`COhDWkDgoEGyF_}Fx?GHAGpIHXMHxPIXUJh_KhfLPlLxuNX{Nh}OICOqL_WE?wH@OJ@_L@oPAWSBW\\Bo^CGbD?iD_mDwqFOzFg~G@@GPDHPJHhOIHSIh[KHcKpjL`mMhyN`|Nq?OaDPsB?gE?wG@GK@gM@wOAGTBg]Bw_CObC_dCofF?yFW{Fg}FxBG`IHXKHhMIPRI`VJ@eKxgLHoNPzN`|NyAOYI__D?oF@OJ@_M@wQAWSAoVB?XBOgDGkDgmEGqFGyFg}G@@GPDGpFH@NI@XJ`\\Jp^LPjL`lMHqMhuMxwNqFPQX__D?wG@GJ@_L@oPAWSAoVBGZB_]COfDooEwwFG{Fg}Fx?GPBG`DH@LIHSJh`KPdKxgLhmM@sMhuN@xNiHQIT__E?wG@GI@_L@wOAWSAoWBOZB_\\CWeDgnEowFGzFg}Fx@GPBG`EGxMIHRJ`_KXcKphL`nM@rMhuMxxNaGQAU__D?wG@GJ@_NA?PAOSAgWBGYB__CgfDGjEguFGyFW{Fp@G`EGxGHHKHpUJ@ZJxdKxhLhnM@pMPsN@xOIBPqW__E?wG@GI@_MA?PAORAgVBGYBW^C_eD?iEgvFGyFW{Fh?GXDGxGHHKHhTIxYJpcKpgL`mM@pMPrMxxOAAPiV__D?oF@OJ@_L@oNA?PAgVB?ZB_aCWiDWkEWsFGzF_~G@@GXCGhEHHIHXVJ@XK@`KpfLPjMHqMXsMhuNyCPyS_gE@?H@OJ@_L@wPAORAgUAwYB_]C?`C_nEOrFGyF_|Fx@GPCGhFHHJHxQIXTJXaKpgLHiLXlLpnMXwNQEPaQ_gE@?H@OJ@_L@oOAOSAgUB?XBW\\Bw`CgoEGsFGyFW}Fx?GPBGpGHHII@QI`UJPbKxgLHiLXkLpnM`vNYDPYRsS?_?@_?@_?@_?@_?@_?@_?@_?@_?@_?@_?A?[??OC_?A@C??OH_?A@[??OI_?A?{??OD_?A?s??WCAWS_?B?wOAK??WHA?S_?B@?PA[??WI@gN_?B@WL@s??WD@wQ_?B?oMAS??_DAGQAwY_?C?oOAOWBK??_J@wRAo[_?C@OMA_UB[??_F@oNAwW_?H@OQA_WBO^_?G@WQAWVBG__?F@WLA_WB[??wI@gRAw[_?D@WPA_XB_`_?E@OOAWYBW`_?D@GLA?WB___?E@?LAGVBW^_?D@?NA_UBO]Ck??oH@oRAoXBgc_?G@GLAOTAoeC{?@GJ@oPAgZBod_?G@ONA?TB_\\Cc??gF@oOAgXCOh_?E?wNAGTBObDC??_H@_PB_bC_fDOn_?C@?KA?ZCOdCojEC??wG@_SBG\\BwfD?l_?F@GKAWYBo_CohDs??gE@_LAo`CObDgm_?D@OK@oVBo^C_kEC??oJ@_NB?\\C?dD_n_?I@WKAOTCGgDGiD[@?OB@_\\Bo^C?`CObC_dCofD?hDOjD_lDonEC@?OC@gUAwWBW[CObCofDOjDgmDwoEWsEovFC@?ODAWUAwXBO[Bo_CGaC_hD_lDooEGqEWvFC@?OEA_UB?XBOZBg^CGbCggD_lDonEGqE_uFC@?OJA?TB?XBW[Bg_CGaCghDOjD_oEGrE_tEs@?OIAGTAwYBW[Bo^CGbC_gDOjD_nEOrE_tE{@?OFAOTAwWBGYBw_CofD?hDOjDgmEGqEguE{@?OH@wTAoWBO[Bg]C?bC_dCofDGnEOrEguFC@?OG@oTAoVBGZBg]BwaC_dCofD?oEGsEgvFC@?WC@oNA?PAOTCObC_dCofD?hDOjEWsEguEwzFc@?WJ@wPAORA_YBo_CGdD?hDWkDonEGqEWtEoyGK@?WI@oOAORA_XBg^CGcD?hDOkDgoEGqE_tEwyGC@?WG@gNA?QA_WBg^C?dCofDWlDwoEGsEguF?zFs@?WH@gMAGQAWVBo^C?cCofDOmDwoEOrEgvF?{Fk@?WD@gMA?PA_ZBo^CGaCWdDWkDooEGrE_vF?~GC@?WE@gNA?PAW[Bg_CGaCWcDOkDgnEOrE_uF?~GK@?WF@gM@wRA_UBg]CObCofD?hDgmEGqEovF?|Fs@?_G@wPAORA_UAwXBO[C_fD?lDwqE_tEovF?xFhF_GC@GMA?QAWSAoWBGYBWdCohDooEGrEguEwwFG}HC@@OJ@gM@wRA_UAwWBW[Bg]D_nE?pEOrE_tF?xFxA_GD@OL@wOAOSAgUB?YB_`CWhDWlEGqEWsEovFH@Gs@?oJ@gMAGQAWTAoVBGZCGaD?iDopEOrE_uEwxG@D_GF@GL@oOAGSAgWBGZB_\\BwbCwiEOsEguEwwFGzG[@?wG@gNA?PAWTAwYBW[Bo_COeDWpEWtEovF?xF`C_GD?oM@wOAGQAgVB?XBO^C?cCgkEGqEWsEgwFGyHK@?_F@_OAGRA_XBOZB_`CObD?hDOjD_lDouEx?GHDGpKIK@?_J@_MAGQA_VB?XBW^CgfD?iDWkDonE?sEgzG@GHHLIC@?_I@_NA?QAWVB?YB__C_eDGiDWkDgnE?rEg{GHFHHMH{@?_D@_M@wOA_UB?XB_\\CObC_dCwhD_lE?sF?}FxBGpMIC@?_E@_M@wPAWUAwYBW]CObC_dCogD_mDwrF?|FxCGhLH{@@GJ@_LAORA_UB?XB_\\C?`CwhDOlDonE?qEo}GHAGxOIHU_GG@OK@gQAWSAoVBOZBo^CGeD?jDgmDwoEGvFh?GPGHxPIk@@?H@_OAGRA_XBOZB_\\Bo^C?`C_dD_nE?tF?yFxAHHLHs@?gF@_L@wPA_VB?YB_]Bw_CWfDWkDgmDwqF?|GHAG`JHs@?oF@_L@oOAWVB?XBW\\Bw_COeDOkDgmE?pF?}G@AGXIHk@?gG@_LA?PAOTAwXB_^C?`COcCwiDWlE?sEw{G@BHHJH`U_GE@GK@gOAGQAgWBOZBw_CGbCgeDOjDonEWuFX@G`HHPKIk@@?J@_L@wPA_TAoZB_\\Bo`CObCgfD?jDwsEozFxCGpJIHSJ[@@GI@_L@oOAWTAoZB_\\Bo`CObC_eDGiE?rEw{FxBGhIIHRJS@?gH@_MAGQA_TAoXBO]Bw`CWcCgfD?hDoqEwyFhDH@KI@SJ[@?oG@_NA?QAWTAoXBO\\C?`COcCgeD?hDgpEoyFpEGxKHxRJS@?wI@_M@wOA_TB?YBW\\Bo^CWdCogDGjD_pEgyFXEH@IHpVJc@?wJ@_M@wPAWTAwXB_\\Bo_COcCwgDGiD_qEgyF`DGxJHhWJk@?gJ@_L@oNAOTAoVB?]C?aCgeCwhDWmE?pEW{FpCH@IHxQJK@?oI@_L@oNAOTAoVB?\\BwbC_eCwgDOlDwqE_zFhBGxJI@QJKA?WC?gG@WQA_UBG`COdCwgDGjDgmE?pE_uEw}G@EH@KHxOIHSIpXJPZJh^KPbKkA?WC?oH@OQAWUBO`CWcCogDGiDgmDwqEWuEw|GHDGxKHxOIHRIhXJPZJ`]KPbKcA?WC?wI@WNA_WB_\\CWfD?hDOjD_lDwqE_tEozGHEGxJHpOIHUJ@XJX[Jh^K@`KcA?WC?wI@WMAWVBW]COeD?hDOjD_mE?pEWtEw{G@DH@IHhNIHTIxXJP[Jh]K@`KkA?WC?wG@GPA_YBW]BwbCgeCwgDWmDwtEovF?zFhCH@KHhMIHSIhVJX[Jx`KXcKkA?WC?wG@GOAWXB_\\C?aC_eCwhDOlE?tEovF?{FpBGxKHhMIHRIpWJP\\Jp_KPcKkA?WC?gG@WNAGVB_]C?aC_dCwjD_nE?rE_tF?{FxCHHJHhMHxQIxWJX\\Jx_KHaKhf_OB?_E@GI@oOB?ZBg^CWcCgeDOkDwoEWsEgwFW~GXHHPLHpOIPVJ@YJ`]K@`KXcKsA?WC?gE?wNA?WBO_CObCgeDGjD_lDopEWuF?}GHCGpIH`MHxRIxWJH[Jp^K@bKkA?WC?gE?wMAGVBG^CObC_fD?iD_lDoqE_vF?|G@BGhJH`LI@SIxWJH\\Jp^KHaKcA?WD?oI@WLAOVB?^C?`DOjD_lDonE?pEOrE`?GHAHHIHXNI@QIhUJH]Jx_KHaKXiL[A?WD@?H@OLA_UB_\\Bo^CGbC_fDgnE?qE_vF?|FxAGXJHpOIHQIXSIpZJh`KPbK`hLsA?WE@?H@WLAWUBW\\Bo_CGaCgeDonE?pEWuF?}FxAG`IHhNIHQIXSIhYJ`_KPbKhgL{A?WD?wH@WMA_WBG\\Bo^C?dCwhD_mE?pEOtF?yFpAH@IHhMI@QI`UJ@[Jh]Jx`Khf_OB?oF@?I@wRAwYBg]Bw_C_eD?kDgnEGqEgwFO|GPFHXLHpNIPRIhVJ`\\Jp^K@cKsA?WD@?H@OOAOTBO]Bw_CGcCgeDGjE?pEWtEwyF`GHHIH`MHxQIXTIpVJPZJpbKhgL{A?WE@?H@WPAOTBG\\Bw_CGcCgfD?iDwqE_tEoyFXFHHJH`LI@QI`TIpWJPZJxaK`hLsA?WD?wH@WLAGTB_]C?`CObCwhDOjDoqEWuEw{GHCGhIHXKIHRI`TIpWJHZKHaKhlMcA?WE?wG@OLA?TBW\\Bw`CObCogDOjDgpE_uEwzG@BGpIHXKIHRI`TIpVJHYK@bK`kM[A?WD?oI@WM@wTAo\\Bo`CObC_dD?hD_pEOrE_yFxDGpIHXNI@QIXSIxWJHYJX[JhiL[A?_D?wG@OOA_VB?XBOZB_^DWkDgoEGsEgvF?xG@AGXEH@HHpUIx\\Jp^K@`KXcKhfMPuMxx_OC?oF@GJAGRAwWBGYBW[C?iD_mDwqEWtEowFH@GPCGhFHHLIhWJ`]Jx_KHaK`dKppMhwNKA?_D@GI@WMAOTAoVB?XB_cCwhDOoEOrE_tEwxF`BGhFH@HI@QIpWJHYJX\\Jp`KPiLpqM`w_OC?oG@OJ@wQAgUAwWBOZCgeD?jDwpEWsEguFGzG`EGxGHHNIPTIxXJPZJ`^K@bLXnMHrM{A?_D?oG@GOAGTAoXBOZB_`CObC_dEWsEovF?xFxBG`DGpHH`RI`VJ@YJXaKXcKhgLHoMXs_OF@?H@OJAWSAgUBGYBW[Bg]CGgDGpEOtEovFGyGPDGpFH@PIXSIhUJPZJ`\\K`dL`lLpnMCA?gE?wG@WL@wTAoVB?XB_\\C?aCwlEGqE_uF?xFpAGXCGpFHXQIXSIpWJH\\Jx_KPfLHiLht_OD?oF@GI@gMAgUAwWBOZBo^CWeDopEOrEwwFG|GPBG`DH@IIPRI`TIxXJ`]KHbKpgLXkMsA?_D@GI@WKA_UB?YBW[Bo`CWdDGjD_mDwoEW~GHAG`DGpGHHIHpNI@PJX[KHbKhjLxsMpwOCA?_E@?I@WKAWUAwXBW[Bg`COcD?iD_lDwoE_~G@AGXDGpFHHJHhNI@PJP\\K@aK`iLprMhvOKA?_D?oG@GKAOUAwWBGYBw_C_dCofDgmDwoF?|FpAGXCGxGHHKHhMHxOIP]JxaKXgLHvN@xN{A?_D?wG@OK@wTAoVBO[BoaCWcCofD?hDWlEw{FhBG`DGpFH@JH`MHxPIXVJHZJhlM@qMXuN[A?_E?wH@WK@oTAoWBGZBgaCWdCofD?hDOmEozFpBG`DGpFH@IH`LI@PI`WJHYJ`kM@pM`tNSA?wG@GI@WK@gTAwWBW[Bg]Bw_CofDOjDwoEgzF`AGXCGxGHHIHXLHpPIPTIp_KHeKxmLxxNsA?gE?wG@WKAGTAwXBOZBo^C?`COdD?jD_mEGyG@AG`DGpGHHIHXKHhNI`TIx^KhfL@jLhpNcA?gE?wH@OKA?TB?XBO[Bg^C?`CWcDGiD_lEOyGHAGXDGpFHHIHXKHpOIXUJ@]K`eLHiL`qNkB?_G@GI@WLA?PAORA_ZB_`DOjDwoEWsEguEwxFW{Fx?GHHIHTIpYJX_KHaKXcKhmLxrM`xOA@OaDOqGPKB?_D?wH@OM@wOAGRA_YB_]CWcDGkEOrEgvF?xFO{Fg~GHDHpRIxWJX[Jh]KHcKheM@qMpwNi?OYCOqFPCB?_E?wG@WM@wOAGRA_XBW\\COdD?kEGsEguF?xFOzFo~G@EHhSIxWJP[Jh^K@cKhfM@pMhvNa@OQCOiFPKB?_D?oH@WL@oNAGQA_UB?bCgfDonEOrE_uF?xFW|Fo~GHCI@QI`WJHZJ`^KHaKXhLXsMhwNP~OYDOyGPKB?_D?oG@OL@oNA?QAWUAwaC_eDgoEGrE_vF?xF_|Fo~G@BHxQIXVJHYJh]K@aKXgLPrMpvNX~OQEOyGPKB?gE?wG@GLA?PAORA_XBO^C?`DgmEGqEovF?xFO|Fp?GHAH`RI`TIp]JxaKXcKhgLHkLhxN`|OQBOiEO{B?gF@?I@WL@oNAGQA_TAw]BwfD?jEGqE_tEwxFOzF_|G@GHXQI`TIpVJHZJh^KHfLXlLpqNX}OQBOaEPKB?oF@GI@WL@oNA?QAWTB?\\C?eDGiEGqEWtEoxFOzF_}GHFHPQIXTIpWJHYJ`]K@eLPkLxpNP}OQBOaDPCB?_D?wH@OK@gMA?QA_WBwbCofDGiDWlDooEwzF_|Fp?GHBH@IH`MI@PIpXJp`KXkMPsMpxNX|Np~OABPAK_WC?oF@?J@_L@wPAORAw_COeCwgDOjDgmDwuFW{Fg}G@@G`FHXKHhNIHTJH^K@aLhpMXtNHyN`}Ny@OQHP[B?_G@GI@WK@oNAORA_UBg]C_dCofD?hDwoEgyFW{Fg}FxFH@LHpNI@PIPYJX[JhmLxoMxwNPzNq?OIGPIN_WC?gE@?I@_NA?PAOSBO^CGbC_dD?jD_lDwsFOzFg~G@@GpHHXKHpNI@VJX^KXcLHjMPrMxzN`~OA@OqFQKB?_D?oH@WK@oOAGQAWXC?`COcCghDOkDooEWyF_}Fx?GHDHHIH`LHxOJ@YJpaKhgLPpM`wNP|Ny?OIDOyO_WD?wG@OJ@_L@wOAWSB_\\Bo_CGaDGjD_lE?pFO{Fo~G@@GPEHPJHpNIHRIp\\K@dKxiLhnMpzN`|Nq@OaEPkB?oF@GI@WK@gMAGRA_ZBg]Bw`CWgDOkDonEOyFW|Fx?GHAGhIHXLI@PI`TJ``K`eLXkLptNP{Nh}OACOiM_WD?oF@?H@_L@oNA?PAg\\Bo^C?aCWcCgeCwwFOzF_|Fo~GXCHPJH`LHpQIXSIxWKpfL@hM@yNX{Nh~OQBPSC?gE?wI@WK@oNAORA_UAwWBGYD?hD_lDopEOxFO|Fp?GHAGhEGxGHxOJH[Jh]JxiLXkLhpMPtMpvN@}OyIRKC?gF@?H@WK@gMAGRA_UAwXBW[BoaCwmE?vF?xF_|Fo~G@AGXCGhGHhPI`\\KHaKhfL@lLpoM`tMpwNH|PIPQkC?oF@?H@OK@gNA?RA_UB?YBW[BgbColDwuF?xFW|Fo~GHAGXCGpFHpPIX[K@bK`eLHkLxoMXtMpvNH{PAOQsC?gF@?H@WK@wOAGQA_TB?XBO[C?dCwhDWtEoxFOzF_}GHCGpFH@HH`MIpWJX^KhfLHlLxoMHqM`wNI@OYMRCC?oF@?H@OK@oOAGQAWTAwXBOZBwcCogDOtEwxFOzF_|G@BGhFH@HH`LIhVJP]K`eL@kLpoMHqMXvNI?OQLQ{C?gE?wI@WK@gM@wOAGTAwWBW[CObDOjD_rE_xFW{Fx?GHBG`DGpHHPJIxWJH_KHeKxiLXpMPrM`tMp~OaNQcD?oG@GI@WK@gNAGQAWTAoVBO[Bo_CGcDwqEWxFO{Fg~GHAG`DGxHHXNIPRIhZKPeL@hLPjLhmLxrN@yOqKQSD?oG@GI@WK@gMA?QA_TAoWBGZBg^CGdE?pE_xFOzFo~G@AGXEH@HHPOIPSIpYKXfL@hLPjL`mLxsMxzOiJQ^";
DIGRAPHS_NamedDigraphs.("loupekinessnark1") := ".U___gFecGdHcEdFcFdE`GHbILaJKbNaM`RSu___gFecGdHcEdFcFdE`GHbILaJKbNaM`RS";
DIGRAPHS_NamedDigraphs.("loupekinessnark2") := ".U___fEcdcdIeHfGcFdE`GHbILaJKbNaM`RSu___fEcdcdIeHfGcFdE`GHbILaJKbNaM`RS";
DIGRAPHS_NamedDigraphs.("m22bipartitedouble") := ".~?AY_C?_C?_C?_C?_C?_C?_C?_C?cGPAORA_TAoVB?XBOZB_\\Bo^CCPAORAcZBg^CCXB_]CCWBO]B{UBG\\BsVB?[CCTB?ZBsUBW[B{VBGYB{TBO\\CCTAoYBcTAwXB[UAwWBkTBOZB{UB_\\CCVB?XBsPAgUA{PBG[B{PBW\\BsPB?YCCPBo^CCPAoXB[PAgWBkPAwYBcSBO\\BsRBW[BsSB?ZCCRBG\\B{QB?[B{QBGYCCRBGZCCSB?\\B{QBO[BsSAwZBcRAwYBkQAgXBkQAoWB[SAoXBSRAgWBcSAgXBcQAwZBkRAoWBSSAg]CCRAo]B{QAw^CCSAoVB{RAgVCCQAgUBsPAOWBKPAW[BkPA_YB[PA_TB{PAWUCCPAOVBsRA_WBKQA_[BkQAWYB[QA_UCCRA_VBsQAWTB{G@WMA?aDwqF@@G`EHPJHxSJkK@gNA?aEGqFHAGxGHHJIHQJcH@OM@waE?qEx?GXDHHII@RJ[F@?I@waE?sF?zF`EH@KIHUJKD@?L@oaDwtFG{FpBH@MI@TJSF@GJA?aDwsEwyFhDGxLIHTJSD@WK@waEGtF?yFxCGhKI@VJCE@OLA?aEGrEwzFpBGpLHxVJCE@GK@oaE?rFG|FxCGxMHxUJKE?wG@_aE?tEoyFp@GPJHhRJ[D?wH@gaDwrEozFx?GPHH`SJkD?oI@WaEGsEo{Fh?GHIHpQJcA@?IA?gDGmEOsGHBH@LIXTJcB@?L@wfD?kEOtGPBGpKI`VJ[C@OL@ofDGlEOrG@EH@MIPUJkC@GKA?eDOkEOrGPCGhLIXVJkA@GJ@weDWlEOsG@CGxKI`UJcB@WK@oiDWmEOtGHDGxMIPTJ[@@oNA?kDgmGXCGhEGxGJ@XJS@@?J@wgDWmFO{HPJI@PJPZJc@@GIA?eDGlFW|HHIHxPJH[Jk@@_L@ofDOkFo~HHJHxOJ@ZJkA?wNA?eD?kEwwFOzHHJIXSJCC?oMA?hDOmEwxFg}HPJIPRJSB?gM@wfDWlF?xF_~HHIIPSJK@?gG@_dCwjGHAG`GH`MJH[Jk@?wI@WbCogG@@GhEH`LJ@ZJk@?oH@gcDGiG@AGXFHhMJPZJcB@?KA?dD_mF?xFO}HxPIhVJKC@GL@wcD_lEwxFW~I@PIpVJSA@OJ@obDgmEwwF_|HxOIhUJCC?gG@ObCwgEowFW}HPOIXVJkA?oG@gdCwhEoxFW{HXNI`UJcC?wJ@_dCojEowFg~HXPIPUJkB?wI@gcD?hEovF_}HHPIPTJ[A?gH@_cDOjEoxFO|HHOIXTJcB?oH@WbCoiEovFO~HPNI`VJ[A?gE@obCwiEWtG@@GXCIXSJCC?gF@wcD?jE_tG@AGhGIPRJSB?oFA?dCohEWsGHAGpFIPSJKC?oG@WbCgmE_tG`EHhMIpVJSA?wK@gcCgkEWtGxGH`LIhUJCB?gH@ObC_lEWsGXDH`MIhVJK@?gE?wbC_dFOzF_|Fo~J@XJSF@?H@obDGjD`RI`TIpXJPZJkD@WLA?dD?iDhQI`TIxWJP[JkE@OK@wcCofDpQIXUIxWJHZJcB?_G@GhDWnE?qE_uFHAHPMIKA?_J@ggDOnEGqEguEx?HXLICA?WI@_eCwoEGqEWuF@@HHKH{@?OF@ohDWnE?{Fh?GHFH@HH[@?_E@weCwoEGzFx?GPCGpIH[@?WDA?gDOnEGyFp@GPBGhHHS@?OG@GbD_nE?yFXBG`KHhNIC@?_I@_cDooEG|FpDH@LHpOIK@?WJ@gdDgnEG{FxEGxKHpNIKA?_DA?dDgnEGrE_wFGzFhCHCA?WE@wcDooEGsEgvFGyF`BG{B?_F@obD_nE?rEgvF?}FxDGs@?OB?_bC_dCofD?hDOjD_lDuY_C?_C?_C?_C?_C?_C?_C?_C?cGPAORA_TAoVB?XBOZB_\\Bo^CCPAORAcZBg^CCXB_]CCWBO]B{UBG\\BsVB?[CCTB?ZBsUBW[B{VBGYB{TBO\\CCTAoYBcTAwXB[UAwWBkTBOZB{UB_\\CCVB?XBsPAgUA{PBG[B{PBW\\BsPB?YCCPBo^CCPAoXB[PAgWBkPAwYBcSBO\\BsRBW[BsSB?ZCCRBG\\B{QB?[B{QBGYCCRBGZCCSB?\\B{QBO[BsSAwZBcRAwYBkQAgXBkQAoWB[SAoXBSRAgWBcSAgXBcQAwZBkRAoWBSSAg]CCRAo]B{QAw^CCSAoVB{RAgVCCQAgUBsPAOWBKPAW[BkPA_YB[PA_TB{PAWUCCPAOVBsRA_WBKQA_[BkQAWYB[QA_UCCRA_VBsQAWTB{G@WMA?aDwqF@@G`EHPJHxSJkK@gNA?aEGqFHAGxGHHJIHQJcH@OM@waE?qEx?GXDHHII@RJ[F@?I@waE?sF?zF`EH@KIHUJKD@?L@oaDwtFG{FpBH@MI@TJSF@GJA?aDwsEwyFhDGxLIHTJSD@WK@waEGtF?yFxCGhKI@VJCE@OLA?aEGrEwzFpBGpLHxVJCE@GK@oaE?rFG|FxCGxMHxUJKE?wG@_aE?tEoyFp@GPJHhRJ[D?wH@gaDwrEozFx?GPHH`SJkD?oI@WaEGsEo{Fh?GHIHpQJcA@?IA?gDGmEOsGHBH@LIXTJcB@?L@wfD?kEOtGPBGpKI`VJ[C@OL@ofDGlEOrG@EH@MIPUJkC@GKA?eDOkEOrGPCGhLIXVJkA@GJ@weDWlEOsG@CGxKI`UJcB@WK@oiDWmEOtGHDGxMIPTJ[@@oNA?kDgmGXCGhEGxGJ@XJS@@?J@wgDWmFO{HPJI@PJPZJc@@GIA?eDGlFW|HHIHxPJH[Jk@@_L@ofDOkFo~HHJHxOJ@ZJkA?wNA?eD?kEwwFOzHHJIXSJCC?oMA?hDOmEwxFg}HPJIPRJSB?gM@wfDWlF?xF_~HHIIPSJK@?gG@_dCwjGHAG`GH`MJH[Jk@?wI@WbCogG@@GhEH`LJ@ZJk@?oH@gcDGiG@AGXFHhMJPZJcB@?KA?dD_mF?xFO}HxPIhVJKC@GL@wcD_lEwxFW~I@PIpVJSA@OJ@obDgmEwwF_|HxOIhUJCC?gG@ObCwgEowFW}HPOIXVJkA?oG@gdCwhEoxFW{HXNI`UJcC?wJ@_dCojEowFg~HXPIPUJkB?wI@gcD?hEovF_}HHPIPTJ[A?gH@_cDOjEoxFO|HHOIXTJcB?oH@WbCoiEovFO~HPNI`VJ[A?gE@obCwiEWtG@@GXCIXSJCC?gF@wcD?jE_tG@AGhGIPRJSB?oFA?dCohEWsGHAGpFIPSJKC?oG@WbCgmE_tG`EHhMIpVJSA?wK@gcCgkEWtGxGH`LIhUJCB?gH@ObC_lEWsGXDH`MIhVJK@?gE?wbC_dFOzF_|Fo~J@XJSF@?H@obDGjD`RI`TIpXJPZJkD@WLA?dD?iDhQI`TIxWJP[JkE@OK@wcCofDpQIXUIxWJHZJcB?_G@GhDWnE?qE_uFHAHPMIKA?_J@ggDOnEGqEguEx?HXLICA?WI@_eCwoEGqEWuF@@HHKH{@?OF@ohDWnE?{Fh?GHFH@HH[@?_E@weCwoEGzFx?GPCGpIH[@?WDA?gDOnEGyFp@GPBGhHHS@?OG@GbD_nE?yFXBG`KHhNIC@?_I@_cDooEG|FpDH@LHpOIK@?WJ@gdDgnEG{FxEGxKHpNIKA?_DA?dDgnEGrE_wFGzFhCHCA?WE@wcDooEGsEgvFGyF`BG{B?_F@obD_nE?rEgvF?}FxDGs@?OB?_bC_dCofD?hDOjD_lDv";
DIGRAPHS_NamedDigraphs.("m22") := ".~?@L_GA?_GA?_GA?_GA?_GA?_GCAB`A@?_sN_OGJBGCB@?m@AOgJ_OKDBgCC@`A@?o[N_OOGBWCDA_u@@_cN_O_HBgCFA`A@@OWK_O[GBGGDA_{SDpo^__[IB`OXEpyA@_cLD@W\\FwGGAP?SE@g]__KCB@gZF@uAAOgKD`[WEWGB@P?TEPk^__OGBpSUFPyA?o[LDP[[FgGC@_wTE@g^__SEAp_XF@uA@o_JD`[YEwsMBp?SDPw^HASiIwkKBO{UDpo\\GQKeIWkKB`?WEPgZGAGfIGSHB?{PD@c\\IAcja?gKC@GSD`kgIQiFAOoMC`OVEaWfIgWIB?sPD@_[Ha[j`O[LC@KSEpodHqeEA?wNCpOYFQSeIGKIBp?REPo]GacjKWOHBp?RD`g^GQ_iKgOHBOwRE@s]GAWjKGKIBOwRDpk^Gq[iJwODB@?QDP_[GQGjLBaB@_oMC`SXFQ?bIrKu`?[KBPCTD`k_GqgsLWKGB?{PDP[YGQGiKr]D@owNCpSVEQGbHB?q`__LC@KTD`__GQOnKWODAowQEps^GqSgKbS{NgKEAp?QE`o^GQSfJr[zNWKGAosPEps]GAShKRWxNWOFAo{PE`o]GaSeKB_yNgSHAosPDp_^GQOfKBOtMg_IAowQDp_]GaOeJrKuMw[HAp?QD`c]GAOhKbOwNGWIAo{PD`c^GqOgKRKvMWSE@o_PC`OTHASlJbSuLraB@?cICPG]FqgjJQwrLBs}_oSGAPGRDps`IQomKblAPgODA?gPCp_ZGa_kJRCyOCUC@_[IC`KUFAKeJAwnNCDB_oWFAPCREPg_HqolKBc~PKu?_GA?_GA?_GA?_GA?_GA?_OGMCGCABO}@?_kK_OKCAwCHA_m@?oSM_OOECGCB@o}@@?_L_OSIBWCEAO}@A?cM_O[ICGCD@_q@@o_K__SIBpOVF@}A@ogMD@cZFgGEAOsSD`s^___HC@OWE`yA?oOKE`k[FWGHA_oUDp_X__KDC@SXEp}A@?_NDPW\\FgGB@osTDpo]__OEB`SWE`}A@OWJE@c[FWGFA?kUDpgZbOwNC@OTF`{cHQgjaooLBpWVF@s`GqWhaooMC@_XE`k_Ga[g`OcKBpCSEPsgIQmGA_oOC`OUEq_hIg[HB?wQD@[YHa[i`_gKBPCSE@oeHqmD@osOCpOZFASfIWWGB_{RD@g\\HQWg_ogNC@KXF@waIQkp`?cNC@KUE`{`IAgq`?cLB`KWFPw_Hako_ogLB`KVEp{bHqgn`?SKC@GTE@o`GaksMGKEB?wQDPc\\GAKjKrYC@ooLCPSUEq?bIbOt_o_KBpCTDpg`GagrLwSFB_{RDP[XGaKcKBIEA?sOCpSUEA?`HA{p`?SJB`GZFP{bHQ_qLRo}_oWJC@GYF@{`HQ[nLrk|_o_JBPCZFPw_HQcpLbc|`?[JBpCYF@waHQWoMBg}`OcJBPCVE@{`HA[oLBSya?gJB`GVE@waHAWnKrWz`ocJC@GUEPw_HAcqLB_{`_gJBpCUEP{bHA_pKr[x`OWFA@CQD@ScHQsmLRWvMGKCAOgPC`w^IaklJbKsNRyB@O_HC`KVFQChJAwqMsHE`?SGA`CRE@kaIAolKRh?PWOE@ogQCpW[GqWkJa{{OSMB@_[HCPKXEa?fJAsoMR|C";
DIGRAPHS_NamedDigraphs.("mcgee") := ".W___fEegfkCHdIcJdJcdO`EJbGPaFObIMaHN`KLw___fEegfkCHdIcJdJcdO`EJbGPaFObIMaHN`KL";
DIGRAPHS_NamedDigraphs.("meredith") := ".~?@E_GA?_GWDaoaFageJBWoMb?yKBgkLaouNCpQNCpQOCPIOCPITdh]Wf`uXE`uXE`uXE`uZF@yZF@yZF@yDAgSI`OiEAWWH`_edHa]dHa]gIQigIQi^GAEaGqQljimkkI}pKbYpKbYpKbYrLBUrLBUrLBUFAG[G`oa@NRw~_bs}NwOvMBeBMbk{_OGB@GCA?oRE_GA?_GWDaoaFageJBWoMb?yKBgkLaouNCpQNCpQOCPIOCPITdh]Wf`uXE`uXE`uXE`uZF@yZF@yZF@yDAgSI`OiEAWWH`_edHa]dHa]gIQigIQi^GAEaGqQljimkkI}pKbYpKbYpKbYrLBUrLBUrLBUFAG[G`oa@NRw~_bs}NwOvMBeBMbk{_OGB@GCA?oR";
DIGRAPHS_NamedDigraphs.("meringer") := ".]_____fEhGdddd`EIcHLMbGKMaHIbFIcGI`GJbHJaEJR`HKQU`FLRTaGLQcEKTXbELPScFJOPaFKNS}_____fEhGdddd`EIcHLMbGKMaHIbFIcGI`GJbHJaEJR`HKQU`FLRTaGLQcEKTXbELPScFJOPaFKNS";
DIGRAPHS_NamedDigraphs.("moebiuskantor") := ".O___dCcci@DG`EHaFHbDHbFGaEGo___dCcci@DG`EHaFHbDHbFGaEG";
DIGRAPHS_NamedDigraphs.("moth") := ".Eob_QLMob_QLN";
DIGRAPHS_NamedDigraphs.("nauru") := ".W___hCEdGdEcFcGdFfge`HNaIN`JObMPaLPbKOw___hCEdGdEcFcGdFfge`HNaIN`JObMPaLPbKO";
DIGRAPHS_NamedDigraphs.("nonane") := ".\\sPQRmNO_@cDUaBUeFVgHWkLTXiJSY|sPQRmNO_@cDUaBUeFVgHWkLTXiJSY";
DIGRAPHS_NamedDigraphs.("nonuniquelythreecolorablegraph1") := ".KcAL_DaOsrQMa?qDRCPAgxGCZ?JC`hec\\C@cIeGaDP";
DIGRAPHS_NamedDigraphs.("nonuniquelythreecolorablegraph2") := ".KgACwceS{aeNG`cPGSvI?`MHHdNGhbqGXCQDL";
DIGRAPHS_NamedDigraphs.("noperfectmatching") := ".O___dCfEhG`CD`CDIaEFaEFKbGHbGHMo___dCfEhG`CD`CDIaEFaEFKbGHbGHM";
DIGRAPHS_NamedDigraphs.("octane") := ".YqNOPkLMaB_@ScDSeFTiJRUgHQVyqNOPkLMaB_@ScDSeFTiJRUgHQV";
DIGRAPHS_NamedDigraphs.("pappus") := ".Q___eCdddccaEGbEHaFJbFI`HJ`GIq___eCdddccaEGbEHaFJbFI`HJ`GI";
DIGRAPHS_NamedDigraphs.("parachute") := ".FkYMQLGCbRzEbcbQ@Gs";
DIGRAPHS_NamedDigraphs.("parapluie") := ".FbAlHGsVwojQQLD";
DIGRAPHS_NamedDigraphs.("pasch") := ".Ik?GwAHC`XBJio?b_GcQDcKn";
DIGRAPHS_NamedDigraphs.("pentagonalhexecontahedral") := ".~?@[d?A@DGI@_wQ?_@YD`g]A`piRcgC[`@]DcXAAEGMRFweI_peCHxAHIWaDFWYPHWqL``yFGhIKKwmRJGiJMWyPLg_nbW}HKWozbRUJNw|Abg`BaamNQWw|cAYQGXOUDp_XdPg[FPyZFq?`GiKcHQWfiAciIqqlJa{oKZGrLBSulr_xMbm{NRw~R{@@OcLMpChJRCvDPc\\GQ\\qS?GCS_gEB`GA?DgUE`wIFEhMQ_PqCDwUPcGGW_xK^aWiBEWOfcGchaGS\\`hCdbGuEFg[acgoraxKkagkxbhCuaA}LbwcpbBmLLWk~bsIMaCMIIw|HbbuOHhG`d@WVE@eTE`o\\Fhk^GACagqOdHa]gIQgjJIsmJr?pkbKsLRYvMBcyMzo|Nb|NoCDAOszCQclKR[TEPs`H";
DIGRAPHS_NamedDigraphs.("pentagonalicositetrahedral") := ".ec?GGaD@`oG?iLD`PPkQAN`aw{YCKacwx]HEGOqs[vCGq@OkXHEbqCwaQHtAhWnWMfb|csZMXh?AAG`OW[A?IbPWSSZC_bwWmNE`BGhMMVaPaCKlFLpAK_SJEQPg{`MGca\\OiUJuBhw~XLErf";
DIGRAPHS_NamedDigraphs.("pentakisdodecahedral") := "._e?@?e?GDbG?cSK_@pWu?FDB`{CCB___`EABAcWGOKQ_`OpKhA@`qKGGIHOogkfDBAAkOKHIO_wg_UJoowk]UKP`?cWXLP@Gg[VLPP?kYWKwH_?OH_A@Wq?HDBG?[UL_@pOw^@@?wGGGP__ohEACBCgGSKRIO_W[bAAAaSKIJHpOo_jCBAQcGMIGDa{KMJFdbCWOHEERSOQIFDrSSOJEeBN";
DIGRAPHS_NamedDigraphs.("pentane") := ".PkEFGhIJ_@cDKMaBLMpkEFGhIJ_@cDKMaBLM";
DIGRAPHS_NamedDigraphs.("perkel") := ".x_OGCA@?daXOmRIdQX`AUNerh?{hFGCx?sdFFCX@MMSKoOwaIOHEwo{aZ`APPmAQKoQHiAKJoPXaELFgG_w_^Pp@hEEMHO`XMCKIPaXS{aQPQ`W{`Po`hgyaRO`pcw`QpA`\\DBHeCKW_XRIGg{sdRpPXT?mVp``W}lWOqp[ygaDR@pMraA@I@PBBaby\\jECAciyIFChdj^r?_OGCA@JCq`[eTIcrACk^LfQ@xQMOHq@hIMMGqA[[gX_`pCS_QLp`xCvACab[CcX_cRSCWV_arCKYNOP@p?}baBQKK[Q`Aq[GWSbCqhxCcadApxAb`BRPtCe`BbHpAdaDAyIERKGWp?qeSPPxhIfaaqi@[nbBAp|Yo`daxtPCIeBa\\fCCASAaEFDFszUKGDHTsSMHRJU~";
DIGRAPHS_NamedDigraphs.("petersencone") := ".J`ACWqHKhhccTC?aEOhbgRmCGPbGcqeeQPSOAGXAeM`N";
DIGRAPHS_NamedDigraphs.("pgammau34onnonisotropicpoints") := ".~?BO_C??K?_C?_C?_?B_?F_?E_?D_?Cb_QacRa[QasTakSasQBcRBsRB{QCCSCcUCKTCSSBkTC[UCkZD{XDsXD_pbWjECYDKVDSWC{YCosb?gEsVDgtbO[aw\\awcF[ZCkWB{XC[WBo}bGaF{YC?ybW`FkMB_yaGdFkOCW~`o_GSNC_{`w\\F[OCP@aG`G[LBw}`g]GCPDwoH[ODopHSMCovG{NDgxHCOD_qGsMDGsGcLD?wHkLCwuHcPDWrGkNDOtHK@@gM@wOAK@AORA_TApW`?LDWkFOzJCK@whDo|G@W`GOCwlGPBJCJ@oiDw}FxW`OPCogF`@JC@B_]CGbCc@Bg^C?aCk@EGrEguEx__GoEOsF?xJ{@FW|Fp@GS@FO{Fx?G[@HpPIPRI`b_HNI@TIpVKcAA_[CO}G[AAWbCg{GSAAo]C?zF{AAg^C_yFkAAO\\CH?GKA@gqEXRIxg_OPEwwHxPLKAA?uFHMI@i_OME?tIPTL[A@wpE`SIpf_oRBGeEg|JXn_wRBWlE_~Jho__QBWmEozJPmMSDA_XDwwFPYLhp_WTAwjEw}Jhl__TB?hE?{Jpn_gUB?iEHAJ`oMsBAoYD_xG@ZLsFA_YCwrGH]L`w_oQAwgEPBJ`kMkKB_nEo{HP\\KKJBgmF@AHXZKSHBwjE`@H@YKSIBokEhBGxYKKKCOfE?zG`[KSIC?iEO|Hh\\KSHC_hEW~H`ZKKJCGgEGyHH]KKGCgeFG}Gp]KSGCWlEx?Gh[KKCB_fFhHHxlNI?_w^DHBGpPLhuNsEBoiGHDI@mMx|_WbCozH`UM@pN{DBggFxCHpmNQ@_WdDgyHhQLxqOSDCGmFpFIxkMYB__aDx?H@RL`sOcEC?kF`JI`oN@z_wcDXAHPTLxtNcHB_mEhLHpfNkJCOhFHDIhfOSKCGiExEI`jN{HCgkF@HI@gOKGC?gE@IIxhOcGC_fEHJIXiO[ICWjEpCIHgOCIBgnE`KHxjNsJBoeEXGIPhNcKBwlEPFIpiN[CDWxFxFI`cOiO_gkEw|H@TKYHQKEDgrFpCHxbPiV_weEP?HHMKaMRCFDwpFXLI@bOqS_gfE_{GhQKaJQsBDGuGHJIxbPQT_omE?yH`PKaFQ[BDOwGXIIXcPAQ__gEhAGpUKYKP{BAWoEHFH@XRIY__SEOvHXKJI^SCDAOrFHIHhXRi`_oUE_uGpHJI[SSFAgtF@CGhXRY]`?RDGiHpNJPgQAPRY[`_SD?jI@QJXfQYTRi]`WQCwkIHUJhjQQSSAa`OUDgmIXTJphPyWRQ``GTConI`VJ`iQqVRI^`?UAw[H`QKhvOID`GRBO`HPPKhpNYJ`ORAwaHXOKpqNaKTkJA_WCXFHpdNH}PCIAOXC`DI`dMYCPsKAOWChGHxeNP|PQo`_TBW]HHRKhtOQF`GSBW_GpTKpsOYLU[GAgYBhLIpeMq?PIm`WUBG^G`VKpwNyEUKFAGWB_kIx^MQRRSB@oZC_gHx^NAUSSB@wXC?fHp_MiNRsC@gXCGlI@^MqQRYx_gOAw]DHUJxsRA^_gLBWaCpPK@vQi[VCF@oVCgmI`_NIPSAz_oNBObDxTJxyQA\\__PBO^DPQK@rQy`VsEA?WBgjIX_MISRIv`GPAwpG@CLYTSYsV{IA?YE?}HHfQQbUIz`GMB?qFXDLIOSamVCI@wWEWyGpiQIdTyxW[J@gVE_|HPiQYeUAyWKK@gYEg~HXhQafUQ{WSKAGXFHAH`gQqdUYv`?MBGuGXGKyVSqtVjF`?NBWwGHFLYWSylVsJA?ZEw{HhgPycUr?XKB@g[BhDGpkNX{OqFS[EAGaC`FHhlNy@PQJSsC@w]ChCHPoNqBPIMScFA?_CHGH`mOAAPAKS{D@o^CXHHXnNiCOiLSnO_C??K?_C?_C?_?B_?F_?E_?D_?Cb_QacRa[QasTakSasQBcRBsRB{QCCSCcUCKTCSSBkTC[UCkZD{XDsXD_pbWjECYDKVDSWC{YCosb?gEsVDgtbO[aw\\awcF[ZCkWB{XC[WBo}bGaF{YC?ybW`FkMB_yaGdFkOCW~`o_GSNC_{`w\\F[OCP@aG`G[LBw}`g]GCPDwoH[ODopHSMCovG{NDgxHCOD_qGsMDGsGcLD?wHkLCwuHcPDWrGkNDOtHK@@gM@wOAK@AORA_TApW`?LDWkFOzJCK@whDo|G@W`GOCwlGPBJCJ@oiDw}FxW`OPCogF`@JC@B_]CGbCc@Bg^C?aCk@EGrEguEx__GoEOsF?xJ{@FW|Fp@GS@FO{Fx?G[@HpPIPRI`b_HNI@TIpVKcAA_[CO}G[AAWbCg{GSAAo]C?zF{AAg^C_yFkAAO\\CH?GKA@gqEXRIxg_OPEwwHxPLKAA?uFHMI@i_OME?tIPTL[A@wpE`SIpf_oRBGeEg|JXn_wRBWlE_~Jho__QBWmEozJPmMSDA_XDwwFPYLhp_WTAwjEw}Jhl__TB?hE?{Jpn_gUB?iEHAJ`oMsBAoYD_xG@ZLsFA_YCwrGH]L`w_oQAwgEPBJ`kMkKB_nEo{HP\\KKJBgmF@AHXZKSHBwjE`@H@YKSIBokEhBGxYKKKCOfE?zG`[KSIC?iEO|Hh\\KSHC_hEW~H`ZKKJCGgEGyHH]KKGCgeFG}Gp]KSGCWlEx?Gh[KKCB_fFhHHxlNI?_w^DHBGpPLhuNsEBoiGHDI@mMx|_WbCozH`UM@pN{DBggFxCHpmNQ@_WdDgyHhQLxqOSDCGmFpFIxkMYB__aDx?H@RL`sOcEC?kF`JI`oN@z_wcDXAHPTLxtNcHB_mEhLHpfNkJCOhFHDIhfOSKCGiExEI`jN{HCgkF@HI@gOKGC?gE@IIxhOcGC_fEHJIXiO[ICWjEpCIHgOCIBgnE`KHxjNsJBoeEXGIPhNcKBwlEPFIpiN[CDWxFxFI`cOiO_gkEw|H@TKYHQKEDgrFpCHxbPiV_weEP?HHMKaMRCFDwpFXLI@bOqS_gfE_{GhQKaJQsBDGuGHJIxbPQT_omE?yH`PKaFQ[BDOwGXIIXcPAQ__gEhAGpUKYKP{BAWoEHFH@XRIY__SEOvHXKJI^SCDAOrFHIHhXRi`_oUE_uGpHJI[SSFAgtF@CGhXRY]`?RDGiHpNJPgQAPRY[`_SD?jI@QJXfQYTRi]`WQCwkIHUJhjQQSSAa`OUDgmIXTJphPyWRQ``GTConI`VJ`iQqVRI^`?UAw[H`QKhvOID`GRBO`HPPKhpNYJ`ORAwaHXOKpqNaKTkJA_WCXFHpdNH}PCIAOXC`DI`dMYCPsKAOWChGHxeNP|PQo`_TBW]HHRKhtOQF`GSBW_GpTKpsOYLU[GAgYBhLIpeMq?PIm`WUBG^G`VKpwNyEUKFAGWB_kIx^MQRRSB@oZC_gHx^NAUSSB@wXC?fHp_MiNRsC@gXCGlI@^MqQRYx_gOAw]DHUJxsRA^_gLBWaCpPK@vQi[VCF@oVCgmI`_NIPSAz_oNBObDxTJxyQA\\__PBO^DPQK@rQy`VsEA?WBgjIX_MISRIv`GPAwpG@CLYTSYsV{IA?YE?}HHfQQbUIz`GMB?qFXDLIOSamVCI@wWEWyGpiQIdTyxW[J@gVE_|HPiQYeUAyWKK@gYEg~HXhQafUQ{WSKAGXFHAH`gQqdUYv`?MBGuGXGKyVSqtVjF`?NBWwGHFLYWSylVsJA?ZEw{HhgPycUr?XKB@g[BhDGpkNX{OqFS[EAGaC`FHhlNy@PQJSsC@w]ChCHPoNqBPIMScFA?_CHGH`mOAAPAKS{D@o^CXHHXnNiCOiLSn";
DIGRAPHS_NamedDigraphs.("poussin") := ".NeIOzBJOkqDK]?WAEYGaF_SqTPI\\IuOAHId~eIOzBJOkqDK]?WAEYGaF_SqTPI\\IuOAHId~";
DIGRAPHS_NamedDigraphs.("propane") := ".JoGpxcJ?ICzoGpxcJ?IC~";
DIGRAPHS_NamedDigraphs.("reye") := ".W______l@AFGHIbEHIJKcDFGJK`BCFHK`BCGIJ`DEFHJ`DEGIKaBDFIJaBDGHKaCEFIKaCEGHJw______l@AFGHIbEHIJKcDFGJK`BCFHK`BCGIJ`DEFHJ`DEGIKaBDFIJaBDGHKaCEFIKaCEGHJ";
DIGRAPHS_NamedDigraphs.("reyeline") := ".[o?FGH_IKN_JLMcDFNbEFMdEILbCJK`BGIaCGLaEHK`DHJ`AMN{o?FGH_IKN_JLMcDFNbEFMdEILbCJK`BGIaCGLaEHK`DHJ`AMN";
DIGRAPHS_NamedDigraphs.("rhombicdodecahedral") := ".Mo?GaoAKc@BLC`RqIW{qENz??aJ?GqOCKsQDNGhbrGW~";
DIGRAPHS_NamedDigraphs.("rhombictriacontahedral") := "._i?@`?aR_ApxCd?EbaPMAABa@SSKGCagWOMHDOOg_WOa@`Gs\\A@`pW}AADbAKSKMGCwWOYNHWI_?WOGcw?k]PHO@gwcR___w_TDBA@GiECBaPSCIGECG_WQLFO_W[UN__`WobDBBa@MECEbqV";
DIGRAPHS_NamedDigraphs.("risingsun") := ".FkYM@LGGsVzEb_RQALD";
DIGRAPHS_NamedDigraphs.("robertson") := ".R____eD```bGaHcFaDFIcEHIaEGKcDGJbEFJbDHKr____eD```bGaHcFaDFIcEHIaEGKcDGJbEFJbDHK";
DIGRAPHS_NamedDigraphs.("robertsonwegner") := ".]_____hG`abcKdJLbEaFNdEcFPdFIcEI`HMNQ`GLOPcGJNRdHKOSaGIKQbHIJP`EFJKbFGMSaEHLR}_____hG`abcKdJLbEaFNdEcFPdFIcEI`HMNQ`GLOPcGJNRdHKOSaGIKQbHIJP`EFJKbFGMSaEHLR";
DIGRAPHS_NamedDigraphs.("rollingcube") := ".W____eDdddm@EIJ`EKLaHIJaHKLbFIKcGJLbFJLcGIKbCEH`AFGw____eDdddm@EIJ`EKLaHIJaHKLbFIKcGJLbFJLcGIKbCEH`AFG";
DIGRAPHS_NamedDigraphs.("roylegraph1") := ".G`KS@pCQ?PB_COqDpAWgBaGc?aF?G`cJ";
DIGRAPHS_NamedDigraphs.("roylegraph2") := ".GcAK?qEQ?PBJ?G`cJaOGoBGXG@CKk?aEOn";
DIGRAPHS_NamedDigraphs.("schlaeflidoublesixes") := ".in?Apkw\\_Dr@ctMHCq`UYNGCQkCCBEdg_[ONLpPGk_[bA``CzCA``p]AHDCbCKOKIEgO[URKyjo?kZMFW@[oXLRaPKgTebq@Cj@@?phYGFCBr[SQJGFGogWPMp?gW[V_aPPGpBCBA`iCFDcrN";
DIGRAPHS_NamedDigraphs.("schlaefli") := ".Z__@_@A_@AB_@AB_@ABD_@ACD_@ACEF_@BCDF_@BCEGH_@DEFGHI_ABCDFH_ABCEGIK_ADEFGJKL_BDEHIJKLM_CFGHIJKLMN`ABCDFHK`ABCEGILP`ADEFGJMPQ`BDEHIJNPQR`CFGHIJOPQRSaBDEKLMNPQRSaCFGKLMOPQRTUbCHIKLNOPQSTUVdFHJKMNOPRSTUVWeGIJLMNOQRSTUVWXz__@_@A_@AB_@AB_@ABD_@ACD_@ACEF_@BCDF_@BCEGH_@DEFGHI_ABCDFH_ABCEGIK_ADEFGJKL_BDEHIJKLM_CFGHIJKLMN`ABCDFHK`ABCEGILP`ADEFGJMPQ`BDEHIJNPQR`CFGHIJOPQRSaBDEKLMNPQRSaCFGKLMOPQRTUbCHIKLNOPQSTUVdFHJKMNOPRSTUVWeGIJLMNOQRSTUVWX";
DIGRAPHS_NamedDigraphs.("selfdualgraph1") := ".Fa@oBcsgsVw_[?xLILD";
DIGRAPHS_NamedDigraphs.("selfdualgraph2") := ".FkQ_WHGCbVzCgEAQ@Gt";
DIGRAPHS_NamedDigraphs.("selfdualgraph3") := ".FaGchLILD}GaQcsgsV";
DIGRAPHS_NamedDigraphs.("shrikhande") := ".O__@_@__AC_BC`C`BF`AFcEFGcDFHaDGIbDGJKaEHIKbEHJLMo__@_@__AC_BC`C`BF`AFcEFGcDFHaDGIbDGJKaEHIKbEHJLM";
DIGRAPHS_NamedDigraphs.("singleton") := ".@~";
DIGRAPHS_NamedDigraphs.("sixhundredcell") := ".~?@w_G?@_?CA_?CA_?CC_?CB@WA??_KF_?GC@oa?@?SFAW?B@_[G_?SE@ogJb_uLbO}LBpALBwsNC@ILBpCQbOwQCwsMC`OTbOwOCpULB`?PDwsMCPOUEGwTDgwTDpiMD`cYb`[WEwwWEPo\\bp?RbpGRFw{PDG{QDA?`bp?PFqEPD@c`c@CWGxGRDQAQD@Wac@KVFxOUEPocHxGTD`geHxKTDpkeIH?VE@sdIHCWEPwcHWwYEpo\\Fg{^GACaGxO`GaOfIXK^GAWgIxC`GqOdJX?^GqSgJHG_GaWfIhc[FaOhJXWYFA[hIhSYEqWiIx[ZFQ_jJH_\\FaSkJXs]Jbe[FawtMhgZJb]YFAwuMrqZFQwwMbq^Gq{rgQKnKb}_Ga{sgQGnKC@@fq?nKR|@hAcoLY[hKBXChAsqLSQfIbOuPYWiLB\\FhakpLsagIrCwQYSkKredJRGxPcmgJBKwQcm`HB?qOCHCPiGfKBP@OcTFgAWpLCDBQCebHRGrNs@JRH{gKRK~OshLfActLbk|PCUYIbWvNBtFQHkjLr_{NcdIfQowMRg}Qsu]JRSxMblERGCmMbk{NRyFJr|?OSHBaB@APCTMRwcsOS\\GRtAIKSLHQd@QarH?PcpMSWorNslLSTIBLRlCPdLV_bW|PS\\RTGOvNC`HTDUDMBxIRTTU`bcyQspUTwCA?rk|St`^WGCA@Bo|TD`_WUQ@@?S{NdTWWUHd_OSEMbxUUEHbXgCB@bgzTt`^WuPf`o_HOSHNUThZ`ocIOSLOUTl[YW[JBB|?STd\\Vg[GAs@ARddYVUdj`ogKNsLQUTp]YemBA?lCPcxYVT|k___HPS\\NUdl_YWOHAc`HSDl[WUiDA_pIRTH[VeHl`_kKQspPVTxbYwGBACPDSth^WEPmZwGCAS\\GTDl_WUTn[GODAcdITTp`WeXo[WKEAsXKTtt^Wu`m[gSEBClLTdxaWu\\p[na?_?E??OI??OI??OQ??OKD_G?A?o]??_OFAG?C@O[H_?KE@oa?@OWFA_mMBWuLBwsNCGsNbO{OCgsNCPILB`GRbOwQD@ULB`?RDWsMC@CVbOwPD@WWb`SUb`SVEgwUEPiMDp_Zb`_XF@uNC@MNC`K^bpCSbpGSGAENC@C^GXCSEQEOCP_bc`KTGHGSDaIOCp[^d@WXFAOfc`SUEaWfcpSVEqWgc@[WFQSgcP_XFaOdb`gZF@s]bp{_GQGbdACaHA[hcp{_Ha_jcQCbHASlc@{bHQ_kca?aHa[iePo]HAcld`g[HqcidPgZHagjdpk\\IAkke@s]HQolfPwmMXo]JbSye`kmLxg[JbWzNHk\\Jb_yNH{bJrM`Gq{qNy?aJrQ`Ga{oOCE^GA{pNsEcIR?thqcoLcQcJRGtPI[iLBXDhagsLs]eIrCvQI_jKR`HhQorMYSlKbdEQy_kKr`IQyCcKBH?OcPEga[oLCDAPS]_HbCsOSLGQYKdKbK~OClKfq_pKr|BQcu[IRSuMrtCPXgiLb[{NS\\GeqkvMBo}QSi\\JB_xMbxJRXwlLRcyMsXK_QwyMro|Ng[nNs@@OcMGKCHCPSxNaRP@Ps`NSGgpOsdISDIJKc@ERCxPbBK~QstPSgKtMsPESt]ALbtDPtLS`B[{QCdSTWSwNchLTTYEMRhJRDXV_OGBMrtRUD|__OGCNBtSUE@`XGCC@Ro}TT``WeU@@OWyNdXWWeLe_OKEMblVUD|bXE]FA?d@Oc|XUdmFAOh@Ot@XUtph`okKNs@PUTt]`o_JOCHMUTh\\YUmFA_o~OtHXVDxiYwKGAsPERdh\\VuqAA?dDPs|YUu@h`?cIQCdOUtp`YgSIBChLSdp]WeuEAopJRDD\\VeLj__KGPCTRUd|_XExn__OHPs`SUu@`XU|o`?SIQShTVEDaXf@p_oWJPcpVVT|bYExq`OWKQstUVeHbXvDq";
DIGRAPHS_NamedDigraphs.("smallestcyclicgroup") := ".HeASWBHG`RPESp{r@IK@ccOhgbIW~";
DIGRAPHS_NamedDigraphs.("smallrhombicosidodecahedral") := ".{_OGC?FGBqGweKHbAh[qLEFq@GuMIFQphAWMFsHxUIGa`zD]aVLSb@elTiFYXyiUnVXiA[^ngsm@CcnGSTtAbmHZYQsjkikylXeTJ\\M``aWwuCH]r_jeSDaq`KEZYGDLOohdp@AJEkGElY^GCA@?@q?{aMHbAWoiVKbP`{_QLba`skYOeB`|A]TaaGg]pVgdrTGoXjTY`ue]idjtuY_fFzyLJ_PHJqDD\\OgzaUuclIzIjMjUXdQvRgWWeML`AVkwIxd@WkgR@eua@RSKIX[OOapjA@jUf";
DIGRAPHS_NamedDigraphs.("smallrhombicuboctahedral") := ".W____BhDEFdEGeeJfHKgIJkjN`MOaLNcDFbDGR`ANOaCFQ`BGPw____BhDEFdEGeeJfHKgIJkjN`MOaLNcDFbDGR`ANOaCFQ`BGP";
DIGRAPHS_NamedDigraphs.("smalltriakisoctahedral") := ".Mo?GaOAKiGAEXAWcI\\AXCK\\AdrIWyTJXz??aH?Gqg_GXcHaOhsHcOpsIVKhbhSlf";
DIGRAPHS_NamedDigraphs.("snubcubical") := ".W____AB_@BfEeFiHigHgHIKfJfGJLMdJMbIJOaEK`EFbCHIP`DMORaCHKQ`AEQRw____AB_@BfEeFiHigHgHIKfJfGJLMdJMbIJOaEK`EFbCHIP`DMORaCHKQ`AEQR";
DIGRAPHS_NamedDigraphs.("snubdodecahedral") := ".{_OGC?CB_?O[[LFdAX_m[KEs@G}KFbBQK_^FCCypUiUidxxAoXjUydKg\\ebbdo~ZMGJ@iaQprQP|?boGSePLVOhc|[oaRiJiPNXLFcTCybQIjhxCiTqqpmUKLfDsgiUVQQ@ctHDDUBEEGFcw_WMG_a`[GWM`BPrYAJELUsKINGKwOOKsZoOgcSt}OGCA?A@o?GMMEbq`KoVMEBY?c^EBp`hEONbaA\\WjTJTQ{{`WKtj\\QeSMrPpqw^lfCD_tPHWxhG}_PwCIRGejgSq]mWPHtDtGfkebqIa\\PhDTs{aTIxXWvJEEraySTJJhH?qYcaaj@bBCBq[OKFCOPOmCKFO`gxl@DbEjYEDFcE[GGEYLwGSQIY~";
DIGRAPHS_NamedDigraphs.("soifer") := ".H`AKgaEQ?PoCOsqDK^KGPdCOqOAM?aEeOhb~";
DIGRAPHS_NamedDigraphs.("sousselier") := ".OdAbD`_F_`cEIcDHaFHJbGIK_@ABCodAbD`_F_`cEIcDHaFHJbGIK_@ABC";
DIGRAPHS_NamedDigraphs.("squarehexagon") := ".I`A_xsJ_HxFtAD@rgV?RqN";
DIGRAPHS_NamedDigraphs.("squarepentagon") := ".H`A_xsiYGko`O[yTLCV";
DIGRAPHS_NamedDigraphs.("sylvester") := ".c_OGCA@ICQXgqLF_`_{OOLGPPGo`@CAWGgV@FBwgkYTaA``UCJFcWWg[QKPP?}GHFEgOs[R`b@xMCEEDBSKMLIEw_WU\\a`pPp@ABaQ{KKGJGkcA@?_OHOaRLEPg{CKFaA@hAIHECGG_R@DAwGw^DDbQkOSKIo`W{bBDBaPaIGFp@GwtAEba[KWNHo_oogY``phOvCBArkSMIMGGO[QV```@XD";
DIGRAPHS_NamedDigraphs.("szekeressnark") := ".q_OGCSHCc@GwaUEcAwkULKFq`kcVNfRQcylVkUYXi_QkEyHQmYheYhYTImGziI{`ngKULPFbosymCeTOSieAcSOd[LSoxOGCAICaQ?c[PJBQ@[UJEeBxOuQJfrhhQ\\UjuJLKtOHUB\\ChVLSrLSlIdVC\\tD]OvsEJEgbpwY\\VARIgITR@QIGQmEiW^";
DIGRAPHS_NamedDigraphs.("tesseract") := ".O____hDEFG`ADE`BDFaCEGaBDG`CEFbCFGo____hDEFG`ADE`BDFaCEGaBDG`CEFbCFG";
DIGRAPHS_NamedDigraphs.("tetrakishexahedral") := ".Mk?Gg@EY?aE_K`sPCSsPdMiPAgTK`SjXyo?a_CXgAGY?qFPCPRPESyhCI`SqDQlf";
DIGRAPHS_NamedDigraphs.("thetazero") := ".FaqsBQQRwkl?scc";
DIGRAPHS_NamedDigraphs.("thomassengraph105") := ".~?@haoiIawoL_GCN`WOP_wGRaGcT`wWV_GEC_wID`gaFaXeYgII]eyE^fHunJZCojjOrlbUNGx?cdAqRIx_idqeUHhSdcaaPHyGugRUXJhgleq}_LHwrfRI^KXoohAukKiKmibEjKISshrMgJyWuiRVBQKPDqSjEP{DAlraxMjk{nsA|NklQUdvKRThZrc|ZVLLSVDzOSTt]_?CITTYC@OpUUGGBAtTVa?cMUDeE@otVU]eJAgiJb?u?_O}D`@EB_`MGaPUF``]?_WQB_gUEaG]HeXi_ghyZgX}[fY{lkRAmlBMuLW{bcAQSJHKjeAiVIXWedQUQIHCfgbY`LXcmeauZJy?sfbM\\Kh{pfBAcJYoqgqyiKYkohRQfKy_nhbYhL[LGpCVHQkXFoSIvMJcymrq~OJs}qtHYV[pLUdnMRtl[stP[Vl@PVTy??OhTTgODBDXW__KJTT]GAOxWUWWFBT\\X";
DIGRAPHS_NamedDigraphs.("thomassengraph20") := ".Sd?`D`ab_FaHbGccdFLgHMNeIMjKLP_@ABCsd?`D`ab_FaHbGccdFLgHMNeIMjKLP_@ABC";
DIGRAPHS_NamedDigraphs.("thomassengraph32") := "._c?GKIFCapg{?Z@DO`KKXCDpPCW^FFSAXOjPHDa{_ZMFehWotHDErc?IEBoOOKG]oQ?CEDBaPWs]?L_agOeEKaAwgaKNbbi@KgTgcap]OLfBrSkWYcabXq?DB@wGGECN^";
DIGRAPHS_NamedDigraphs.("thomassengraph34") := ".a_POSMH?`AGS?T@_ow[O\\CFpPcSZFbpacWjGDRaK{`KHrQScVSJtQt_y]NuRPkxa_POSMH?`AGS?T@_ow[O\\CFpPcSZFbpacWjGDRaK{`KHrQScVSJtQt_y]NuRPkx";
DIGRAPHS_NamedDigraphs.("thomassengraph41") := ".hd@gscVHDPx?yNGGRaK{`IEQplMdSipGkOvDLOA{Cn?JFwHW|AKorKGob`eCSWQ[Mpp@gv??__WOJhd@gscVHDPx?yNGGRaK{`IEQplMdSipGkOvDLOA{Cn?JFwHW|AKorKGob`eCSWQ[Mpp@gv??__WOJ";
DIGRAPHS_NamedDigraphs.("thomassengraph60") := ".{e?GKQJEbqHKiVA`o@kCXGHqQSO^DFPaK[`IIqqcGnBJXSe\\LhSUBHgv[Mgc\\w}_Or@iQcRQjXo|LUJht|OihgCTBE?@dAu[_Qr`?w_Tiub@ubZIeqIVM\\Ox\\^H_ABCapg{aRIdog[?Z@EQA[cdCFpPsWbFGQakkhAJoquTHfRYTD_qYLvBiHF]NgCKoYchCsiu[NRTay\\^SIiY@DOp_?XOlfGCkwOMGDYlgo\\guqhkadrfSMVV";
DIGRAPHS_NamedDigraphs.("thomassengraph94") := ".~?@]_GA?`_UD`WQEb?YFBG_IaOmJBGiJaWQSchKScPIRcXANEWwYbPmG`yGCfqE_GXWadQI]GHs^fHkheHgjeX[lkQsoiq}mKIonhiStiYgviI]cMiKxgXUUnBuzNZW~lRywMjCvnj}xlCaqPzLEkI~DRKdLqsrIR[DCoCbEotRFodY@RdI@RtMBSTYASDQBTwHTVLy?_GAE@WUD`GYK@g[Ka?iHAwkKagmH`HQQcpQPChMPcG{Xb`iLEwaFg_Q^GY?`daITGhw_fP}[eqeWeamXdqupJRAjJywojA}ehRUhib]ghyOygre`dXY{NZk|lb}tNj_ykR]}nzesQJHFksYoj{TKqSvJRKhLoSR?QKZBTK^ATgDMSgDNSwLPTgHOTGLV_dT[";
DIGRAPHS_NamedDigraphs.("threehexagons") := ".Q__cBbfEeiFHgHlCKdKo@NaNq__cBbfEeiFHgHlCKdKo@NaN";
DIGRAPHS_NamedDigraphs.("threepentagons") := ".N`A_xvLY}CsimKe~`A_xvLY}CsimKe~";
DIGRAPHS_NamedDigraphs.("threesquares") := ".K`A_xvLYyIDaJcGSFMxjVPOkP^";
DIGRAPHS_NamedDigraphs.("tietzes") := ".K`A?icmYHBPI]XBqIZcGODSdrPGYHRrG]PR^";
DIGRAPHS_NamedDigraphs.("triakisicosahedral") := "._i?@WocR_BPxCeS_Ba@CcSIoOOKKH____SSV__o_[OVKO_gWUKIDrCKMHEbqh[rCCA`p?kWKp`Go]RIDQxgvDDAq@GgUKEbc[OLFCQhWqZMWI_?UKHCw?s]PHdG?w_PHDAkCCBBAWGGGDDDwGKGFCDrCGIEDbA`[pBBaPg{iVKp@?g[OJEBKWQKFcq`SmYLpPOk_QIDb@gxFCBPpCiUKerf";
DIGRAPHS_NamedDigraphs.("triakistetrahedral") := ".Gg?Gg@EQ?`cJCOqDLa_?a_CXGAEOkPBGSv";
DIGRAPHS_NamedDigraphs.("triangle") := ".BcNcN";
DIGRAPHS_NamedDigraphs.("triangleheptagon") := ".I`?L@rmY`iftA?YBf[tBTN";
DIGRAPHS_NamedDigraphs.("trianglehexagon") := ".H`?L@rmOtRX`?L@rmOtR^";
DIGRAPHS_NamedDigraphs.("trianglepentagon") := ".G`?L@rhSuGOBO[yTL";
DIGRAPHS_NamedDigraphs.("trianglereplacedcoxeter") := ".~?@S_GA??gSC`oYHAGkIaGeIawqLbwyCCW[O`PMEChcWa?qHBWkNa_yPDH?Tc`]RDhaWGheXHHiZfXqdgiMcfai^Iy?lgQqehy_riRIeKi[riRUgLIgmiq}lKIoplj]xmJs}nB}yOZl?ncI~O{@DoSREpsjHqCrIQ{pL_S}@RdAB@?UA@_^S_GA??gSC`oYHAGkIaGeIawqLbwyCCW[O`PMEChcWa?qHBWkNa_yPDH?Tc`]RDhaWGheXHHiZfXqdgiMcfai^Iy?lgQqehy_riRIeKi[riRUgLIgmiq}lKIoplj]xmJs}nB}yOZl?ncI~O{@DoSREpsjHqCrIQ{pL_S}@RdAB@?UA@_^";
DIGRAPHS_NamedDigraphs.("truncatedcubical") := ".W___AdCcDdecfghhKjilNkMnOmP`R`QSbFIaGJw___AdCcDdecfghhKjilNkMnOmP`R`QSbFIaGJ";
DIGRAPHS_NamedDigraphs.("truncateddodecahedral") := ".{_OG?IICa@XGWNDCPxCqULaPpK_TKEqps}]ShsiLAjSkuJXus[jtjHqo\\lFzXzFaqxKI^?esyDe\\TeTyTe`Znv[E^DdrY[ekDn@ZMGX\\@AJG^cA@?@POcOJHB@wgaNGeQpkQMHcAh_uUMfrqdMdPgTYdepZMubd]lXMUBlg~ZNW{UVH`RwCufOkrikqnQksJ\\}z_rwkmZRct_lwJXpBJgGPXB";
DIGRAPHS_NamedDigraphs.("truncatedicosahedral") := ".{_OGC[GDBQ_sSTCCP@KSPECrX`A\\NjAxS_jHISq|GlPIsAdKrQKWbQAu`jtceLJaQVsLx@[miilRUltJLIvOkrxeUBB@XlWJHhBWlWR@hBVodrjx?_OGwOIEd@ggiGGaAWgaKHerACy^UDqh@UQShdyPYaTgDIXecWpEcDlBVjHKYVCcngZqAx\\TTYelZiUYTm`XfrKkEEArYoURQEpYoeBQEn`JfV";
DIGRAPHS_NamedDigraphs.("truncatedoctahedral") := ".W___dCcgFflHJgKgHIeMdLdEFbIJbIKaJP`KO`ACw___dCcgFflHJgKgHIeMdLdEFbIJbIKaJP`KO`AC";
DIGRAPHS_NamedDigraphs.("truncatedprism") := ".Q___AdCfEdEFefcIhiJ`L`KMbCDaHKq___AdCfEdEFefcIhiJ`L`KMbCDaHK";
DIGRAPHS_NamedDigraphs.("truncatedtetrahedral") := ".K`A?ichUHgfPK`hDNqCGAiQdWea\\CqEcS~";
DIGRAPHS_NamedDigraphs.("tutte12cage") := ".~?@}_GA?h?OD`_]GAWgOb?yLCWkNa?iGAwcKaOuICgsSapUKCxCTb`INDH?Rc@EMBxWVd`uVFx[[d`yYFXk^ePyWFHgZe@eYGH_`eqMXGhwbfAA\\GX{a`aAFGW[b`aICDWOQ`PQDCyOdhbewOz\\AntJ?S{DOndErNZOzlRiuNIlDjCQmPytEjsaqQjDJkCegRIdLhs}iRlPTtd^cX}TeuEBXW\\laueN[YLthvun]YgDkZWDmZwHo\\GLp\\WLr\\wHq\\ny?_GAc@?UE@w_Ha`AKBgsPao}GAg_JaOqHBWgQbPQJDWoRcPUMCg{Sc@MOCWwNd`]UFX[^dpqUFhg\\ep}XFh_[e`mWEXg_eAEZGxcafaM[GHs`fqIEGG[``qMEGgOT`@IDDGSRhAUeMZ`BlsI~Sk@RoTA}SZK|lBmtMjW{isUkPIxFjSYnQJHIkSmoQY`KiSufRyhMtDVUT}PfxUZWWLd`uuJYW|pgvUf^Y|xi_Upl_Uxn_f@s_vDt_vLv_fHu";
DIGRAPHS_NamedDigraphs.("tuttes") := ".m_OGCSHCc@xOeNJfahKiUMcbWw}OKecHxAWMgcyp]iWhDHkyq\\mFjXys\\cS[eLHERPSi]E^POsAECEPoOiUAAAZgCA@DAPH?]SHbqxwiRIdbhGuMNcBHhA]OeBiHMkVieIPQZMkfZ`yu]lFXDFHbQPcsTIf`fsSL?``@c[CId___f";
DIGRAPHS_NamedDigraphs.("twentyfourcell") := ".W__@_@_@__ABD_ACD_BCDi@HaHIbHIcHIdHfHJLMeHJKMgHKLM`ABEIJKO`ACFIJLN`BCGIKLPaDEFJMNObDEGKMOPcDFGLMNPw__@_@_@__ABD_ACD_BCDi@HaHIbHIcHIdHfHJLMeHJKMgHKLM`ABEIJKO`ACFIJLN`BCGIKLPaDEFJMNObDEGKMOPcDFGLMNP";
DIGRAPHS_NamedDigraphs.("twocubes") := ".O___dCcchDEGdFGeFGl@AK`BKaBKo___dCcchDEGdFGeFGl@AK`BKaBK";
DIGRAPHS_NamedDigraphs.("twodecagons") := ".S__cBbfEefgdHcImLlmnaO`Ps__cBbfEefgdHcImLlmnaO`P";
DIGRAPHS_NamedDigraphs.("twoheptagons") := ".M`A_xvLOtRzTDSd}`A_xvLOtRzTDSd~";
DIGRAPHS_NamedDigraphs.("twohexagons") := ".K`A_xvGYhl@RHNGOgM\\qEiZOSqR";
DIGRAPHS_NamedDigraphs.("twooctagons") := ".O__cBbfEecFdGkJj`KaLo__cBbfEecFdGkJj`KaL";
DIGRAPHS_NamedDigraphs.("twopentagons") := ".I`A_xsiYKcY`A_xsiYKc^";
DIGRAPHS_NamedDigraphs.("twosquares") := ".G`A_xsJCUGOgM\\ApD";
DIGRAPHS_NamedDigraphs.("twotrianglessquare") := ".I`?L@rmYpyDtA?YBf[tbsJ";
DIGRAPHS_NamedDigraphs.("u33") := ".c_O?K?B??o?KA@?_?_WWN?A@Oo]?AA@`C?EDBAG?KGFCO?OSMG_`_w_QI_`_w_UK_`_w_YM_`?g_QIGCWGGEGCaa@EAAA@pWo]P__og[UKFcWGGIEEb`xAABA@`gw]Oa@P?cULFcqhYGDCAP_w]RIDwOSKHDbQHGiVKoog[QJEcAPOkX`@?wcWMGCah[pBA@`Go[PHDApaCBCA`Ww]QIdrXuCBCA`_s]QIDbPqEDBa`Ww_RIDr@gy^``?ogUMGcqhWqZMFwOOMIEBQ@KiUKebhyCDBA`_saRIDr@kw]qOG?E?@_?W?E@?_O?OKKF_@?gWN?@@?oa?BA``C?ECBaG?GIFCOOo[OHDOOo[OJEOOo[OLFOO_SOHDCAKCCBCAPP?b@@@?wkWNGoOWSMJEBqKCCDBBPo{`@@`?os[NGP?g_QJEbqXSlCAa@Go[NHdA{GIECaphCcTJeWWSMHDbQ@GgUKo__[QKFCAPSmW``?ocWMGca`WpA@a@Ok[NHDQxkzA@a@OoYNHDApgxBA`pOk[OHdAx_s\\Noo_WSJFCQXSkXLfB{GGFDB@h?eTJERPs|AA``OoYPHdAx_u[N^";
DIGRAPHS_NamedDigraphs.("uniquelythreecolorable") := ".K`KSGPaUXRRGW{aFPG`R[`KSGPaUXRRGW{aFPG`R^";
DIGRAPHS_NamedDigraphs.("walther") := ".XgA`mLiJKhMgMeKLeJPcFKdFLbDObCI_IJxgA`mLiJKhMgMeKLeJPcFKdFLbDObCI_IJ";
DIGRAPHS_NamedDigraphs.("watkinssnark") := ".q_OGCSHCc@xgaOJfSaKo^JEcxccjIdCAScnTKta{s^QgCY@uMPcag{XDDPPcIQI_Pqpdo~\\NUb[QCiTotAeAj_hwY}CmW[gCA@DAPH?]YGcAxtGbKFqphMXHIqhP?dHJtRLWnLFsi@E_\\bcXGiNEPPSSXAcagC[kX[NvRtgvC`Id[LOh_iwI]En`JeF";
DIGRAPHS_NamedDigraphs.("wells") := ".__OGCA@JB@p?cT@BOp?qIHEP@OqEEa@ggYCF`aQPMIIGcw___R_aQ@UAIHDgG_aS``ppEIFFcG_[YQ`A@x[tADBbHqCHEeB]A@?_OGCkKFCAPSCLBCBGgcXCDBGWYGEa`gO]EHHCwggaRaAA@MAHGDWGgcU_aAHQEFFCWg[]Oa@phICGFdrSGSMKfGOcYWL~";
DIGRAPHS_NamedDigraphs.("wieneraraya") := ".ia?g[gNEcah?mDBcQWG}?M_cG@EIFa@gweKGaAwgiAB_?jxyZKfEbqA]ZNgYHK{`gCbyAm\\jFI`dKfieCq^TC@OxO]LHDQ@[IFGcoP{?\\@GOAKSNCBPpKWPCDpPSCF?@VrsuXMLFcC{u^OsQXxBOHFsD[zUMTBIXNTKHc~";
DIGRAPHS_NamedDigraphs.("wong") := ".]_____i@EdFIcGIbHIdEcEbEaEbFNdGOcHM`FOS`HNR`GMQcFLPRbGJPSdHKPQaFKMUaHJOVaGLNT}_____i@EdFIcGIbHIdEcEbEaEbFNdGOcHM`FOS`HNR`GMQcFLPRbGJPSdHKPQaFKMUaHJOVaGLNT";
DIGRAPHS_NamedDigraphs.("zamfirescugraph36") := ".c_OGC[LCapWcWPEDppK[TGEqP{g\\JESA\\CdTiTy|WpUKpWcKoZ`eRSCzAMoRaEC[OXGCA@FBPGkUHECP`[[RFDQ@kc^IFQpd?fPHTYd]nUKTbKUHBKEwXct@MobkCw``FCF";
DIGRAPHS_NamedDigraphs.("zamfirescugraph48") := ".oa?W[ADHCQpSwZKiAACcfNGrASg`MHRQ[kbCBP?kSNEBo@CCTBFoP[?RAFOpcGZOMGCLCyaPsryPNQNHStOjWLjJHhXUJjTvAG@`oGScPJDR`kqgGGQQ[{bKHQaCwdLHqqKOLCApO{WN?COPSK^@Do@KG\\BEO`l?w_OsRiHFRNhC|G{dRTAl_ukkeddWmlV^";
DIGRAPHS_NamedDigraphs.("zamfirescugraph75") := ".~?@J_GAIAGeE`xWMbxIRdHUOcWwVbpYQEXKWd@mTEh?\\cPqiIIeehy_jiQieJY[k`GUCKwSqa?mHAgWL`oqA?wCB_?KC@XWVE`mWEPo\\`_[MBwgJC`MGAPOTb?sOCXw^Ha]_GQolhASiIyGbIAemJrOtkBCqKwGuLrm@MBcyqwA?a_aH`g]UBg}QcxQTcHEMDw{Uc`eREHOZdPiOFXC[iaahhi]gIycihaufJGQD`BMDKg_JaOiEBW[K__M@?w?B@?UUDpgZe@c[FWWFB_}IApGRa?cSDWoLC@E]FqWfgACkJYOdIamaGq_hja{sLZ?pKbMALb[z_R_xMn";
DIGRAPHS_NamedDigraphs.("9-lace-1-14-13-13-13-1-13-13-13") := ".aa?g[E@@Ao?ckQIbagsMIIFPPWs\\ADp@_sbAEBWWkYNaA@pCfBBAgO[OM_CaX]AOIDgGKT?@A@scSLGeXGo]PJEgw{cRJEgoo_SKFBsOMJKEr`uIEIdrHgvaa?g[E@@Ao?ckQIbagsMIIFPPWs\\ADp@_sbAEBWWkYNaA@pCfBBAgO[OM_CaX]AOIDgGKT?@A@scSLGeXGo]PJEgw{cRJEgoo_SKFBsOMJKEr`uIEIdrHgv";
DIGRAPHS_NamedDigraphs.("kalbfleisch") := ".b_OGCA@?_OGKORDCowokXADBH?cYMba`Wo^CBaPcWOHEOP?k\\BBapsSMIEo`?gZAAAQ@KgT_`Ph?aQIO_Wg]PHCwGOUNHDQsSKKFcq`YEEFCAHSlA@`?wceW_`Oo_WSKfWGGGGDdR[KIEBbap[~@A@Owsc[`?oo_SPLG[[A@?_OGCA@@aAWgeFEDbGOgXGCbPs[SJEBw_[QKbA@GqAGDbgW[UMa`pOuCGDBWOOQOHdAkCILGCQPQCBDBqHGf@AApxGiUa``_{eSJOoow_PIdgOKGFCcrCCIECBA`cz@@@@?kiZ``Oo[[UJfwGOIFEcbcGEECAaHhB";
DIGRAPHS_NamedDigraphs.("pointgraphofsplitcayleyhexagonh(2)") := ".~??~_O?KA@?@o?ccPFebH\\C`NiCypUUReDhXU[SedXhMWSfDh@CnFGFh?{rIFexHGzIGfGxGtHGEGG[OHDOPWoYMRoPx?aQ_cq`Skh`DrX|DAKfCIQC\\NGCYqCWLHSqmGJHDs[KWNKGW_s_ZOOppCscbB`xxDEECRh}IJGESsSYQMHWW[eVOKG__gYOkWWciZPkg_gkWQJwo[iXQlGoce[RKwg_k]Nlgggg\\PL^{A?@_OG?M?CcaG{sXJgcH|OfUIqq[olJIracsjLHrAcwlGGdwx?|GFeXO{vHHFXPCxFHEhH?p@Ba@GiAJEBPq]ANGCQSCeSIddKGmZNggPcw`QObhx?bUOb@hIeTp@XGmb`b@x`BCEcBYAEMGeccW[NNGgooa\\NpPX?qeabQPpJBBcqy@`CCDBQDbBCdRYLdCDDbAP^EBdRIThECcraXfDCDbp|lDDDBiHj";
DIGRAPHS_NamedDigraphs.("changgraph1") := ".[__@_@A_@A_@BC_@AC_@BD_ABF_ABEG_CDFG_CDEHI_EFGHIJ`ACE`ACEL`BDFL`BDFMN`EFKLMNOaCGILMaBGHLNQaBGHMOQRcDIJLNQRcDIJMOQSTbDHJNORSTUfGIKLNPQRTfGIKMOPQSUWeHJKLNPRTVWeHJKMOPSUVXY{__@_@A_@A_@BC_@AC_@BD_ABF_ABEG_CDFG_CDEHI_EFGHIJ`ACE`ACEL`BDFL`BDFMN`EFKLMNOaCGILMaBGHLNQaBGHMOQRcDIJLNQRcDIJMOQSTbDHJNORSTUfGIKLNPQRTfGIKMOPQSUWeHJKLNPRTVWeHJKMOPSUVXY";
DIGRAPHS_NamedDigraphs.("changgraph2") := ".[__@_@A_@AB_@AB_@AC_@ADE_BC_BDG_EFGH_CEGHI_DFGHIJ`BCG`BDHL`EFILM`CEJLMN`DFKLMNOaBCGLaBDHMQaEFINQRaCEJOQRSaDFKPQRSTbGHILMNQRSeHIJMNORSTVfGIKLNPQSUVcGJKLOPQTUXdHJKMOPRTUWY{__@_@A_@AB_@AB_@AC_@ADE_BC_BDG_EFGH_CEGHI_DFGHIJ`BCG`BDHL`EFILM`CEJLMN`DFKLMNOaBCGLaBDHMQaEFINQRaCEJOQRSaDFKPQRSTbGHILMNQRSeHIJMNORSTVfGIKLNPQSUVcGJKLOPQTUXdHJKMOPRTUWY";
DIGRAPHS_NamedDigraphs.("changgraph3") := ".[__@_@_@AB_@ABC_@A_@BE_AEF_ADEG_BCFG_BCDHI_EFGHIJ`BEF`ACEL`BDFL`ACDMN`EFKLMNOaEGHLMbFGILNQaCGIMOQRaDGHNOQRSbCIJLMQRSbDHJLNQRTUfGIKNOPRSTeHJKLMPQUVcIJKMOPSUWXdHJKNOPTVWXY{__@_@_@AB_@ABC_@A_@BE_AEF_ADEG_BCFG_BCDHI_EFGHIJ`BEF`ACEL`BDFL`ACDMN`EFKLMNOaEGHLMbFGILNQaCGIMOQRaDGHNOQRSbCIJLMQRSbDHJLNQRTUfGIKNOPRSTeHJKLMPQUVcIJKMOPSUWXdHJKNOPTVWXY";
DIGRAPHS_NamedDigraphs.("bidiakiscube") := ".K`ADQTeYXhCPChwaHqCGTHUXhecPCQfaGf";
DIGRAPHS_NamedDigraphs.("counterexample") := ".KgI[aRIYGaF_KhcPCOhbgR?XAeM`ImSDMPHdLCPBoESqGaGSpsH_K`RFOdV";
DIGRAPHS_NamedDigraphs.("orthogonalitygraphofmagiccube") := ".LiQ`W@HGXBoCUGqhaGs@eVuhID_Cc`cN?PWbIeGbOEX^";
DIGRAPHS_NamedDigraphs.("gapbetweenschrijver,lovasz,szegedy") := ".GeI?gbaGc@dLaWgAaMGaOESv";
DIGRAPHS_NamedDigraphs.("bipartiteclassic") := ".FklQ@}qtGF";
DIGRAPHS_NamedDigraphs.("w(3)") := ".g_O?K?AA_OGC?H?A@w?Y?D_@PS?KH_`OweAECAwGOSK_`Owc^@A`pGyACDB@pEAECApkCGIEBQ[CKGDbqCGGDBBPo}EEBa`p?gU``@Gk]PHdgWSOKEcahYCFDbAHOiU`A@Gg_QHdbSKKFDBqHKqY``_wgYQIeB[KGHDbQPSmY``P?o[OIEBPuEDCB@xCeVLfgWOQJFCA`cu[`@pWo_QHdr@eCGCaaHOiVKEWO[UKEb`xt?`Po`?cSLFBr`w}a`@?gWaSIfSADCd`@?gW_QHfBp|EcsOG?E?@@OGCA?C_@?{?L?Ao?gi?ECoOg[R@BA@[CGIEOOg[QN_`Owc\\@AA`_wb@BA@WuACDB@hMAECApxACCA``gw^BB@pOw_SJOo_cUNGcqsKIGEBQPSlABap`CgTJO`?cSOHCqpiEEBa`xCeXLOoo[SLHDR@mECCaphGiVLOog_WMGDB@gzBAa@_{aRJersKGHDba@OqZMO_wkWOHCqx_rACAPPCgTJeBKGMJEBPo{y_OgwO_QIEb`xo{^PO__SKPIDRi@AaQo__SKOHCr`w}bQ^";
DIGRAPHS_NamedDigraphs.("q4(3)") := ".g_O?K?AA_OGC?L?AO?k?IH_@_{?GG_``OqACDAwGSUK_`pGo\\@CAPWuAFCA`{CGECbqSCIECBaCCGDBbQKGGDBBPo}EDBa@x?aU```?cYOHDgWOMHFCQPYCFCAQXOiU`A`Wo_PHDbSKOIDbPxOqY``pOo[NIeBSKQJEBPpKmY``?gkaRIEBXuECBAaPOiVLfgWSKKGCqhcu[`@?wgYOHdr`}CECb@xCgXMgCSGIGDbaPSo]Ogc[GIFEBQPOq]No__cUNGDR@p?d`@`?g[PHdriDIesOG?E?@@OGCA?E_@G?U?DCo?o]?CCOOogX@AA`[CIJEOOwcWM_a@GkZ@Ba@O}ACBAPxIADBA@pAACA`phECCA``gw^BA`p?{_PJOoo_QLGCasKGFCbaHGlABa@HKgTJO`OkWOGcapiEGDApg{gXLOowgWMFdR@iEHDb@gweVLOo_SUPHdB@kzBA@`PGgTJersKIEECAXSqZMO__[SLGCqxo~ABAP_{aSKfSAICDCAppGiWNGSQMCDBb@hGgXNFwOOQJFcAh_w_Qo_o_SMGcqxtAdR^";
DIGRAPHS_NamedDigraphs.("kapnigsbergbridges,vertexforeachbridgeandeachlandarea") := ".JmCPXcL?hg@CKez`CUXBOIY?PBH";
DIGRAPHS_NamedDigraphs.("kapnigsbergbridges,vertexforeachbridge") := ".FaGIQMQLGCbP^aGIQMQLGCbP^";
DIGRAPHS_NamedDigraphs.("kapnigsbergbridges,vertexateachbridgeend,landascliques") := ".McAGgOHEHhBpPLSikdQzhSlfh?aICAPaYO{SRTIjHSmyTJX";
DIGRAPHS_NamedDigraphs.("trellis") := ".M`ADQSlYXXGRChxfPCPNO`AhIUlKkcHaS{rgaGf";
DIGRAPHS_NamedDigraphs.("hamptoncourtmaze") := ".NmAHxGriqdsJLZ^v@C{cXtXQyDeln";
DIGRAPHS_NamedDigraphs.("hamptoncourtmaze,twoislands") := ".LiAHXEnbA[ieP^Y_aUPjwofIhcV";
DIGRAPHS_NamedDigraphs.("beinekenon-linegraphg7") := ".Eaq_bcbyJIAMQN";
DIGRAPHS_NamedDigraphs.("beinekenon-linegraphg8") := ".EgH_bcbRiAWGxGs";
DIGRAPHS_NamedDigraphs.("beinekenon-linegraphg5") := ".EaqcbcbRgkhGxGs";
DIGRAPHS_NamedDigraphs.("aolta(c)snon-linegraphh1") := ".FgHgwHHLD}_eb_ccsV";
DIGRAPHS_NamedDigraphs.("aolta(c)snon-linegraphh2") := ".FgX_gLHGsVyEWIBQQLD";
DIGRAPHS_NamedDigraphs.("aolta(c)snon-linegraphh3") := ".G`KS@pD?HAqEOhkOeI?wa_C`XBGSv";
DIGRAPHS_NamedDigraphs.("aolta(c)snon-linegraphj2") := ".G`EGiccOkqDpAKPTHG`XcJ";
DIGRAPHS_NamedDigraphs.("aolta(c)snon-linegraphj3") := ".FkgEQMQLNkgEQMQLN";
DIGRAPHS_NamedDigraphs.("thomassen24non-planarcubichypohamiltonian") := ".W___eCdEcdfehJgIhgcdbMPbNOaJP`IO`LNaKMw___eCdEcdfehJgIhgcdbMPbNOaJP`IO`LNaKM";
DIGRAPHS_NamedDigraphs.("treewiener(g)=wiener(l(l(g)))") := ".HkOk`d_CuZCJGXW@L";
DIGRAPHS_NamedDigraphs.("treewiener(g)=wiener(l(l(g)))") := ".IkKkiF_CiLUEUTBoATF";
DIGRAPHS_NamedDigraphs.("treewiener(g)=wiener(l(l(g)))") := ".JmKlBTL?GczmKlBTL?Gc~";
DIGRAPHS_NamedDigraphs.("treecontainingall5-edgesubtrees") := ".ImOtROAGYLVGYhg@CLF";
DIGRAPHS_NamedDigraphs.("tricorn") := ".I`A?icaQHXCLKh|O`?TQPHCkaEeS~";
DIGRAPHS_NamedDigraphs.("subgraphrelationsamongbeinekesnon-linegraphs") := ".HeA[XBGUGaDL?G`cIW}X_fEOqDaG`ROAGXAeN";
DIGRAPHS_NamedDigraphs.("rocket") := ".Fki_xP^ki_xP^";
DIGRAPHS_NamedDigraphs.("subgraphrelationsamongall4-vertexgraphs,1edgedelta") := ".Jk?cQsJGXcQE\\Ez?HClAqEXC`fP";
DIGRAPHS_NamedDigraphs.("complementofsubgraphrelationsamongall4-vertexgraphs") := ".JmQlbgk\\A\\vHUpsUM`N";
DIGRAPHS_NamedDigraphs.("tesseractinducedsubgraph,distance<=2fromonecorner") := ".Jk?GwAHC`XBJWyCzk?GwAHC`XBJWyC~";
DIGRAPHS_NamedDigraphs.("cubeplex") := ".K`ADQSeUXWeNGqGaHqCGTHQXXdaW{bGaGf";
DIGRAPHS_NamedDigraphs.("twinplex") := ".K`AD_qGYXRQJKaWcJqCGUBG`hdLGkqHaOn";
DIGRAPHS_NamedDigraphs.("ramseycomponent") := ".K`A_yPJG[@tMaPcPGdVGOgMcQqF?\\RgcXCQHT";
DIGRAPHS_NamedDigraphs.("3triangles,connectedtofirst") := ".H`KTQQFOk?eNeCpTHG\\AoAW~";
DIGRAPHS_NamedDigraphs.("gosperisland2") := ".~?@}b?YFaGeIax_RchQTdxYcFiKcgaQ_gQWffqSfk@ybgX}_ghwpgrM^LI?qgRUaKIijjYykjzr?nj~@nZ@?kRqtNjG~krusOYguir]kMysyjramM[NApKVEp{`MqS~ISKlPrTNKSlPYtTnUVL\\\\uT~WVh`_eUEYWhlbfUU[XG?iXgCjXwGkYGKlYWSnYwOmYgoQTWsRTGwSTg{TTxCVUX?UUG?EOgGFOwKGPGOHPWCJPwSIPnyK@g]GaWiJe@MQdHUVdiO]gqQaHIA`Ha]^HQ]oFiM`fyAafbEbKx{sgBI`LYGoiimljiqnnKA}n{E|kCApNJS}kb}rNZP@ibYjLyozjRinMIwxo{JCp[ZFqCzHR{hOqtFLS{pQtDjTU|X[ttvXV|`]eEAXWXhaeuM\\XXpc_Ahe_Qlf_apg_qth`Q|j`Axib@HTbPLSb`PUbpTVcP\\Xc@XW_?XA__\\B_o`C`?dD_OlF`OhE";
DIGRAPHS_NamedDigraphs.("triangle-replacedpetersen,ct66") := ".]___AdCfEiHgcGdHfIeJgKhLjNiMorSpqUsTuV`X`WYbCDaEF}___AdCfEiHgcGdHfIeJgKhLjNiMorSpqUsTuV`X`WYbCDaEF";
DIGRAPHS_NamedDigraphs.("smallestpseudosimilar") := ".G`O[PQJK`kOgMGhDeOv";
DIGRAPHS_NamedDigraphs.("vertexremovalofsmallestpseudosimilar") := ".FaqtJP^aqtJP^";
DIGRAPHS_NamedDigraphs.("pseudosimilarfromthreek4-e") := ".GeICWcJGXRWeICWcJGXR^";
DIGRAPHS_NamedDigraphs.("fourcolouraprilfoolshoax") := ".~?@m_oAB`_UEaOUH`?iLBGuOA?}@__y@CW?Ba?{Oa@QB@@QSD`]CAp]VE@e\\Epq_F`}aGYCacqCacYUPCqOdHiIiIYGbII_jiAgkjocIJioljBAjJBED@jSskJ?pLjCqgqkqLZGsLR_x`_[rkrOzmJk{N[?~b`G|bsAKBpTAa@?UNw{ONs@APGsMLsEKLcHBlc@APzCvMBt@PgoLLb\\EPzOwMbo|NiwrLBqmLBTK`OcmJrLKiR|?iasoLcahIc@GRtAeIAglhaciSiGcHa[gSi?`fqSeSx{hStY]LSu]GrSxUHw_GQLTUX{_HTTUfPw^GHk~RxomRTa[Ja|]eqdNTtuZFP|VVEA[FPxWVDyYEpo\\a_kXJt}UEB|CVXcYFD|bXH_YEttbXWCRGQPT_PCdTTlg_`G|NcEA@rk}YgGFB?sMC`TBYw?B@?SE@o_HA_kSDPXkzgK?_wWD`gcDaWOIbOqLc?_N_WGM_PE??w_NCG_S_oOSd@WV`?kVdp_XfPk[g@w^gaE`GhK`GhEdcPKcHQYaiaeaGqagIy_iJI{HAaykJYooiqop`OYtLJAoKRYpKiKjKbUqLBSwMWWFKzKsMzazNBv?NwwQNW|?b?{TOg_ODb}NCB|?OcQLBb\\@bBXAOzX?Oc]pLr_|OSYKBRWvPc]sMBg{NRymKrO{jbOtRGSHJa{rRIc~OIglKBXGiQh?QC|Oha_iJYWhIdIaHAWfIDI_GX{dHdM^ITLUfbTLfaKtMTa]GACbTTe^GATTThs]FqAZNs}[JctWfAwnVhkhRt\\\\eps^Ttp_f@s]UDp]e`k[FWgJEQ|^d`_~PDuXE`p^WuQWE`l\\WuU@CqCcTWCPHTTZYGGQNRx@__[zNeiA@ooLB`GTOum??oOD@_[GAOgJD@SUZN";
DIGRAPHS_NamedDigraphs.("linegraphn=8pseudosimilarmostedges") := ".GcES@pCM?QD_GXAepGKgBaG[?cJ?OqDL";
DIGRAPHS_NamedDigraphs.("linegraphn=8pseudosimilarsecondmostedges") := ".GeE?gacKk@cLaWWAaIOqoEOv";
DIGRAPHS_NamedDigraphs.("linegraphn=8pseudosimilarsecondfewestedges") := ".GeAGxccUGbWeAGxccUGb^";
DIGRAPHS_NamedDigraphs.("quartettree2") := ".YiGhdc_`abeftQRSjOPiMNgLShKSyiGhdc_`abeftQRSjOPiMNgLShKS";
DIGRAPHS_NamedDigraphs.("quartettree3") := ".~?@}haUckAahiimkjY}mnBczmBi?_gEC_wYDaG]IaWqJbguObxEQcxQTdh]WeXiZfHu]fyBcKTDQkC|OktTUkdLSlTdYlD\\Wltt]ldl[jCp`jSl`iC`biS\\bichaisdajsx^jct_hsX^hcT_msPbmcLagRx`gsH^gcD_hRd?hB_~gQGb~iWdhJ?giYijjIunjjoxMz_y_GI@`GME`WaFageKawyLcG}PchMSdXYVeHeYexq\\fh}_xBDPSj@NSJLTTjHRTJTXUjPVUJ\\\\VjXZVIpKWYtJWY`GWydFWyhIWilHWi|MVyxLWI\\EVyXDWJlCWzhBWiC}WYLAVyH@WISxOIOwNyCaG~";
DIGRAPHS_NamedDigraphs.("as4s4molecular") := ".Gg?K@PEUOqWg?K@PEUOq^";
DIGRAPHS_NamedDigraphs.("self-complementarygraphwithintegralspectrum") := ".H`?KGBaIGhcIYXAerA?WOFCSPRGSsqDL";
DIGRAPHS_NamedDigraphs.("(3,3)-graphwith20vertices") := ".S___eDccfedgIhJ`GJ`HIbDFaCEbKaLQs___eDccfedgIhJ`GJ`HIbDFaCEbKaLQ";
DIGRAPHS_NamedDigraphs.("circulant17steps1,2,4,8") := ".P__@_@_@_B_AD_BCE_ACD`ADE`CEFH`BCDH`ABGIcDEGIJKbDFGHIKaBEFJKLaCFGHJMNp__@_@_@_B_AD_BCE_ACD`ADE`CEFH`BCDH`ABGIcDEGIJKbDFGHIKaBEFJKLaCFGHJMN";
DIGRAPHS_NamedDigraphs.("alternateterdragon2") := ".GeIhGAFC`RWeIhGAFC`R^";
DIGRAPHS_NamedDigraphs.("alternateterdragon3") := ".QbAdCiFegIhJK`FI_EJaBGIKMcDHJLNqbAdCiFegIhJK`FI_EJaBGIKMcDHJLN";
DIGRAPHS_NamedDigraphs.("sevensegmentleddigits,immediatesubsets,7withoutleft,9withlow") := ".IiEcPB_KsaDPigYPCM?rQGTF";
DIGRAPHS_NamedDigraphs.("distancepalindromictree21") := ".TkInJLMdEkOhMPfGL_@ABCJtkInJLMdEkOhMPfGL_@ABCJ";
DIGRAPHS_NamedDigraphs.("doubleladderofdesa!etal") := ".VlGHKgIJeFHdFHcEKbCK_BIaDJ`AJ_@IvlGHKgIJeFHdFHcEKbCK_BIaDJ`AJ_@I";
DIGRAPHS_NamedDigraphs.("subtreerelationsofasymmetrictreesn<=10") := ".JkEsBQGSyHcI\\Ez@l?saDMaXAfP";
DIGRAPHS_NamedDigraphs.("subtreerelationsofasymmetrictreesn<=11") := ".YpBNaINOpphJKLRScFIJOS`DEHMRgKLMNORS_DEFGMRSypBNaINOpphJKLRScFIJOS`DEHMRgKLMNORS_DEFGMRS";
DIGRAPHS_NamedDigraphs.("windmilltreeofdesa!etal") := ".TkGhij_@KaBLeFMcDNkLMNtkGhij_@KaBLeFMcDNkLMN";
DIGRAPHS_NamedDigraphs.("threeplugtreeofdesa!etal") := "._g__[?BCBpOs_RKEq`[wjNJSA|Kh\\HERdCs[kEreAaA@o?KONDBQ@KoZIDrak{lOJsqdscXMSRPqoZM^";
DIGRAPHS_NamedDigraphs.("mostminimaldominatingsets8,connected") := ".G`AK`qF?`WcIZCGPcMOwCJC`R^";
DIGRAPHS_NamedDigraphs.("mostminimaldominatingsets9,connected,18edges") := ".H`A_wa_K`WaEQG`cJeCIBaI?qDaGXGaEOn";
DIGRAPHS_NamedDigraphs.("mostminimaldominatingsets9,connected,19edges") := ".H`AK?qFG[@cIYGaDLeCGoBG[`oEOhgaGSv";
DIGRAPHS_NamedDigraphs.("mostminimaldominatingsets9,connected,20edges") := ".H`AK@qEQOqPGSsPDK^KGP_MOqQEQHAeaGhb~";
DIGRAPHS_NamedDigraphs.("mostminimaldominatingsets9") := ".H`AGichStRFrACPTHQhieN";
DIGRAPHS_NamedDigraphs.("mostminimaldominatingsets10,connected") := ".IcAKWQJGhhcLK`sYcAKWQJGhhcLK`s^";
DIGRAPHS_NamedDigraphs.("mostminimaldominatingsets10") := ".I`AGicgU`YENSpsY`AGicgU`YENSps^";
DIGRAPHS_NamedDigraphs.("4-cycleswithvertexbetween") := ".HcAKWVCSsqErGCWomGhhcL";
DIGRAPHS_NamedDigraphs.("mapofsumofcubes5") := ".{caHCcTPGSqTWjSm?q@a?PKoAPeARLOQ`iCVLoqpaCTLp@kS[cabgo|LEFp@k[WhbbDC_QKEiTS_SLFHCkgUMFhc{cUKFiD\\?acQsaYXNSJiD\\SkhTPA@_wcWPQHcydWPQPc{eWpaXg}fWpa`h?gXPqhlEiXpAp`ChXpqxlAjX^HGaPHDSQDKdUItJ_K_W_CRK?cX_crSCgY`Dr[KkW`DR[OZDFHGgyKNRP`{OZFEIWwpPGCb@idTGDBPqPJIDb`yXNHDb@y`VOGhClGeeRtAy`VTJITSO_WMHECSaXMhUCScXNHeKWeYNhuKWgYOIES[iZPie[OkWPIU[[mZOiuV";
DIGRAPHS_NamedDigraphs.("mostminimaldominatingsets,11,connected,19edges") := ".J`A_zDeOhhcIYGapCXMwOgMpXcIYXAeaGkPER";
DIGRAPHS_NamedDigraphs.("mostminimaldominatingsets,11,connected,21edges") := ".J`AgqRGUXHcNCPSPCWy\\o`SXHcJKcqFaGiGaK\\N";
DIGRAPHS_NamedDigraphs.("mostminimaldominatingsets11") := ".J`AGgaFWlRTLSyJFOf[GPDCOzDiYhifPWyC~";
DIGRAPHS_NamedDigraphs.("nodisjointminimumdominatingsetsthoughnovertexcommontoall,19") := ".Rh?a`bEdFcGoEHNfJNgINrh?a`bEdFcGoEHNfJNgIN";
DIGRAPHS_NamedDigraphs.("nodisjointminimumdominatingsetsthoughnovertexcommontoall,16") := ".OgCb_DaE`FdeHLfGLogCb_DaE`FdeHLfGL";
DIGRAPHS_NamedDigraphs.("subdividedbi-stars4,5and4,4connectedat4-starmiddles") := "._eah[qaFcQWCEDBaPWtoYMFcI`SkVLuBHgvOGcaXz@LDQxdCNGcoGKIFCapj_s[NGTAhWmZkERPm_PHCrv";
DIGRAPHS_NamedDigraphs.("twosubdividedbi-star4,5connectedat4-starmiddles") := ".ag?GKIFCapg{aRIdrHkwgTJDxx?aQHrbXozNKERQBDO?OWSMHDbPxCeTJeRXpOiUJrq@CcRfEr`u]WKecF";
DIGRAPHS_NamedDigraphs.("twosubdividedbi-star4,5connectedat5-starmiddles") := ".ag?GKIFCapg{aRIdrHkwu[Mrr@ctMIDQp]]OGcaYBDO?OWSMHDbPxCeTJeRXpkw\\feBHi[SIda{{_PHCsF";
DIGRAPHS_NamedDigraphs.("3-regularmatchnumber(4n-1)/9,n=34") := ".a_OGCSHFBQPCkTCAp?kgVIDp_{_RGCp_{oZOHsQSw^XKUBL_rSJtQsKsZ`fBkCD@@GKSA@?a`GwYQGdag_UGDdAxOmKFcAX?eKFeBY@MaQfBzHaoXkEY`]iU`eb[Kw\\__gGH@";
DIGRAPHS_NamedDigraphs.("3-regularmatchnumber(4n-1)/9,n=88") := ".~?@W_GA??g[EaOaJAgsKboyPCHKQdPQE@wWFb?uKBW_Ha_mIAw_Hb_}MBxOTd@UOCXGRc`MOCXWVe`uZFH_Xf`}aHYKcgAEnJjCokrItLJ[umRamJz?pkBEmJzGrlBUsLZGrlb]uLz_xmBeeIY[giaujJJg|mrq}OZ|?pCVAO{`Hpc]DRCuCQcmCRdADRtE@@?U@?_NW_GA??g[EaOaJAgsKboyPCHKQdPQE@wWFb?uKBW_Ha_mIAw_Hb_}MBxOTd@UOCXGRc`MOCXWVe`uZFH_Xf`}aHYKcgAEnJjCokrItLJ[umRamJz?pkBEmJzGrlBUsLZGrlb]uLz_xmBeeIY[giaujJJg|mrq}OZ|?pCVAO{`Hpc]DRCuCQcmCRdADRtE@@?U@?_N";
DIGRAPHS_NamedDigraphs.("planar4-connectednonpancyclic") := ".]bAdCcDdeccHfIgHdFeGlMhKiJfJgKoPbPSaOSaRUbQT_LN`MN`RVY_QWX_@AB}bAdCcDdeccHfIgHdFeGlMhKiJfJgKoPbPSaOSaRUbQT_LN`MN`RVY_QWX_@AB";
DIGRAPHS_NamedDigraphs.("bridgelesswhosesquareisnonhamiltonian") := ".^_bAdCfEhGjIlKnMpOsKLmNoP_@QaBRcDReFRgHQiJQsTUVZ[sTUWXY~_bAdCfEhGjIlKnMpOsKLmNoP_@QaBRcDReFRgHQiJQsTUVZ[sTUWXY";
DIGRAPHS_NamedDigraphs.("doyle") := ".Z__cBb`aefhgIlKklcNdM_BiPQjOQ`KRaLShNTgMUdFWcEVXz__cBb`aefhgIlKklcNdM_BiPQjOQ`KRaLShNTgMUdFWcEVX";
DIGRAPHS_NamedDigraphs.("levy") := ".]i?fhJgJ`KaLeMdNcIPbIO_fSgShSa`WdVXeUWYcTVXbTUW}i?fhJgJ`KaLeMdNcIPbIO_fSgShSa`WdVXeUWYcTVXbTUW";
DIGRAPHS_NamedDigraphs.("bistar3,4withextramiddlevertex") := ".GkK`W@CVD`cJ?Ga~";
DIGRAPHS_NamedDigraphs.("mostmaximummatchings,tree181") := ".~?AtoGUAxjb?XKkYBXsb_\\MK]Bxqc?`M[aCXkcofLscChldOjMCgDHndonLSkDhhe?pKssEhgeOrK{uExxf?xNSyFXzf_|Nc}Fxvg@@NDAGX|g`DNtEGx~h@HODIHXth`LMsOJxtMsPK@vNCRKPzNcQKHxNSTK`~OCSKX|NsFI@RLcGIHSLkHIPTLsIHpUL{JHxVMCMJX]M[KJP[MKNJH\\McLJ@]MSEJ`\\L[?JX^Kk@IX_KsBIhaLCAI``K{DIxcLSCIpbLLOIHQJTMHxWJMtoGUAxjb?XKkYBXsb_\\MK]Bxqc?`M[aCXkcofLscChldOjMCgDHndonLSkDhhe?pKssEhgeOrK{uExxf?xNSyFXzf_|Nc}Fxvg@@NDAGX|g`DNtEGx~h@HODIHXth`LMsOJxtMsPK@vNCRKPzNcQKHxNSTK`~OCSKX|NsFI@RLcGIHSLkHIPTLsIHpUL{JHxVMCMJX]M[KJP[MKNJH\\McLJ@]MSEJ`\\L[?JX^Kk@IX_KsBIhaLCAI``K{DIxcLSCIpbLLOIHQJTMHxWJN";
DIGRAPHS_NamedDigraphs.("mostmaximummatchingstree34") := ".ak?OPUECJPOp]MGHqPPQUKGrPpI?NHdI@SkVfcAHJDW?_akKGUa`a{[ORcaackWPebaS?]RISAhWnNGCQV";
DIGRAPHS_NamedDigraphs.("mostmaximummatchingstree34other") := ".ak?OPEECHPOpMMGIQPPUUKJRPp]?NGCY@GkVfcq`VDW?_aKKGQa`a[[OScaakkWUeba{?]OGsAPWnNHdAn";
DIGRAPHS_NamedDigraphs.("dogsbone1,3,1,1,3") := ".IkKcapH?iLUEQPWc_TF";
DIGRAPHS_NamedDigraphs.("dogsbone1,1,1,2,4") := ".IgMoie_E`SYgMoie_E`S^";
DIGRAPHS_NamedDigraphs.("twindragonareatree5") := "._dahomTLePWc]LHcRaC{bLJrAsKiZ`DBSCeW_CbMAUIfAxSuXDaPwsePMGRqKsnKJOqhmCSLOQXa?QK~";
DIGRAPHS_NamedDigraphs.("disjointdominationnumbermostways,tree12") := ".KkMcg@i`LRFxW\\HOBTAYeN";
DIGRAPHS_NamedDigraphs.("path7withleaves3,4") := ".HgMoapH?h{sFWPWc_S~";
DIGRAPHS_NamedDigraphs.("stararms1,3,4") := ".HiQPWb_KeYccUGwBH";
DIGRAPHS_NamedDigraphs.("stararms1,2,4") := ".GgI[QoERD@RaU?q^";
DIGRAPHS_NamedDigraphs.("tree5differentdominationquantities") := ".ShG`HkFmKLeLjcDOaBJ_GNOshG`HkFmKLeLjcDOaBJ_GNO";
DIGRAPHS_NamedDigraphs.("mapbiusstripoflength5,width2") := ".N`A_xpdStBoFGigcTXJJgUv{GSFMKieg]?xDTCajHX\\Au~";
DIGRAPHS_NamedDigraphs.("bistar3,5withextramiddlevertex") := ".HmOhg@CKuZcIY?PBL";
DIGRAPHS_NamedDigraphs.("automorphismc3,n=10,e=15") := ".IgASWBGUOqeaK`lS@IK@cJGXRPEOv";
DIGRAPHS_NamedDigraphs.("automorphismc3,n=10,e=15") := ".Ig?[`pF?kbFaOyLS?MOwb_UPbpG\\F";
DIGRAPHS_NamedDigraphs.("complementofsmallestasymmetrictree") := ".FaY_R_bQQLD}Hi@MALHGsV";
DIGRAPHS_NamedDigraphs.("erdasandra(c)nyifigure5") := ".IeIKHCaSsqDK]?`eMbSpP`GcQeeOhboCKps^";
DIGRAPHS_NamedDigraphs.("erdasandra(c)nyifigure5,edgedelete") := ".IeASWCcSsPeNK`RFPiWHP_QPRPEW{qDK\\F";
DIGRAPHS_NamedDigraphs.("erdasandra(c)nyifigure5,otheredgedelete") := ".IcA_wrJOhgaK]?qEMbS_SFEXYDLCPboEOps^";
DIGRAPHS_NamedDigraphs.("erdasandra(c)nyifigure5,otheredgeadd") := ".IeICWDaGXhCIW{qDK\\Eha`E@WaEYPAeNK`RFP";
DIGRAPHS_NamedDigraphs.("asymmetric,deleteedgetosymmetric,noaddedge,n=9") := ".HcASPoBCPXcIY?`cK^K_QaM?WaJK`ROCK`b~";
DIGRAPHS_NamedDigraphs.("asymmetric,deleteedgetosymmetric,noaddedge,n=9") := ".He?K`pEQGaOCSsADK^Ko@cMGqPCQ?ae_Ohb~";
DIGRAPHS_NamedDigraphs.("tri-hex9,3symmetricconfigurationlines") := ".He?S?pD?XXCIYGqENeW@OBCS@dcOhgbGW~";
DIGRAPHS_NamedDigraphs.("dudeneypuzzlethecycliststour") := ".KcED`CeUlCPKe@sIxGKJAHKlYGaXKBgT";
DIGRAPHS_NamedDigraphs.("dudeneypuzzlevisitingthetowns") := ".OeBDaChGdcI`FH_EGK_AGI`BHJoeBDaChGdcI`FH_EGK_AGI`BHJ";
DIGRAPHS_NamedDigraphs.("bulgariansolitairetreen=6") := ".JiQcxEaU@sziQcxEaU@s~";
DIGRAPHS_NamedDigraphs.("bulgariansolitairen=8") := ".UgCdeblKfma_O`JPkOlNRgHIugCdeblKfma_O`JPkOlNRgHI";
DIGRAPHS_NamedDigraphs.("bulgariansolitairetreen=10") := ".igByCu\\Fcqh\\cOWmBB\\iDQHsRCQLDNGji@BBIgWHPDUJubY]?NNF|T?^Oerg{eTJkaBDoWZlOiPMaWaPghxD\\OGWXTB@IGip]sZRo@xw~";
DIGRAPHS_NamedDigraphs.("bulgariansolitairetreen=15") := ".~?AogX@gS^gw_cKac[ccksekkdkid\\SCsgdKfdsneCqe[pe{ufCxfTgF\\jLTlLdqF`hfhmfpng@pfxok`gkpfkXdaCPaPza[SNkUakWOCVN{XbcZbS]OYEbiCOlnlTpmDfNiI_XH_`I_hJ`x``pa`h_``^OSJJq@_pKOsAHAChhMj@XjP\\jX[hxOiHQPLSIyKihUP\\EL`yiXjN|FLqIgpekXcPMXRQZ_x{`@yQIQ`H}OAR_HzOYOkiDS\\lPYKSlhPq[`PRSqk_@DPyegXCRAiuDBGLAb|FCC`cSbccdectdcldSji_edChc{md{oeSreKveswfKyl?zlXilhkmO{LK|Ls}L|?MK~MDcLDeK|bKkOaKQN[Ra`|asTbA?ax~bK[b[YbqBOs\\OaDl|imLokx|PSBHKCHSDH[NKKMKSLKCKJyA`X]OKEHaE_PGOdLHtWJLYJlZJdNIDPIQHi`VPdTIqJgpkNTRLX~gxmPTEKtbKaHrIYR[FNcGNQPQSHNq?Q[@NYBQDdOibliJPadlIMRcIIYeTc?GiNStBGaWTV";
DIGRAPHS_NamedDigraphs.("truncatedk33") := ".Q___AdCfEcdfHeGhIgJ`L`KMbCDaEFq___AdCfEcdfHeGhIgJ`L`KMbCDaEF";
DIGRAPHS_NamedDigraphs.("bulgariansolitairen=7") := ".NiJ?yP_TTbik^AU~iJ?yP_TTbik^AU~";
DIGRAPHS_NamedDigraphs.("bulgariansolitairen=9") := ".]jIlKl`MfcghkjsqRaTdePWbOX_UXiTUV}jIlKl`MfcghkjsqRaTdePWbOX_UXiTUV";
DIGRAPHS_NamedDigraphs.("bulgariansolitairen=11") := ".wiAYpAbXDQxcu\\NpCcLNF^OOGIYPEbiDLUojqJG`dNDLOaaau[mfihakVUwCJIEhVkzyXUp{T@LW`Pkah[qZMfwaQEfbngGCDLGbPtAejWTxDcOqfaegPPPZMVRtSpUJj[ADdBSju\\|KjW~";
DIGRAPHS_NamedDigraphs.("bulgariansolitairen=12") := ".~?@Lf?qLbg}OdxaPchMTdhQiEiojj`clkA}hkZIiKwEvAwiALWeG_r_ykBYnLb]DFgW^MW[_MwO\\MiCaeqmbHISgha]ZJBm[JRY|PSXF_B[wNyctOsRLf?qLbg}OdxaPchMTdhQiEiojj`clkA}hkZIiKwEvAwiALWeG_r_ykBYnLb]DFgW^MW[_MwO\\MiCaeqmbHISgha]ZJBm[JRY|PSXF_B[wNyctOsR";
DIGRAPHS_NamedDigraphs.("bulgariansolitairen=13") := ".~?@dhQMci@E\\fhIRdHUUexqVeHeYlp}xMJkynRr@GByaOIC~lb]JaGeIPwyLcG|JbCdInjdEnzpOnSqCJGSl_qmEJcaFJsi@ICmoKZGrRjOtSIcihrmfMcUuMCzNSTIAPcdS_CTFS{@TVDudHdd_xYSbhI_PfXyQcxQTdhm[dxaXej[^mRazMjs{oQ?}gcA`NzWvawaHac]MbXANQwpHQjyxPj}{SJtK`AqDJWKj`axG`q|I_Q`JkBEqKsysLTAhIi[zhrhDlb`MrtDQ_cXHTG@DPtN?TTp\\hQXXWN";
DIGRAPHS_NamedDigraphs.("bulgariansolitairen=14") := ".~?AFeGneCUa{WbKYccdcSbcC`b[[bs^bkec{ggwjhXIhghHdODPNhtRITVD`SdpUdhThLGJTEG|]A[Q`[KKCTacM`xcaKOKsLKPbh|JJh]iP\\J|GiLULlTLlMK@k_Gr_Os`PC`HD`@B_WtKKCEpe_gvK[wFKyF[{FhhfxAL|?GHmfpSLdEHPh_p^MXs_x]Kxt_@aK`qhHdiX\\O\\KMHyOcpEPxO]FeGneCUa{WbKYccdcSbcC`b[[bs^bkec{ggwjhXIhghHdODPNhtRITVD`SdpUdhThLGJTEG|]A[Q`[KKCTacM`xcaKOKsLKPbh|JJh]iP\\J|GiLULlTLlMK@k_Gr_Os`PC`HD`@B_WtKKCEpe_gvK[wFKyF[{FhhfxAL|?GHmfpSLdEHPh_p^MXs_x]Kxt_@aK`qhHdiX\\O\\KMHyOcpEPxO^";
DIGRAPHS_NamedDigraphs.("bulgariansolitairen=16") := ".~?Bfioec|XHdLdD]DKid[kdkmgdDe[sekneKqeCufCxe{yf[{fs~fl?gLAg\\Eg|GhLIh]FH|MpQHqHPPdOP\\RPtQPlTQDSP}XQyWoyWouDR]BQ}CPE@OU`?KA_Y`_aa_icbKdccZc[abs^Tc`cAmbagbiiT[YSyhjA_ja_ji_i}IqENpqvpivoquoQkVcGJsHJyb`P_b@~ay?aXzap|ah}a`{```SsLKQh`pbTsFJYik`dj`\\mHqmXum`tnHymxwlhoV\\mLyylHkVLiLYwkqJUtWPI`xjMX{NUCOSinW[PXbX`Z@WRZaQlWbD_qfTB?pbEYEBTYroaeUj^oiaUe@VAxZC?Iq}ZLfLBP[TXJRJ[Ffioec|XHdLdD]DKid[kdkmgdDe[sekneKqeCufCxe{yf[{fs~fl?gLAg\\Eg|GhLIh]FH|MpQHqHPPdOP\\RPtQPlTQDSP}XQyWoyWouDR]BQ}CPE@OU`?KA_Y`_aa_icbKdccZc[abs^Tc`cAmbagbiiT[YSyhjA_ja_ji_i}IqENpqvpivoquoQkVcGJsHJyb`P_b@~ay?aXzap|ah}a`{```SsLKQh`pbTsFJYik`dj`\\mHqmXum`tnHymxwlhoV\\mLyylHkVLiLYwkqJUtWPI`xjMX{NUCOSinW[PXbX`Z@WRZaQlWbD_qfTB?pbEYEBTYroaeUj^oiaUe@VAxZC?Iq}ZLfLBP[TXJRJ[F";
DIGRAPHS_NamedDigraphs.("cospectralmates") := ".K`AC@BbEPXdNGpxeOeHAexACGAFEKarI]PbrK`KQDL";
DIGRAPHS_NamedDigraphs.("twinalternateareatreelevel7") := ".~?A?co`cCbcSdccfcshdCjdSsE[qfovesxfCzfS|fc~ft@gDBgTDgdFgtHhCVasPaCRaSTac\\H[[HS^Hk]HcXH{WHsZIKYICte`afSzfhcf`deHXe@WdhTd`SdxVL[mIpieXRLKqIPg`g``__`wb`oa`Wd`Oc`HfMkGKps_xcM[EKhq_hb_`a_WxN[AF@y_GvMK?EpooCeCK_c[ackcc{edKgd[ie_reS}E{ufKwf[yfk{f{}gL?g\\AglCg|EhLGa{UaKOa[QakSbhJb`IbxLbpKbHNb@MbXPbPOeksKSyf[|Kc{KkpJKoJClIkkIcnIxjdpULSrIXhePQLCLCKKCCNC[MCSJCkICcHKxt`@eMcFK`r_pdMSDK[CKSBFHz_OwNS@Exp_?uMF";
DIGRAPHS_NamedDigraphs.("seidelcospectraltreen=12") := ".KmQpwaeTKAex[dbpDKiWDL";
DIGRAPHS_NamedDigraphs.("seidelcospectraltreen=12") := ".KiQmBoLLAWaNqhIwN?ssHaG~";
DIGRAPHS_NamedDigraphs.("kreweraslatticen=4") := ".MgACxpGYPBpI]PROE`IdpCOhb}gACxpGYPBpI]PROE`IdpCOhb~";
DIGRAPHS_NamedDigraphs.("kreweraslatticen=5") := ".if_?gY?GCo_wyAFEp@GyEGEp?oiEDDo`OqAJERqDOlRIsQ|Gp?A`_xgvCBA`hcxBAappcz@Cap`gxACA``kz?CAP`gv@B@phgxAA`ppkzCCappcxBCA`hczNIdax`EcQhc|?aQHdBp|?`PYhw?IE_A@KGMM_`pkOQM`a@kOKI``P[GSK_apc{`SJSqlCnQKO?gWMYLp?ogYXMOogk[XMoPGkWYMO`?gWZMo@?cWYLoOo[YYMO_g[[ZMp@Gk[XMOp?gYXMrqhWmWPhCiXNOGcaXO{^OGSV";
DIGRAPHS_NamedDigraphs.("kreweraslatticen=6") := ".~?AC`gIa?NaWJagKaSMbWYbC]BK\\aKS_gRAkC@O[_WLA{?@w__?JAsA@_`_GMAS@AG^_W\\BsAB?Y__SBKDA?Z`S^`{KaKM`k`aCRaS]bCZa{_as\\acYbK[akJcWkDp?GPBcwjEGtFP@cgeE?|FpDc_iE_xF_~cOlEovFXCd?hDwqEWw_oKAwmGCH@OREGy`GTBw|GkE@gNEOr_o_CGuGcFAoWEwz_wJBonFCGAOSFG{_wYBhAG[G@oZE?}`GPB_sF{GA?XEh@_WmDwrGSDE?pGHDJC?EOuEww__sEgxFPY_OzG@BGc@F_|Fo~JcB?oLAwmE[C@GIB_sFSD@?OBWoGK@@GPBw|F{@@?MAO{FsD@GRAgpGkC@?SBGtFK??oNC?qEsA?oKCH?GcA?wWBOzG[B?w\\BonGS??wJAovFCkDpBHp^cooFhKKkfFP@IPfLSgDwqHh`L[bG@AIhcdWpEhNKpmdGrF@VKSlExCIpbL{aEozIX_LXkcg}GhSLHoMKcFG~IHgLhmdOsF`OJpiMCJ@_aDonHhMJHjLcI@ocE?pH`NJ@lLsOAGdE_tHxOJPoMKQAWeFGyIHQJPlMSSAgfF_|H`OJ`iMSXBwiGHDIPSJ@iMCZB_jFo~IHSJ`mMK\\C?hGXCHpUJhrMkUAwgFX?IXTJhjMcNBOlEXAIhVJHnM[]CGkEowIXVJXkMkLB?bEOvHhUJXnMdEHPJJpdKpfL@hMp|Np~OA@gxGHH^K@`KPbK`vN@xNPzNeC`gIa?NaWJagKaSMbWYbC]BK\\aKS_gRAkC@O[_WLA{?@w__?JAsA@_`_GMAS@AG^_W\\BsAB?Y__SBKDA?Z`S^`{KaKM`k`aCRaS]bCZa{_as\\acYbK[akJcWkDp?GPBcwjEGtFP@cgeE?|FpDc_iE_xF_~cOlEovFXCd?hDwqEWw_oKAwmGCH@OREGy`GTBw|GkE@gNEOr_o_CGuGcFAoWEwz_wJBonFCGAOSFG{_wYBhAG[G@oZE?}`GPB_sF{GA?XEh@_WmDwrGSDE?pGHDJC?EOuEww__sEgxFPY_OzG@BGc@F_|Fo~JcB?oLAwmE[C@GIB_sFSD@?OBWoGK@@GPBw|F{@@?MAO{FsD@GRAgpGkC@?SBGtFK??oNC?qEsA?oKCH?GcA?wWBOzG[B?w\\BonGS??wJAovFCkDpBHp^cooFhKKkfFP@IPfLSgDwqHh`L[bG@AIhcdWpEhNKpmdGrF@VKSlExCIpbL{aEozIX_LXkcg}GhSLHoMKcFG~IHgLhmdOsF`OJpiMCJ@_aDonHhMJHjLcI@ocE?pH`NJ@lLsOAGdE_tHxOJPoMKQAWeFGyIHQJPlMSSAgfF_|H`OJ`iMSXBwiGHDIPSJ@iMCZB_jFo~IHSJ`mMK\\C?hGXCHpUJhrMkUAwgFX?IXTJhjMcNBOlEXAIhVJHnM[]CGkEowIXVJXkMkLB?bEOvHhUJXnMdEHPJJpdKpfL@hMp|Np~OA@gxGHH^K@`KPbK`vN@xNPzNf";
DIGRAPHS_NamedDigraphs.("binarytreefirstrotaten=4") := ".Mk?kXsdUydwQnt_DbMcivSnAT~";
DIGRAPHS_NamedDigraphs.("binarytreefirstrotaten=5") := ".ihCXSmDCaowsaRF_bhg}WO_CYXQiWjDzHqsZfGYQY]cQhyAHEftSaKiVAaPW[YPHboPss^KGOALKhTKTa|cxYLrcLHLNQHS|@CbR~";
DIGRAPHS_NamedDigraphs.("binarytreefirstrotaten=6") := ".~?ACfGteswe{rEcYb[?_SB_KC_sF_kG`SJ`KK`sN`kOaSRaKWbKSakUa{[Ec\\FK]DS^D[_EC`FSbFcaF[hESgEKdDccDkfDseD{}F|OI\\PIT?G\\@GTCG|DGtGH\\HHTKH|LHtSIlWJLUI{tdPgMsoM[pMcqMkuNCjLHiN[kLhmdhjLcmMHqdxnMCrMpzf@xNy?OKvNP|Nq@ocxEkufCveWsbSZ_CA_[@_cE_{D`CI`[H`cM`{LaCQa[PbCXacTasVb_sbgxboibwjc?ocGycW{cOzdGqd?pcgkc_lcwmconfo~i@RiHQg@BgHAg`FghEh@JhHIh`NhhMi`Tj@XipVekiL@ue@reHsePtepwdXhLPzd`lLslLXkdppMSnLxoeXuN[wNH~OA@exyNh}ON";
DIGRAPHS_NamedDigraphs.("binarytreelastrotaten=4") := ".MoS|BRci@dPMnuAfgY[dOKiHt~";
DIGRAPHS_NamedDigraphs.("binarytreelastrotaten=5") := ".ihAzGshKHrqswjPKSA{aNIOQR{WoaadsK?dCJHWXTJ@HfssGg_RyiOmqLIRA[{lMIsRD?nGbqcCc~EKGgh\\B?HPAqUETQoQX|LAIGC~";
DIGRAPHS_NamedDigraphs.("binarytreelastrotaten=6") := ".~?ACfGteswe|NCW}c_|cg~cpHcx?d@@dHEdPFdXGd`AdhBdxDdpCe@IeHJePKe`MeXL_KVb@abKYb[\\bc]Kc^Kk_KsaLC`K{BFsCFxbaPILKRHXia`KL[UHplahLLcDFP``pEaHHa@G`xF`HAKpx`@@Khv`O{Mpx_wzMxw`XBLsAFha``CKxwN{LGhgNsEG@cMq@_?xKH}N}CfGteswe|NCW}c_|cg~cpHcx?d@@dHEdPFdXGd`AdhBdxDdpCe@IeHJePKe`MeXL_KVb@abKYb[\\bc]Kc^Kk_KsaLC`K{BFsCFxbaPILKRHXia`KL[UHplahLLcDFP``pEaHHa@G`xF`HAKpx`@@Khv`O{Mpx_wzMxw`XBLsAFha``CKxwN{LGhgNsEG@cMq@_?xKH}N~";
DIGRAPHS_NamedDigraphs.("alternateterdragon4") := ".k`_ggQMEcaIOmTLeRxtgbOkEjHkx@Fdg?wnQISqk_QVKfwo[kWNP?gscSOO_WoeTOqq@Gu_dCQXhBSJeSAIiUKGS\\cu\\OHdD_s[OhtMqEAa`GwYQGhAxSuXNfUaLAoYkercC]U_Ba|GhRIq@H[q^b@qp_|CAbQPP@A@bAXTBJGCbYASPHecLOmXOGihWo`PuRXt?eSUBPpAfS~";
DIGRAPHS_NamedDigraphs.("alternateterdragon5") := ".~?@qcpITDH[UePaZEhs[fpy`GIWagyUcjAmiJGeG`WQE`wiJbGuNbhAP_auBJGCPMw?OMkH?o[@AoSNBP[HCmSYwPzPDlSQvPC\\JlcTEQjCqQJ?rQZKvPsdKRzGuPc`LRi\\BhcIhOCHRiCDBSiSmhA}fMSLEQDIeMCHFQTMaJrlVgqwyTil?PCmiOSTIgq\\QTdmaHdLVUiOhStPVVySgSdTUVikkNS@[ias{OTuBIRt?TEIAIBp@TUMDBr?wQS}CBbCxQCyJBRSvQsqIBBOuQcu_GQKyNtm]FqGzNdi[FQktQtqYEqgsQduFE@coKs}ED`[pKcyHD@SdJdYGC`KcJt^qcpITDH[UePaZEhs[fpy`GIWagyUcjAmiJGeG`WQE`wiJbGuNbhAP_auBJGCPMw?OMkH?o[@AoSNBP[HCmSYwPzPDlSQvPC\\JlcTEQjCqQJ?rQZKvPsdKRzGuPc`LRi\\BhcIhOCHRiCDBSiSmhA}fMSLEQDIeMCHFQTMaJrlVgqwyTil?PCmiOSTIgq\\QTdmaHdLVUiOhStPVVySgSdTUVikkNS@[ias{OTuBIRt?TEIAIBp@TUMDBr?wQS}CBbCxQCyJBRSvQsqIBBOuQcu_GQKyNtm]FqGzNdi[FQktQtqYEqgsQduFE@coKs}ED`[pKcyHD@SdJdYGC`KcJt^";
DIGRAPHS_NamedDigraphs.("twindragonareatree6") := ".~?@?d@MQDH_VdhwVdheWexi\\fGeGawiLbG}McXA]Gh{bgQU_HHkleaq\\Jxom`pKr``Gq`Qkt`Ags_qcv_a_u_Q[o_AWpoHORc`QWDxY]DxYXeHmYfXqHaGmIbWqNbhEOfaI^GyCdgAQZJXgkfQ}[Jg[RKwWQKgSjLWOiLGKhLwGgLgCfKG?eK^";
DIGRAPHS_NamedDigraphs.("twindragonareatree7") := ".~?A?d?bcSdDKcDCfcoke?ndssE[qfondspeCreStecvesxfCzfS|fcPaCRaSTacVasXbCZbS\\bc^bs`cC}Gs~G|@HL?HDBH\\AHTDHlCHcvJKuJCxJ[wJSzJkyJc|J{{JsNCxn`oeLsLChb`_cKSJCXf`OaKsHIxh`@ULCFIhj_pSLSDIXl_`QLcBIH__POKK@Hxd_@MKe?d?bcSdDKcDCfcoke?ndssE[qfondspeCreStecvesxfCzfS|fcPaCRaSTacVasXbCZbS\\bc^bs`cC}Gs~G|@HL?HDBH\\AHTDHlCHcvJKuJCxJ[wJSzJkyJc|J{{JsNCxn`oeLsLChb`_cKSJCXf`OaKsHIxh`@ULCFIhj_pSLSDIXl_`QLcBIH__POKK@Hxd_@MKf";
DIGRAPHS_NamedDigraphs.("excessconfigurationstepsn=3") := ".FoYLGGs~@gs_bR";
DIGRAPHS_NamedDigraphs.("excessconfigurationstepsn=4") := ".KmClxCKaXRG_K`RGx[HZqGXCqeP?XAeP";
DIGRAPHS_NamedDigraphs.("balancedbinaryfillingn=4") := ".MsXLsTSm?PBGrubHmaidoAGXE^";
DIGRAPHS_NamedDigraphs.("balancedbinaryfillingn=5") := ".inEBMCw\\NubYAiUJfyPKg_cFsCg]OGqp_s\\HNHSq]?@@?o_SKFSYjp_r`MFR|gu_idax}cRIGH@|@IFcAKkWLFQRqTKf_?OOKGDB@tF";
DIGRAPHS_NamedDigraphs.("balancedbinaryfillingn=6") := ".~?ACkPYJlZJd]KL^KDfI`TKtiJ@XipVl@ilHj`oNA@eiHQIXkhpNI@laHgLLcKhhkPbLDIH\\KHk}Fx?GLAGxGHLBG`DGsyFW{FhseovF?xMkpEOrE_tM[kDgmDwoMSLKppM`tc?fD?hDOjcGaCWcCgebOZB_\\Bo^McSAgUAwWBHtaORMpvN@|Np~_?@?OB?_D?oF@?H@OJ@aAodaJP\\jX[jp`jx_kxSIhelPWJLUI|gLThL[M@wOKtPIPRLdMHxOLkPL@hk`dLLaKXghPJh`Lfo~G@@gPFH@HgXCGhEfOzF_|McuEwwFHteGqEWsEhrd_lDonE@q`heMHsMk_CwgDGiD[`CObC_dCsYBW[Bg]Bxsa_TAoVB?XMkQAXuMxwNh}N{??GA?WC?gE?wG@GI@WKOV";
DIGRAPHS_NamedDigraphs.("binarytreerotateb-emptyn=4") := ".MgMU?f_YlDQKiXt}gMU?f_YlDQKiXt~";
DIGRAPHS_NamedDigraphs.("binarytreerotateb-emptyn=5") := ".icaG{sXJ_`g?UiShrhk_hFHqapiQTKpBCKnAHOaKOdBGp``aIJJvsDSy`jFcTGgdRSQYTNicaG{sXJ_`g?UiShrhk_hFHqapiQTKpBCKnAHOaKOdBGp``aIJJvsDSy`jFcTGgdRSQYTN";
DIGRAPHS_NamedDigraphs.("binarytreerotateb-emptyn=6") := ".~?ACa?N`sSA[Q`kK_GJ_?Ib_ZbS^BsaB[Ybgbb_acK_e_reSwE{ua{U`wr`oqaGtFkOE_{agva_uaWxGKQF@?_gn__m_wn_om`Gp`?o_gh__g_wj_oi`Gl`?k_Wp_Oo_Wz_Oy`WLF[I@_yj?^bsfJKeJC`cCfJkeJcdJkcJcbJx`cP]KCnI{mIs^DXXboiJChI{gIsjI{iIs`DhZc?kJShDkgDdDHXdLLCHPcLDFHhdL\\EH`cLTHHxnh@MLsZFHbbOwKS\\Ex`M{[Ep_MspEhte?sMcrIHRN[qI@QNT[L@kLt\\LHlL|XKxjj@eLQ?ixZJx}OLUJP]Ny?ocO@{Ma_RaSL`c@@[?@S[B[Ybw]cOZbS\\C[[CS`cCsE[qf?vesVasNE[MESPEg|a?sFcTE{SEsRFH@aOwGCDD{CDsFD{EDsHEKGECDDKCDCFD[EDSHDkGDcBEKAECBF[AFSJ@gz`OKFTWB{]cxXcpWcK_cx\\cp[ch\\c`[cX^KKaJp_dxVdpUbwjJK]DPWdHVd@UdXVdPUcGlJ[_D`YdGld?kghJKhhg`IK`ggxLKhjgpKK`ihHNL|GHpmbWxK[YF@abgvKHvb_uK@ueGtMkoE`seXPIXzePOIPyj`gL`mjhhLhnjHfL\\WKpiODVJX^Nq@ipYJp~OF";
DIGRAPHS_NamedDigraphs.("binarytreerotatea-emptyn=4") := ".MiQwgefK|RSQmAT]iQwgefK|RSQmAT^";
DIGRAPHS_NamedDigraphs.("binarytreerotatea-emptyn=5") := ".idahgqDC`sqSyKScphC[lFGqAKc`DFrQ[oc]_dG@WxTJgIxaoZheRTCccQpSYQU\\RqcQLPidahgqDC`sqSyKScphC[lFGqAKc`DFrQ[oc]_dG@WxTJgIxaoZheRTCccQpSYQU\\RqcQLP";
DIGRAPHS_NamedDigraphs.("binarytreerotatea-emptyn=6") := ".~?ACagSawUbGWbWYbg[_[D_cF_sjDSlDcnDs_cS`bwb`{HaSK`cI`sOECNESMEKIC[HCcRDCPDCLDKLC{JC{JCkGBsYbXFbk[HK?E[@ESAEKtFStF|CGlQEwwfG}gPBfO|HksEsxf@Ve_{HcbHXRdHPIcfI@PIcdI@PcovcsjJ{idX`d`HdhIK[nKKmKPddxGdpFK{cDHNcgec_iKCGHpNaHUJKOI{RJCQJp^apELcTJSVHpTMKXKpgb@dKxsa_oMTZKHhL[zLpujh^LTQJpiiH_KPvNKuM@xNT@KXliXVLxqNDQJ@mg@nN@~d@cNi?dhyNy@d`UN`|OUCagSawUbGWbWYbg[_[D_cF_sjDSlDcnDs_cS`bwb`{HaSK`cI`sOECNESMEKIC[HCcRDCPDCLDKLC{JC{JCkGBsYbXFbk[HK?E[@ESAEKtFStF|CGlQEwwfG}gPBfO|HksEsxf@Ve_{HcbHXRdHPIcfI@PIcdI@PcovcsjJ{idX`d`HdhIK[nKKmKPddxGdpFK{cDHNcgec_iKCGHpNaHUJKOI{RJCQJp^apELcTJSVHpTMKXKpgb@dKxsa_oMTZKHhL[zLpujh^LTQJpiiH_KPvNKuM@xNT@KXliXVLxqNDQJ@mg@nN@~d@cNi?dhyNy@d`UN`|OV";
DIGRAPHS_NamedDigraphs.("binarytreerotateright-armn=4") := ".MiESyY@\\IkHTSq^TBI\\L?mdUCiiXN";
DIGRAPHS_NamedDigraphs.("binarytreerotateright-armn=5") := ".if@GkYNHcQxSuX@_OwTSrSKSr\\GtOGr`|uxaPvsIQ{_QUBIHFULGcl[ubRYhoQJEbqXCmTLeOWCMDTKtBDKvQLSAKw^\\mWc\\|AcnGCd_qaPtbQHJVLgsv";
DIGRAPHS_NamedDigraphs.("binarytreerotateright-armn=6") := ".~?ACdOXbCZbS\\bc^bsUa{M`{OaKQa[Sak`cCbcSdccfcshdCL`cH`CJ`S@_CB_SD_cF_s~Hk}Hd@H|?HtBILAIDDI\\CIS|G{{GsqE[uE{sEkyF[wFLXjDZjTTKDSJ|VKTUKLJIHchPOK\\HIXehHQKkpHhge@KK{nHxidpMLKlH@cd`GK[jHXedPIKllLt\\MD[L|cLTbLLyN\\jMHtNTkMPuN\\fL@wNH{kpvNH{NldMxwN`}jprM`yN\\EGxvN@xociBKWb[Ybk[b{]asV`sNaCPaSRacTcK_c[ackcc{edKg`kK`KG`[I_K?_[A_kC_{EfxLfpKgHNg@MgXPgPOghRg`QfhFf`EeOreove_tfOzf?xjLWj\\Yih_i`^ixaip`hXPKdII@bhHRKtHIPdeHLLCoH`fdxNLSmHphdhGKckH@bdXJKsiHPdlhmjhoj`nk`ikXhnPzlXpMhyl`qMpzkxgN@xNdeMxxN`|khvN@{Nt]MXsNPzgpFMxwNN";
DIGRAPHS_NamedDigraphs.("dextern=4") := ".MeEDrTLGdsrMaPRI_Cid}eEDrTLGdsrMaPRI_Cid~";
DIGRAPHS_NamedDigraphs.("dextern=5") := ".id`gkGPBCp?{GPQCBI_gVPcay\\OlRIpasCGF_?pHiWRIsqcgaXNTr@yMVKsaxuAIFEG?SYQlGCQQ]OIgSTk}_OgwgWa\\NHWOKGPQhc}iUEao`CKRCBo`DGOKiA`\\EQJhtAtKjEJOO_]?BCeh`KjRIQaHc|VKFgx[rQJfWGg[W_@PhIs_PHHx?i`PUry@Aba`aHs{d`?o`DIeR~";
DIGRAPHS_NamedDigraphs.("dextern=6") := ".~?AC_wD`GC_sG`gK`wMb?QaSRaKPboZBkZB_\\bSaAkScgOb?Xa{dcsWDKgdgTcOlcCbaokcKB_GNE[A@cvEsvfOcESwfgjEcUdWkgGadGmG@@dwtf_~cwmGCxF[`FckEPCG{yHDCHKpFo~bgo`GPBsIBOn_WUD`F_gdDXJ`WXDH@`?QDPP_ORC@B_GOD?|_?SECCB_\\Hk??_Dh`TIpVhPJI{xHcxHSrGCnFGzgpHJ[fEwzfOzJsoF@EJTDGp^fH?KScEowKCtFPIeW{Fg}FxN`OeCx?`W^CXK`wiDhAJcKD?qG`d`g`F`SLSMAh@IPh_WVDhDJc?C_pHHX_oLAXBHsF@oOFg}IDTJp`K`mgx]KPlcObEx`KT@JPbL@rhp\\JxbK{]C?wJP\\gXZJh_Kk{JX[KXqaw`Ep^K@qj@XJPZMK@?oVB@[K{A@GcChXKkF@?SB`WLC}L`sN@yNlLIXcMpvNP{lXnMhvN@xN[XBOZExsMhua_TAouMHqMX}a?PAOREp}Ny?OMC_wD`GC_sG`gK`wMb?QaSRaKPboZBkZB_\\bSaAkScgOb?Xa{dcsWDKgdgTcOlcCbaokcKB_GNE[A@cvEsvfOcESwfgjEcUdWkgGadGmG@@dwtf_~cwmGCxF[`FckEPCG{yHDCHKpFo~bgo`GPBsIBOn_WUD`F_gdDXJ`WXDH@`?QDPP_ORC@B_GOD?|_?SECCB_\\Hk??_Dh`TIpVhPJI{xHcxHSrGCnFGzgpHJ[fEwzfOzJsoF@EJTDGp^fH?KScEowKCtFPIeW{Fg}FxN`OeCx?`W^CXK`wiDhAJcKD?qG`d`g`F`SLSMAh@IPh_WVDhDJc?C_pHHX_oLAXBHsF@oOFg}IDTJp`K`mgx]KPlcObEx`KT@JPbL@rhp\\JxbK{]C?wJP\\gXZJh_Kk{JX[KXqaw`Ep^K@qj@XJPZMK@?oVB@[K{A@GcChXKkF@?SB`WLC}L`sN@yNlLIXcMpvNP{lXnMhvN@xN[XBOZExsMhua_TAouMHqMX}a?PAOREp}Ny?ON";
DIGRAPHS_NamedDigraphs.("transpositiongraphn=5") := ".~?@w_GA?_GA?_GA?_IKJB?sMcpO]GHcYF@}TD`sadp_ZGXGRFQANEPk^bpSZGh?WFAEOD@o_c`[\\GXCYF`}PD`wabpGREXCQDpiOCPWWbp?SDW{OCPIKCpwaapO]FwsTFQEMD`s_b`g[GGsXFAEKDpkaap_ZFwoLCpeJBPOTaowUEGoMDpiREPk\\GQIUE@o]FqASDPk[FqEVE`s]GAINC@KUGAIPC`_XFqENC`OYFqAOCPSVGQILC@GRDAEJBpCTD`}KBpCXEaIMC@GVEAAJBPKUFPyKB`_XEpqKBPOYF@yJB`SVEpuJB?{RD@mLB`?TD`qLB`GXE`uJB@CVE@yFAQKcHrPHRDDSa?gbHQWsQclQSwCAHASeHsDAOsQCAQ_rNC@FRD@P_ockKBk~PSpMSWOGIRCwNc\\ISDMD@qkpLRxGQS|S`OgnKrh?QClNSgKIIbGxNSTJRdIE@qsqLbtEQStS`__mKB[~PchLSwCCIasuMSLCPtAA@a_nMbp@OcXL_OSkJb[zOsPGRwGBIQktMCDAPSyA@o_bHAWtMBxL_O_HGqWfLrk~SGGHAaKdHrg{OCy@@ogbHASuMRtN_O[IHa[iJRGsQGGFAASfIQkpLCYAAOgcHa_nKrPD_O_HHASkJb?sPwOEAagnKbKuNCiB@?[jJRCqMBdK`OWHIAooKr[yQWKDAAcmKBCtMsmA@?ShIrKyNBxB_OKEJAwqLbc~OgGC@Q_nKRSwOCQ@?oWiJR?vMrt@_oOFIQgtLbs}SWKDAAkkLr_}NtID@_cmJrk{Ns@S`?WIIAsxMbt?S~a?_GA?_GA?_GA?gokKBOyRD@w_ePg[FxSUFQIVE@k`c`K\\GG{XEp}NDPkac@_[GX?SFAAQDps`cPg]FxCUFaINC`KXcPGVEh?PD`aNC@OTbp?PCgoRFaIJD@w^bPS\\GWwUFQAME`o_bPc[GWoVEqIJE@k^b?sREWkLD@UJB`WWb?wVEhKXEps`GhWWF@w^GHOTEpo^GX[YFPw_Gg{OCpW_GhCQE@c^GW{QD@g^GH?PDP[`GgsOC`KSGWkNCPSUFwoNCPcYGgwOC`[WGGkLCpW\\FgoME@cZFGoLD@g[FgkMDP[ZFWkKBpKSEwsMC@SUFGsMC`cYFWkKCP[WFg[HGqOfLCdKSTQGAaKdHbPIQtHR_OGcHQWfOSHBPGOHIBK{OC\\KSDEBAQooMr|DRCxP`?_hKR_}PshOSwSFIrCtNc`HRtQDAa{rMc@GQs|Q_ogiKbc|PSlMSgWFJRGuNSXHRTQEAAwoLr|EQctR_OOiJRWxOsPFSGGEIA{yNCDAPcu@@QomLrlBPC`N__KhIrSwOSHDRgGFAAKcHbSwNcu@A?cbHa[vMr|O__cIGqSfMbp?RgCFAaKcHRWxNS}@@ogeHqglKbPG__[GHQ[hIrCsPgGHAaOeIA{rLCU@A?ccHQomKBPF`?WIIa{qKrW{QgKC@qklKRGwMSqD@_cgJB?rLrhH_oSGIQwoKRSzQwGC@QcjKrg{NcM@?oWkJbGuMR|A__ODIA{pLR`?PGCB@aglKB[zNSEB@?[hIbSuNRxP_oSGIqovMBw~SgSEAQwnMro~ODQC@_ggJRcyNS@R";
DIGRAPHS_NamedDigraphs.("gra$?tzerexamplelattice") := ".KeACXPdKyIDLK`RFPqWGPdESrggSsqDK\\F";
DIGRAPHS_NamedDigraphs.("transposecoversn=4") := ".Wa?``_iAEFaEHcGHbEFdFGcEHdGHbFG_IJMO`KLNPiNOPjKLMaDFHSTbCEGSTwa?``_iAEFaEHcGHbEFdFGcEHdGHbFG_IJMO`KLNPiNOPjKLMaDFHSTbCEGST";
DIGRAPHS_NamedDigraphs.("transposecoversn=5") := ".~?@w__E?_WAI@?YAAWOE`OaAAWSG_o]B@wGC_oU@A_q?BO}?C@E@Aoy_EPu[FxgZe@ySDxOVdPYTDgw_HGkbHWgbHWo_HG{`HhCaHx?`HgsaHwWXEqmHF@sg``_YIg[XEquHF`{ha@w^JG[[FQyGE@gnmaWfhAUXIBe\\IraWIRaYJRe^Jba]IbeZJra[JBeYIbczfqcwMxogMRm\\Jb_yeqkwMxwkMRiWJr_yeQsxMiKtLsPDQSiaKBHCQChJgBKuPc\\GQyCpLCTEPseIHB?{PGkcKRtDbaSsOSYKHRG}QHCeLcLJcA[rOc]NHr[~QWseLS@Ia`GbNCEJCaK|NgoQGBs}cPKaNsMOCqD?Og{RGR|BbPKaOCIMCa?{OWWYErlOSwc[FrlPSg_WFbhUTw[XFRhSTWOVIalOStd^_`OgITDQUDiDDaonTd\\ZVGKTJQxSTTt]rD@PUDfMSdLYV{|TTdp\\rTPVUtyoNSHIWu`jlrxAPU@gYjC{OsdbYEitOSLCWU`jkb|@QuHhYzK{NsX_YUiuNS@GWUdjlBx?PuHhYgWWET`YZEto[W[ZFDl[ZExp[gc\\Fdd^Ze|q[w_YFtt]ZU|o[~aA?WA@_GgC@gGH`?YDAGGH`OaB@wKF__QB@WCIBG?LBw?OCWCJBi?XFXo^e`mWFhOVd@]TDhSUba?caqKdaaKdbA?cbqCecQGfcACebQGf``cZIwc[FQaEE@gi`pcZJWc]FqeGF`{k`po\\Jg_WEa}yHa]cHXcgMXsjMH_hMHglMX{mMHwiMXknMHokMXgiMRm^IR_zfA_xMxsmMBiZIr_zfaoxMh_nMBiXJRcygrSvPCTHQiGoKcPGQcm_KrXEPs`JgRCsPSXFQWgcKBpCaqOpNSUMHRP@PgodKbxGcQWuOsmOHrLAPw{fLr|HbQWtOCiICaK{OWkQGrs}b@G_NRyPCqG~Ox?RGS@AbpK`NsMLCqH?OgwQGBp@``gZMt@RaPo^MtDQa@_]MdXV`pc\\MdPT`@[iIt@RUT}ADA_hSTHWUgSUJA|UTtl[_pSlJdPTVTzKSDDWU[xQSth^rtTUVDvLTD\\ZVj?|OchbYEmvNcHDWE`ikRpBQULgYjT@OsP`YEmqNsDJWedjkro~Pe@hYjW|OC``YUmsNc@FWedi``_XUDhkZV@p`pk[UtpkZfDqaPs]UT|mZvHra@g^VTxlZv@r";
DIGRAPHS_NamedDigraphs.("fibonaccilatticeton=6") := ".`d_`TC`JGt@_{fKFCxP?dTJSaX]aSKR@h?mX``p?eoZaa@g{q[bAPo{e[a@pgwm[_?OWOIEMWXWGTPGQqLOWNHr@pMSOHTQtGeVgdBCoYOJeWW[OHkEwg_YNKfGoc[NHfG_[YMJfG?CECA`bf";
DIGRAPHS_NamedDigraphs.("pappustheoremgeometricconfiguration") := ".Hg?SPoEOkPCJ?G`eNe_@PF?XApCOk?aEW~";
DIGRAPHS_NamedDigraphs.("21,3symmetricconfigurationincidence") := ".i_OGCgMGCp@CWRDBpGsUHEapxKaoSJtQ|WnNISAswjOGrq[wdKGqq[sdALfgXs~@MGGH`BBLGwPdDi_OGCgMGCp@CWRDBpGsUHEapxKaoSJtQ|WnNISAswjOGrq[wdKGqq[sdALfgXs~@MGGH`BBLGwPdD";
DIGRAPHS_NamedDigraphs.("dragoncurvelevel4") := ".OfEhFdEcGaB`H_GaBCDofEhFdEcGaB`H_GaBCD";
DIGRAPHS_NamedDigraphs.("dragoncurvelevel5") := ".\\eCDhFfabnMlonqpsr`U_TjKgHIXbCDMaGLX|eCDhFfabnMlonqpsr`U_TjKgHIXbCDMaGLX";
DIGRAPHS_NamedDigraphs.("dragoncurvelevel6") := ".uc`WkQHMEqIG]eB`t_SIdOipguD@^pW[eNLd_iGA]qYNUr`ycRIR`|_ylifYp\\[ogCTYqYmVkH?kW_baPP~lHApWcQ[LcSO|KFBj?gTI`TbPkIA~aoxK]ZJ@SOC|cs]lfBtGeSfBz@tZTMtayx`OGitct[nWQ@Wp@FCab~";
DIGRAPHS_NamedDigraphs.("dragoncurvelevel7") := ".~?@dc?oLb_}RChWSf@eWexiVdWmJ`gUC`GIA_wMIageH`w]GaJ{}n[A~okFCowDE_CUaGr]qKrqoKRm`Ja}_JAuSLBU[Frq\\FbmeHygjiAexMdEtLx_wShSwRHkdQtAYHChNeRXRdrdLSXKUSjhRlcdTlDP\\cdXVVKxTV|h\\VeEEB_{}`OoLN]UOB?uMBxKQd`Q[EXaZeh]TawmE`WQC_gIB_wiIaWeF`waGnry|oJ~Ao[RB_SY?PYGbLzGrNJ?pMyCmJy?kJXOsLXo^NHs]MyWfiamgIZcySZSveB`QdR`KeqTJSHgcQc}XLdMVMStPcpXQmdMuQTUsTDuQTd\\[rdT^udt]WWWMBryDB?s|";
DIGRAPHS_NamedDigraphs.("dragoncurvelevel8") := ".~?A}awRAcTAs]BkiBcZbS_b{acKhdCfbKeckcc[G`CF_{E_sC_cD_kQaSM`sPaKN`{OaCL`kH`KI`SK`cJ`[B_TpMDnmTpmdrmtt_Hw_@veWsLcuF@megvLkqLPjeHgLKkHhMdpPISlHxOe@TIsnIXShXKLtHHPlgXCKtFH@fghEKkgD[hDSfKSeKcdK[YJa?c@ZOS^Jy@cPcOc`KYBkhij`\\jH]jPZk@ekxhc`WNiEcXVNaDfW{QAVfGyPyTfg}Pc~GAJPt@GQIPk]QK\\SK[PYMQ[ZPQLQSXN\\kSQdjIdji]l\\]Q|`jy`TL_T\\gQigkQGRL`PISjQFRAitYoqqfTmeS}URAiTmaTastYrUehUQuU}KTAnnyXTqxnqSTIq_WTApo_ORA`nvsVAWSagUbo\\dO[b[YcC^cS`dKgc{Xcsdccb`CG_{F_sE_cC_kDaSQ`sMaKP`{NaCO`kL`KH`SI`cK`[J_[AmHol|qmLsm\\umk@NC?M{rE`keowLstExlePiL[pL@hd`LHsmIHQdhNICoIhUdxRIdJH`mhHILlBG`egxGK|DGpdd?jdGicxacpcchbbP[OC_JYAbx^OKaKaCcHbO\\dLT[JlXJtYJ\\_KtfLKcJ@|OsbIx{OkzFaOQ{xFQNQk|FqKfx?PYMgHAPQLbqPbi`baJPqRbYIPiQbHzlaaSlXSl\\RtjjqVkL^SIhkAjlATTDaPAXkIHQdYOyWTUjUEUSylsqfqqWTQlsQkUejUYstIqUqvpagT|~RImVL}QahUSBAgUMCAAWSL~";
DIGRAPHS_NamedDigraphs.("dragoncurveblob6") := ".Pb?eCD`AiGihbCD_HLgJKM`AJKpb?eCD`AiGihbCD_HLgJKM`AJK";
DIGRAPHS_NamedDigraphs.("dragoncurveblob7") := ".aa_?K[HC`ox_IDEiCyHI`JEBXxAKFCR`xeQIKEiX]UQMSQpai\\NTApo~??dAuII??pocQFBe?gSYgRgciCkWLfcGo[PMFeXGgoYhdxXGxPJEIhs|SJFB{?ASJ^";
DIGRAPHS_NamedDigraphs.("dragoncurveblob8") := ".~?@Cc?}S?oQUCxMTcPILBggJfOY^FiK`Gh{_hpud_AU?HWEB@@oi`QeljWSFJwGGJa}GJwcoKg[IArMHBBILBbUKBrYOCPGv_PaTD`[xha_hJgWfIreAHayeHq_jfAGigQ_jgAWffaO|OYKdNsBCc?}S?oQUCxMTcPILBggJfOY^FiK`Gh{_hpud_AU?HWEB@@oi`QeljWSFJwGGJa}GJwcoKg[IArMHBBILBbUKBrYOCPGv_PaTD`[xha_hJgWfIreAHayeHq_jfAGigQ_jgAWffaO|OYKdNsB";
DIGRAPHS_NamedDigraphs.("two4-cycleswithcommonvertex") := ".FoBcgCb~?MQ_QN";
DIGRAPHS_NamedDigraphs.("twindragonlevel5") := ".~?@Xa_eGc?uKbwyC`GUD_GA@_WIA_wME`g]FkaGbKY?`KHCYEx?WEXK]FxG[FXWVd@UkJisniQkxiAgwhrEeKG{dLwwcLgsjLReKIbOwhYQfPIXDjyylNYo{iRK~Qy_qNciJQWhGaSDDRw`?PCzDR{PMqCdQSzhFQTIzPc`Rmb|FQzk}PcjXa_eGc?uKbwyC`GUD_GA@_WIA_wME`g]FkaGbKY?`KHCYEx?WEXK]FxG[FXWVd@UkJisniQkxiAgwhrEeKG{dLwwcLgsjLReKIbOwhYQfPIXDjyylNYo{iRK~Qy_qNciJQWhGaSDDRw`?PCzDR{PMqCdQSzhFQTIzPc`Rmb|FQzk}Pcj";
DIGRAPHS_NamedDigraphs.("twindragonlevel6") := ".~?Aja?N`sSA[Qb_VasXbCZbS?_C@_KA_SB_[C_cD_kE_sF_{G`CH`KI`SJ`[K`cL`lUF_|IkyFXSbgoEK[DonbwsEk]EOrcGwFK_Eovd_ldOjc_dcObd?hcofh`OhhPhxRhpQghHKLCH@_gxJK\\EHPagXTgPSbX@JkYG@[axHJH`apGJ@_bHJJXbb@IJPaf{}gL?gXvgPuhlKh|MiHhi@giXjiPigx^OLEJq?fx^fp]ghVLh~g`UL`}ahza`yaH|a@{aPtPCRMaH`xnMyJ`pmMqIqYQqMKQeLQ]HQuGQlzNiWnP{Q}BOiQoQCQLvPYYmqIRLtQAPQlsQAQQttOiOQTsOaOQLfNiSRDeNaRQ|fOISRdeOARR\\cNHzRtdN@yRlcLhxN|dL`wNuja?N`sSA[Qb_VasXbCZbS?_C@_KA_SB_[C_cD_kE_sF_{G`CH`KI`SJ`[K`cL`lUF_|IkyFXSbgoEK[DonbwsEk]EOrcGwFK_Eovd_ldOjc_dcObd?hcofh`OhhPhxRhpQghHKLCH@_gxJK\\EHPagXTgPSbX@JkYG@[axHJH`apGJ@_bHJJXbb@IJPaf{}gL?gXvgPuhlKh|MiHhi@giXjiPigx^OLEJq?fx^fp]ghVLh~g`UL`}ahza`yaH|a@{aPtPCRMaH`xnMyJ`pmMqIqYQqMKQeLQ]HQuGQlzNiWnP{Q}BOiQoQCQLvPYYmqIRLtQAPQlsQAQQttOiOQTsOaOQLfNiSRDeNaRQ|fOISRdeOARR\\cNHzRtdN@yRlcLhxN|dL`wNv";
DIGRAPHS_NamedDigraphs.("terdragonlevel4") := ".k`_ggQMEcaI?E@LeRh|_lVmAr[gtIGebkk_ZMTBDSrQIgCLKg_Ogh?c]TKgWo[[SKGG_SmYMo_XWu[hDbA@DRJeSIMYQJEsyaWRJecqfXB@POc[LHCS?KAZKfRz@Yn[DexPiSPLFXX?u[iEIhecTOGYXP?`PQ@G{iXOp_wwgWOP?h[s\\`?qpkxQJECAIeVKgS[scULhtCoeVLHdN";
DIGRAPHS_NamedDigraphs.("terdragonlevel5") := ".~?@qcPARChSSdpYXEHkYfPq^FgE?haQdiaahjaql`WQE`waHagmKbWyN_qMAGgG`GbuBGAK{oSB?o[@AoSNCpZmylcXHls\\GmS\\GQz`EQSitMsasMcerMCdLRzGxQCpMjsImOyp?OSQlOCDDicHCPilBPS]hKTMgKDIfKs|OhbHMSYsvPS\\JTYouPCXITI_iOdHUUYcjOtLVUIsnOCHQTYomOSLRTISfLC|YhAWtRdmLIrdFRD]KIb`ERTYHGqOtWw_aHRPaf`{mKTLWf@snKDHXe`kaHRtfe@cbHBpeao{vMs`Ja_wuMcdI`@WVIRDW`POTIB@X`pGRHrLY``?PHbHZ{hCOcpITDH[UePaZEhs[fpy@_IWchYggiYwkjWUC`g]GaWiJbGuMbwKb_aIAGQG|_q?bNKD?oKF?OkDBpKUzmjXEQZ\\FQJdFQCmwPcdIlRlGlBhHkr`HRS}qMS`KRi|AjcMkOCDCjS@@PYhAPCYjOsTFiRDRiB@QhrLNSIWqRdElLsTFQtUkLcPEQdQgIcHQTdehIsLRTtalJs@ASdUkJcDBStQdHrPNUiOeLSxZbQkxPspVbAgwPctUaQKcLUMGGaSsWhw^JbDRUHo\\Jr@QUXgZGaS|Xx_XGqO{XgkNLrlGQwgMLbhHQgOUDqcpUGSSDQ_oUW[QCq[rUgWOCQWqU~";
DIGRAPHS_NamedDigraphs.("hypercube6") := ".~?@?_GA?_GA?bO[GAOgJBG[GaoqIBGgJaOiHBGcJ`oeGAW_Ia?qGAw[K`omFAgwTDgwXEwwWEgwVFHGTFHOTExKTEhGUDxOUEXKUEG{WEXCVEX?VEG{YExCZFH?YFG{OCXCQDH?QCw{RDGGBHq_hJWCBIakkJWGCHQWfKGGDHAWhJwGEHASgJgCCGaKiKGCDGQKkJwCEGQGjJgCAJQwnKGODFP{bHgOEFPwaHWSEFQ?`HGKCF`{fIgKDFq?hJGKEFa?gI{A?_GA?_GAL@o_HA_kK`oaJBGgKa_mHAgcKaOmFAW_Ha?iGBG_J`oqFAw[Ib`SUb`cZb`_Yb`[[c`S[d@SZcpSYc`WVd@WXcpWWbp_XcP[Xc@[WbpgZcPk[c@g[bp?PcPGSc@GRbpKS__KfIAcl_OKiIqol__OdHa[o__ScHacn__WcHQ_m_OOaGqgo_OS`Gqon_OW`Gakm_OGlJa{o`?S\\FqKe`?W\\FaGd`OW\\GACc_oO]Fq[i_oS^GAck_oW]GA_j";
DIGRAPHS_NamedDigraphs.("hypercube7") := ".~?A?_C?_C?_C?_C\\@?RBGYBW[`?SAgUAwW`GI@WKAWS`_L@wQAoX`WLA?PAgY`OM@wPB?Z`GMA?QAw[bg]Bw_CGaC[GAWS`?UBKGAgY`?WB[GAw[`WK@kH@OM`OK@{H@WO`OJAKH@_Q`gNAKLA?Q`oNASMA?P`_RBKJAWY`ORB[HAW[`_SAsJA_T`OSBCHA_V`gTAsMAwW`wUBCQAoVaGTBCOAgV`gXBSMBW[`wXB[QBG[aGYB[OBO[eG{Fx@e?{Fp?eW|G@@eO|Fo~e@AG`EeHAGhFePBG`DeXBGpFdOwFG{dWyFW|dwwFW~dgxFX@doxFP?d_wFO}dOsEhAdWuExBd_sEpCdwsExDdgtExFdotEpEe?pEOrdWlDordWkDwqdOlDwpdOkDoocgeE_wcgfEgxcghEwzcggEoycofF`Ad?hFhBcogFpCcwhGHFcwgG@EcohFxD__DI@UKHaKkD?xTJ@`K`f__FIPXKHbLSD?pSJXaK`h__EIXYKPbLCE?xPIxbK`e_WDHHKKhfLKB?`GHhdL@i_WFHXMKpfLSB?pIHxeL@h_OBH@HHPJJcA?hHI@SIh__OCH@OIPRJ{A?xJIHQIh]_OEHPPIXSJk@?hKIpWJX__GCHhUJHYJ{@?xMIxWJH]_GEHxVJPZJk@?XKHhMHx[_GAJ`\\Jp^KE?_C?_C?_C?_C\\@?RBGYBW[`?SAgUAwW`GI@WKAWS`_L@wQAoX`WLA?PAgY`OM@wPB?Z`GMA?QAw[bg]Bw_CGaC[GAWS`?UBKGAgY`?WB[GAw[`WK@kH@OM`OK@{H@WO`OJAKH@_Q`gNAKLA?Q`oNASMA?P`_RBKJAWY`ORB[HAW[`_SAsJA_T`OSBCHA_V`gTAsMAwW`wUBCQAoVaGTBCOAgV`gXBSMBW[`wXB[QBG[aGYB[OBO[eG{Fx@e?{Fp?eW|G@@eO|Fo~e@AG`EeHAGhFePBG`DeXBGpFdOwFG{dWyFW|dwwFW~dgxFX@doxFP?d_wFO}dOsEhAdWuExBd_sEpCdwsExDdgtExFdotEpEe?pEOrdWlDordWkDwqdOlDwpdOkDoocgeE_wcgfEgxcghEwzcggEoycofF`Ad?hFhBcogFpCcwhGHFcwgG@EcohFxD__DI@UKHaKkD?xTJ@`K`f__FIPXKHbLSD?pSJXaK`h__EIXYKPbLCE?xPIxbK`e_WDHHKKhfLKB?`GHhdL@i_WFHXMKpfLSB?pIHxeL@h_OBH@HHPJJcA?hHI@SIh__OCH@OIPRJ{A?xJIHQIh]_OEHPPIXSJk@?hKIpWJX__GCHhUJHYJ{@?xMIxWJH]_GEHxVJPZJk@?XKHhMHx[_GAJ`\\Jp^KF";
DIGRAPHS_NamedDigraphs.("gardnerdigit-placingpuzzle") := ".GcACPPF?OqDaGXAepGCGaaM?`cJCOqDL";
DIGRAPHS_NamedDigraphs.("gardnerdigit-placingpuzzle,complement") := ".GeISqOCMHAepKShc_G[QDL";
DIGRAPHS_NamedDigraphs.("alternatepaperfoldingcurvelevel6") := ".k`_ggQUIfAI?}ePHTOK?hBaPRD[pXLUgoqKFcAQcgrKErb{l?`ecQQMMHPRsT?m[RVBqTLZMgCLkw\\NOah[pk`_ggQUIfAI?}ePHTOK?hBaPRD[pXLUgoqKFcAQcgrKErb{l?`ecQQMMHPRsT?m[RVBqTLZMgCLkw\\NOah[p";
DIGRAPHS_NamedDigraphs.("alternatepaperfoldingcurvelevel7") := ".~?@Oa?WFb?QIcOyJd_eTf_CDg@}BFP}AF@}e@_[O`?SNbguKHwkec@OjbpKicacjcQ_ijQ}kJhOXKXKWKGghaQahKY_odbUTLJSvlBYrLzGuepuYFHcZNX_YNH[wMZczmBizNZg{gsHBPIS~OSYcNc@DgqTCPiKcOsU_GqOdgAGdNy?`HBy??_K^sG_E@woCahCMaxWHdXw@@Y?^_ps^_`o^h_WFCGODBwyLbA]JHh?SIw{RIhGhIxCgIisnjAySEREREBAIIWcgiREgKHWtdRQtLzOukr]qLhk\\e`qXEruWEbqVMBexMz_ymruyNILAOsQdNsDEhBx?PYKdPCYbHCLDgAKcHY?aHR}_GQO}_?GBF~";
DIGRAPHS_NamedDigraphs.("alternatepaperfoldingcurvelevel8") := ".~?AWaWQagSbGWb_Ubo\\a{`CCbCSdCcfCshDCYb[OAK@_?Ae?nd{tEOsecyFKyfS}Fk}fs}gXAgpCGkqE\\HEwwgXHgXHhDMGsrh|FH@NixLItSItSIlTh`LIXUiPSIlQIXSItOIPRh`OIHRhlBh`LexGHXahX_KHaf@@GhIfx@GPIg@@G`Df`?GczF_|GCuHsuF@DHstExFHCtEovFCpE`ie?qE[oEGqEcQa_laWxMKTDxqaonMS[EHtbxnc@mMsaFPrc`gcHiNCeHpzbKgEXvdHNNsiJsjJcdG`ycxENaAbhTOK]JP|OcZIQ?OKYI@~OCbF`xNSVE@sM{B@wxMKC@wxF[D?o|F{F@HAKKL@pVJCI@`YNkJ@pWJKJ@`XJSG@hVKCG@H_KKE?w~GSC?gzFkA@OWNkBA?PMMWaWQagSbGWb_Ubo\\a{`CCbCSdCcfCshDCYb[OAK@_?Ae?nd{tEOsecyFKyfS}Fk}fs}gXAgpCGkqE\\HEwwgXHgXHhDMGsrh|FH@NixLItSItSIlTh`LIXUiPSIlQIXSItOIPRh`OIHRhlBh`LexGHXahX_KHaf@@GhIfx@GPIg@@G`Df`?GczF_|GCuHsuF@DHstExFHCtEovFCpE`ie?qE[oEGqEcQa_laWxMKTDxqaonMS[EHtbxnc@mMsaFPrc`gcHiNCeHpzbKgEXvdHNNsiJsjJcdG`ycxENaAbhTOK]JP|OcZIQ?OKYI@~OCbF`xNSVE@sM{B@wxMKC@wxF[D?o|F{F@HAKKL@pVJCI@`YNkJ@pWJKJ@`XJSG@hVKCG@H_KKE?w~GSC?gzFkA@OWNkBA?PMN";
DIGRAPHS_NamedDigraphs.("la(c)vyccurvelevel4") := ".OfDegfcIbHaD`E_BCofDegfcIbHaD`E_BC";
DIGRAPHS_NamedDigraphs.("la(c)vyccurvelevel5") := ".]iHgmJilknmhgfLeKcDbTaS_OQ`PRcDEF}iHgmJilknmhgfLeKcDbTaS_OQ`PRcDEF";
DIGRAPHS_NamedDigraphs.("la(c)vyccurvelevel6") := ".xd@wsePOFRh{}VIerHKaNEmfJdu|bNwKyZJgrylMrVDVPDmDDbQKWPhBBLggHdL?KHYPKiVgCQ`ZrIBphKa_MfRx{mTLeQXC]L\\MVJlzE^oXsuVPftY\\emImaJ[IJEcWobQEEZPORIY?WQsaXSnOGdAv";
DIGRAPHS_NamedDigraphs.("la(c)vyccurvelevel7") := ".~?@nb_mIbWqUDXQYEHaXeX]UchIRcw}McXALbGmIdXQnJiynkJEqlrUtljYskz]wmZi{m{SHPG`Bqru}n{BAo\\PRslVSt|ZXuG\\Z`djOS[|PrSzJRK\\HWK`IWWKlQGGkPwSjQgOiQWChSw?gSiGbHQ]_GQOef@s^PHgZFcMGATx^zwwJaguKd`USe`aWeXeVdhIQcxMNbhEObWqJahUSjqymjzApkj[tlZYulJMvmJeynJnDASQGO{k|nj~?okFSS|JTtL^Uu\\aFUwXYsDFNS[tMqsrFQUBGQeEBJSaAJC]DIsiCIce@ITM?IDIaGqSfgACcHho\\FsQYEpxBa?d]V~";
DIGRAPHS_NamedDigraphs.("la(c)vyccurvelevel8") := ".~?BUaWOASNAK\\AsTbCVbSXbcZdwkd[mdktEcrckdcsec{fdCgcS`ccbdKhdSibs\\cC^dcjdslbCVasTbSXbcZaCNaSPjx\\jl]jt^kD`kThK\\cklelxlllmltilLkl\\gk|zMdrmttnDvl|omLqqX{n\\}nm@oUBoe?n|ynMbSU`sebsudtEf`qi`ihqa^qY^qq_qi_uQDOqpoyGpIIU]KPuJPmOQUNQM[RqwrY\\U}XRQuqyWUkKJP[`XXJ[IIxW_hSQSCIYP_xUPcEIiJ`HQPsGIIL_YIQCAPIN_HOSS?Hy`fGyG@KewwFxJfg}GPMfW{GHLghEH@IgXCGxHehZKxxep[L@yeaEPApeYDOype?qTyodwpTimysRA?Q`wPbgUakWa{YbK[b[nDcjdslegse[dckecsfc{gdCacKcc[hdKidS]bk_b{kd[mdkWa{UakYbK[b[O`{QaL^Jl\\jt]j|_kLalHbkddktnLllltmlThldjlDfnXsm\\umlwm|nmDpmURNdznt|oMAo]CoD~nTxsYasMcs]esmgs{MTSLTMSR}RR}USETSEqOiEUMFPEHPQrpaMpYLqAQpyPra]VEZRivrIYUuVRAt``YJcJJHZ`PVJCDIaQ_`RQKFIqK_pTP[HIQM`@PPkBPQO_QHP{@IAa_@NSKxFP?HcvF?~H[|FpAHszF`@HlDGpGHTBG`FHKtJXfNKuJ`gNSsOqGUKrOiFUKoEQnUCnEIlTv";
DIGRAPHS_NamedDigraphs.("tri-hex9,3symmetricconfigurationincidence") := ".Q___eCDcdcdk@EJaFJbGJ`HIbFIaGHq___eCDcdcdk@EJaFJbGJ`HIbFIaGH";
DIGRAPHS_NamedDigraphs.("9,3symmetricconfigurationlines") := ".He?K@PD?XGbIYPAeNeW?oDCS@caKhhCIW~";
DIGRAPHS_NamedDigraphs.("9,3symmetricconfigurationcliques") := ".H`?K?oCM?PC_COwbGSs`cIW}WOB?K@B_CPG@CMGqDLGXAeN";
DIGRAPHS_NamedDigraphs.("9,3symmetricconfigurationincidence") := ".Q___eCDcddck@EJaGJbFJ`FHbGIaHIq___eCDcddck@EJaGJbFJ`FHbGIaHI";
DIGRAPHS_NamedDigraphs.("bicorn") := ".G`A?gpHGcqepAC@PaQPHdL";
DIGRAPHS_NamedDigraphs.("udeltaadjacency") := ".MoI|AuPLIw@Qju@VgUqHhV?IT^";
DIGRAPHS_NamedDigraphs.("largeniel") := ".k_OGCSHCaQogaIGbAWoeMIbahCeTJGrAKsdMHRq\\?f\\IEihikZjez@qq[lUzcCyb_fc[G}c`gckLAd`GCeq@?_POcQHJAaGgaKHbAWwiMIcQXSkbKGrQSwdNHsA\\sgYieipmmZkFJHqtZmORiMA]PobyQEaQosIUC_Q^";
DIGRAPHS_NamedDigraphs.("specialc3") := ".G`A?PTHC`XCJaCG@DScQDcOn";
DIGRAPHS_NamedDigraphs.("dragoncurveblob9") := ".~?ADaOPa_RaoTbWYa{WbK_B{aCKtDgmeWseGqdwodWkdGicwgcgecWca[?ALAGLAgTFGdCGlBG`DGt@hTDg@BGt@G\\@H\\AGxIiPGHdFHDJILPixFghEg@EaO~_@XJSOJ@XJSMIPR`xV`pRI{LHxS`xOI{HI`T_`Q_hKK[EGCFHhMKkIIhU`XU``g`@PK{SFp]J{UKk^LS_IH_LkaHXacHIKHnah?JXkcgeFp]`GJIhUeGqeWs_gOH`W_wLHpN_OCK`t_WIIph_GKL@scWcJhb_oXLc@BXs_OWMkGAxl_WhDPhb_jD`~bglDp}bonE@{bOfD@|okQAKSA[UAkZBSVbCXc?^cO`eglDsrEcpESnECjDchDSfDCdCsbCcR_?PgP@gTAgxCg`DgXCGhEgLIgl?GXEgHBgHJgPFHTQH@KgxGhXPiLVG|DGt?GsQF{?JHYa@WJHY`pQI[NI{MIXV`hNIcNI@V`HSIkCISDH`b_p?_xLHpd`PTIsJIsKLCGIHfa_}Jp^apdbxic@PK@lcPJKS`HP`L{TG@ZLcdCo}JsH@XTIspESrEcDA@KJCF@hMH{A?`cMkB@PULK@@`gMcbC`\\K[EBHk_GZMcAB@t`?VLkBDGiLK[DWkN{\\DgmNs]DwoNcYCwgNn";
DIGRAPHS_NamedDigraphs.("cubicintegralg10") := ".S___fCDEccdeedaGLbGKbHJ`HL`IJaIKs___fCDEccdeedaGLbGKbHJ`HL`IJaIK";
DIGRAPHS_NamedDigraphs.("knuth7.2.1.6exampleforest") := ".NgN@jt`h[cja`fyBoY|WYVHIwgX";
DIGRAPHS_NamedDigraphs.("integraltreen=31(brouwers#17)") := ".^rCdefghrVWsTU`ABQiNOPjKLM_QXY[\\~rCdefghrVWsTU`ABQiNOPjKLM_QXY[\\";
DIGRAPHS_NamedDigraphs.("da1/4rer") := ".K`A?icjSsRPNKaHCRqCGAiQlRPLC{qGcPN";
DIGRAPHS_NamedDigraphs.("kochol74") := ".~?@I_GA?`OQD`gYIAWeMB?uOBwoOcokQa`?QapEQDWoRah[WeHYZe`qHg@s]f@yXEyOYgqQFGw[^HW_`HG__GgObbWyNFwsNfqqeIwUmKIkklQipjY{vkRQtLgQxlbmsLZgzkraqLzQ{_sDB_s@A_Ro~_bs}_cM@OcbI_GA?`OQD`gYIAWeMB?uOBwoOcokQa`?QapEQDWoRah[WeHYZe`qHg@s]f@yXEyOYgqQFGw[^HW_`HG__GgObbWyNFwsNfqqeIwUmKIkklQipjY{vkRQtLgQxlbmsLZgzkraqLzQ{_sDB_s@A_Ro~_bs}_cM@Ocb";
DIGRAPHS_NamedDigraphs.("vo_6^-(2)") := ".~?@?_G?@_?CA_?E??OQ??W?@@g?@_?CG_?E??Oi??_OEA?m??_OEAOi??_OFA?i??_OFAOm??_SEA?gN_?GD@_cJBg?A@O[GAou??_SFAOgK_?KC@__IBpCQCw?B@?WHAowOC`M??oOFA?kLC@CR_?KC@ocIB@?PCg?B@OWGAosMBpKV_?KD@_cIB?wNC`Y??oSFA?gKBO{PDW?B@O[HAooLB`?S_OGC@__IB?sMC@OZ_OGC@_cJB?sNCPSY_OGC@o_JB?wNC`WX_OGC@ocIBOwNCp[W_OGD@__JB@?PC`[WFwCA@OWHA_sOCPKUEPy@?_SFA?gMC@GRDPg\\_OGD@ocJBpCQCpOZFGCB@?WGAooRD@SUE@{`GaM@?oOEAOgLC`OTDpc]GAGb_OKC@o_IB`CSD`[YFQ?`GwCB@?[HAo{ODPWVEpo_GQI@?oSEA?gNC@OWEPg\\F`{bHwCB@OWHAowPDP_XEpo]FqGe_OKD@o_JBPGUE@gZF@s^GQU@?oSFAOgKCp[XE`k[FPw_HGGB@?SKBOwNE@cYEq?`GaKcHQWf__KC@P?PC`KSDPWVF@s]Fq_hIakk__KE@ooLC@CUDpgZF`{aGqOdIAeA?oWFB_{QCpOTE@c[FQ?`Ha[iIqyA?o_HBO{PCpOUE@g[Fa?aHQ[hIwGBA?cKB`?QDP[XEps^GQKcHa_iKGGBA_kLB`?RDPWWEps]GAKdHa_j__KIAooNCPGSDpcYF@{`GaOfIQgq`?SE@ooLC`KSDPgZF`{_GQWfIAcoKRGr`?SE@owNC@CUDp_XF@saGqOdIakoKRGrLGODA?cLBp?QDP[WE`o]GQKcHacjJa{qKwODA?cKB`CRD@WXEps^GAGdHq_iJa{qKrYC@OgJBOwPC`OVE@k\\FaCaHA[gIqwnKBEC@OgJB?{OCpSUEPg[Fq?bHQWhIawnKBCw`_[GAOsMCPGTD`cYF@{_GqOfIAkkJRGrMBeE@o_HB?{OCpOVE@k\\FaCaHQWhIaolKbKwMRiE@ogJBO{OC`OUEPk\\Fq?aHAWhIqolKBCuLwWFA_kKB`CRDP[WE`o]GQKdHq_iJAsoKRWvNG_HA_kKBPGRD`[WEPo\\GaKeHq_hJAsmJrOta?cIAowNC@CSDPgZF`{_GQOdIakkJQwnLBS}oGA??W?@?g?@_?CC_?E??OY??W?@AG?@_?CI_?GC@__J_?GC@_cI_?GC@o_I_?GC@ocJ_?GD@__IBw?A@OWHAoy??_SFA?kL_?GD@ocIBG?B@?WGA_{PC`M??oOEAOkMC@GR_?KC@o_JBP?PCw?B@?[HA_oOCPI??oSEA?kLB_{RDw?B@OWHA_oMBpGU_?KD@o_IB?sNCPU??oSFAOkKBOwODGCA@?WGA_oLB`?SEwCA@?WHAooLBpCTEgCA@?[GAooMBpGUEWCA@?[HA_sMBpKVEGCA@OWGAooOCPGVE@}@?_SEAOgLC@CRD`c]_OGD@o_IB`?QCpSYFWCA@O[HAo{PC`KSEpq@?oOEA?kKCpOTD`_^GQGb_OKC@_cIBPGSDP[XFa?aGwCB@?[GA_wPD@WVE`s_GQM@?oOFAOkNC@SUDpk[GACa_OKD@__IBp?SE@cYFPw^Gq]@?oSEAOkMCPSWEPk[F`{aHgCB@O[GAosQD`_YEpo\\FqCd_OKD@ocIB@KVEPgZF@s]GAQA?oODB?sMBp_XE`k_GQGbHASeHwGB@?SOCPGRD@SUDpo\\F`{gIQgjJGGB@_[KBP?PD`[YEpw^GaKcHQ_h__KE@owNC`KSDP_XF@s_GQWfIakm__KGAOsNCPKSD`_YF@w_GaSfIQmA?o_HB?wOC`SVEPk\\FqCbHAWgIbAA?ogJBOwOCpSUE@k\\Fa?bHQWgIwGBA_kKBpCQD@[XE`o^GQGcHqciKgOD@_[KBPGRD@SYEpw^GACeHq_hKBCqKwOD@_[MBp?PD`[WEPo\\GaKcHQgjKBCqKrQC@O_HBO{OC`SVE@g[FaCbHAWhIqwnKbMC@O_HB?wPCpOUEPk\\Fq?aHQ[gIawnKbKu`?SIAosMCPGSDp_ZFPw`GaOfIAkmJr?p`?SIAooNC@KTD`cYF@{_GqSeIQgmJr?pMGWFA?cLB`CQDPWXE`o^GAKcHq_jJAsqKr_x`_[GAOoNC@KSDp_ZFPw`GaSeIQgkJRGrMBcy`_[IAosNC@GSD`cZFP{_GaOeIQkkJR?pLb]E@ogJB?wPCpSVE@g[FaCbHQ[gIaolKBCuLrqGAOgJB?sQCpWVE@c[FQGbHa[gIQolJa{sLW_HA_kMBp?PD@SYEpw^GACcHQgjJAsmJrOtNn";
DIGRAPHS_NamedDigraphs.("two4-cyclescross-connectedat2opposingvertices") := ".Gg?K?qEQOqwg?K?qEQOq~";
DIGRAPHS_NamedDigraphs.("opticalillusion,filledavatar") := ".GcACWqEOk`cJaOGPbGXAqEOn";
DIGRAPHS_NamedDigraphs.("treerelatedtoquaternary3/2-power-freestrings") := ".~?@YdpYXEHeYexq^Fh}_gYIbhIUgHyahhywkixyqKJIpeHuyLJMVjY}mjKGpmXAPcxIToCM{OSI|Nb}SDSQKDBULLb]NMRmMMBiAQGCUQG@HSWWvPg\\@PwKwPW_HHggJIgODJ\\iVDhcWeXiZfH{]fyA`giMchY_fiIefjaqjfjGokjEWfZgskx]ljyykobExcHERchV?Ozp@Ojs}NxOTPGoSLWsuLw{xMwwwMgHG_PXG_CdP`b\\E`sDF_r`Da?cea_ki`?Sl";
DIGRAPHS_NamedDigraphs.("complete4-arytree,height2(3rows,21vertices)") := ".Tp?@ABOcDEFOgLMNOhIJKOtp?@ABOcDEFOgLMNOhIJKO";
DIGRAPHS_NamedDigraphs.("complete4-arytree,height3(4rows,85vertices)") := ".~?@ToSB?oKA??OGBOZOtLb\\@mBs}NsExMbk{OWOD@_\\AiAciIsIkKRGrOismJr@Aa?cIAsMKBOwNPH?PC`LCd@cYEsQTD`[WPHo\\F`|BgASeHsM`GaKcO|V@OKB?oG?@?_L@lBSuLsEwNRw~OZcyMrp@`?SE@sIgIQgjOiopKbLAjQwnKCIGAOgJOwoLB_|Cc@CQCsQSEPgZPHSUDp`Cf@s]FsM_HQWfOyCaGqPB";
DIGRAPHS_NamedDigraphs.("xf_3^2") := ".GeESqOAK`WAEOhkRBIXG@EOk@BGSv";
DIGRAPHS_NamedDigraphs.("x_{31}") := ".FkY_scbP^kY_scbP^";
DIGRAPHS_NamedDigraphs.("associativity,filledavatar") := ".H`A_wbcOk`pG^KGSFC[aDcMHB~";
DIGRAPHS_NamedDigraphs.("cospectralmatea1") := ".Q___eCDdfcGck@EJaFJbHJ`DIbIaGOq___eCDdfcGck@EJaFJbHJ`DIbIaGO";
DIGRAPHS_NamedDigraphs.("cospectralmatea2") := ".Q___eCDedcdc`FHaFGbGHbIJ`JaIOq___eCDedcdc`FHaFGbGHbIJ`JaIO";
DIGRAPHS_NamedDigraphs.("cospectralmateb1") := ".Q___eCDdfcGck@FJaEJbHJ`DIbIaGOq___eCDdfcGck@FJaEJbHJ`DIbIaGO";
DIGRAPHS_NamedDigraphs.("cospectralmateb2") := ".Q___dCddfeccaHIbGJ`GIbFHaJ`EOq___dCddfeccaHIbGJ`GIbFHaJ`EO";
DIGRAPHS_NamedDigraphs.("whythoffgamestates,starting3,2") := ".H`?KGbcKcPCJ?GqDL?OqDK^KG@`C[`caG`W@EOhgAEOhb~";
DIGRAPHS_NamedDigraphs.("97-cyclotomic") := ".~?@`_G?@_?E??OM??W?@@W?@?g?@?_KC_?CA?o[G_?CA@O[G_?CB@?WH_?CD@_[HA_m??OOD@__IAw?B@g?A@W?AA_wN_?KJB_}??oOJBp?P_?GFA_wOCW?C@_kMBp?PCg?D@ogMBp?PCpQ??oOFB@CQCpU??_OFBP?QCpQ?@OWGAOwN_?GEAOoNC`WW_?KDA?sMCp[W_?OD@_kLB`?RD@[WEg?A@?[HB?{OC`WVEW?B@?[GBOwPCpWVE`q?@OWFA_oNCPGTD`_X_?GDA?cIBO{ODP_XEg?B@__HAooMCPOWEPi@@_oMBp?SD`q@@?_QDp_XFaA@?_KGB_{OEps`GgCD@ooME`o\\FqCaGwCB@?SIC@GRFAGc_OKD@_cNCPs]GQM@?_WIBOwPCp_d_OGFAosNCPGXHwCC@ocJCpSYEqWgaOwQCpSUE@cZGaKfIWsNCPKSEPg\\FqOdHqaDB@?PC`[YEpweIAeFA_{RDp_YFa?`GaWh__OEB`CXF@s_Ha[gIQqC@o_ODPWWFP{gIQom_oSJB_{QE@g[GQOdIAuA@_kOCpcZF`{aGqOdIwWGA`?QD@g^GACeIQolJa}DAosPDPWZFq?aGqOiKW[GAPCSDPo]GAGcHQkpKwKC@_[SD`_ZFaSfIq{pLGGD@_sNDP[[FQKeHqgmKwGBAOkPCpWVFqCeJR?pKwKDA_oMC@SXGASgJAwnKG[IAooQD@[\\GAGbHAglKbKu`?cJB?{SDP_^HqciIq{tLbeCAOgLD`cZF@{`HaciJQwqKr]BAOoLC@[YFA?dHqkkLBSuMBiCA?gLB`OTDpw`GasoLB[wMwGD@o_HD@[WEq_kJr?sLR[wNWGBA?gKC`WYFaKfIaoqLRWxMbqEA?kKBPKUEPs`IAknKBCvMBs}_OWFB`?QD@[WEqCaGqOfIAgkJrGtMRy@?_OMBp?SDPc^GQGbHA_hIqwnKR_yNWCD@_oQCpWXE`w`GaOdHaciIqopKb|?OgCB@?_NCPKUE@s`GaKdHa[jJQ{tLrh?OWCB@?SHApGSDP[aHQWhIb?uLr_xMbs}OcM@?oSEA_sODPW^GQSeHq{qKrSuLr_zNCDCPWCA?o[GAO{SEa?`GqWgIQklKbOtNBw~OcM@?_SFA_oPCp[_GaOdHq_kJQwrLr_xNcDC_OGCA?gJBPCZFaGbHq_hJAwpKrOzNR|A_OWFAOkKBOwTFACcHq_hIb?sLbg{NS@@`OcMCPKWE`k[FQOdHa[hIakkJb?sMrxBPSeFBPKTD`[XE`s_Ga_hIaklJrKuMBc{OCHBPSXFaOsNCPGVE@c]FqGeHq_iIqolLrgzNBt@PC`I__KEB`[WEPg]GAGfJQwoKRSuLr_{NR|APSXHRGGEA?wPCpOTDa?`HqgmKbKsLb[|Nb|?Os\\GQgKFApCQDPWYF`{cIAckJr?pKbKsLR[~OSXHQcyC@O_NDP_YF@{_GaOdIQsmJr?qKrOuMcPGQc|O__kKBp?RDpgZFACbIQolKBCuLrg{Ns@CQShMSDEIAosNC@CQD@o_GQSfIAklKBGrLBczNCXFQSlKa?cKC@WWEPk[GACaGqgkJrKsMBk{Nc@@PCXHQspQSwGB@?SNC`WXEpobHQ_iJb?rLRWwMrw~Ps`NSWGE@ocSDpc[FP{cHakmKRGsLRWvMRk}OSTEQsxO`?[IB?wUE@o\\FaCcHqsmJr?tMRgzNR|AQSlMSwOEA?sOC`[[FPwaHQWkJa{pKbOuNBt?OshLRtEB@ocIB`?RDPk]GqSeIQglKROtMBk{NS\\IRTTW`?SKBPOUDpgZFqCcIqorLR[yMro|Nb|AOs\\GRS|TTwKHAooMC@C\\Fq?bHAWkJa{pKr[wMRg{Oc`LTt`XUgSGA_oNCPOTFPwbHakkLBWwMRg|Nb|?PCTHQtHSTgKGA_sMC`cYFP{bHA_iIr?qMBczNR|?OShLSDTUUTqD@_[JBO{WEps]GqWgJQ{oKRKtLbc}OC\\KStTVUDdZ`?WIAooRD@cZGAShIqsmKRGtMBcyMs@CQDDQTTXXVDuAA?cIAp?QCp_^HQ[iJr?pKb[xMbw~OCLDQs|RTt`ZVmE?_?E??W?@?w?@_?CD_?CA_?CA?oQ??OGB@oa??OGD@oa??OKC@_e??OSE@ocIAw?@@?SEA?gJ_?KE_?GD_?GIB_}??okMBw?B@?kNC@E??_[IB`?P_?OEAowNC@CQ_?SFA_wNC@CRDG?B@?[KCPGRDW?A@?[LC@GRDG?D@__HB_}??_WHB?{QD`a??oSGBOwRDpa?@?SEAosMC@KSDp_Y_?GC@ocKBp?QD`[X_?KC@o_LB`CRD`[YFG?D@_[IB?{PC`SUE@e??_SGAOgLBp?TE@cY_?KEA?cJB?wPD@_XEgCEB?wNC@OUFGCCA@GVE@c]GGCA?o_MBp?ZFQCa_OSFB?wYF@s^GQGb_OKC@OgOC`K[GaQ@?oSEAO{PFPw`GwCA@_gLB`CREAU@?_[JBO{PC`cf_OOFAOkRDPgZHaaHB`GRDPWWEPkaGq[hbO{PCpOXE`s^HASfIGSKC@CQDpgZFaWgIW[IBpKVE@g]GACaHaeA@?WMCPc[FQ?eHq_hJGOFA@?TD`_\\Fq_hJAyB@OkMBpGWE`o`HASgJWGEAp?REPk]FqGbHASj`__IC@GSE`{_GQWhJAsmJwSJBPCTD`k^GAGbHAgp`o_HCPOTF@w_GaOdIrCr_oOE@pOUE@k]HQ[jJrCs__SEBO{TDpo\\GqWfIawr__KHApCRD`[^GQWlKBCr_oSIB?wODPc_HQ_kJa{o`ogJB@GSDps_GaKcIasqKrYCAOkKBpOTE@{fIQgjJrSuMWOHA_sUEPk[FqCeIQglJbGrLwKHB?sODpg[GASfIqosLRWwMgOGA_sMD@SVFaCaJR?sLr_z__SFA?cSDp_ZIAonKBOtLr_|__KGA_oQD`g]Gq[iJBGtLbcyNGWGAooLCpWXFQCgIq{oKR[wNRy@@_[MC@GSDp_ZGQGbHA[gIaonKbSxNgCA@?wNC@OTEP{`GaKcIAcjJa{pMBg|_OSEB@GRD`cYFaCaHASeIQgjJBCqNs@A_OKCA?{PCpWWFQCaGqSeHqklJrSvMc@@_OKC@OcJC`OTDqGdHaciKBWvMBcyNRxAOwCB@OWIBP?TD`{`HQWfJrGrLRWvMBk{OSPD_OGB@o_HBpOYGACbHa_hIqsqLBS{Nb|AOwCA@O[IB@CRDq?aHASfIAolJbKvMBc}OSQ@?_OGA_kLCPk]GaKfIAckJbCrLBk|NsI@@_[HAooLB`S[GQOfIAciKBOuMbo|OCEDAOwPCp_YEpo\\HASeHqciIqomKBOzNcLDQW[LCpSUDpcYFQ?aIAciIqsnKrWwMRp?OcLDPc]HBO{PC`[WEPw^GaWfIAgjJAsvMbk{NSDCQCiA?oWMDp_XE`w_Ga[lJb?pLRWvMBo|NsHDPcdK__WGB`CRD@SUGACfIawqKrOuLrs}Ns@BPs`I_o[JCPGTD`g]FqOgIQonKBCqKrOtLr|@PcdIRgODA?{TE@g[Fq?aHAShJQwnKBGrLBWyPC`IRtAAAooNC@KVE`k[GQKhJAsoKRWvMbo~OCPHQcxOSWgJBO{OCPGSFA?`HQ[gIqsoKbKsMRk{Pc\\HQsqGAOoOD`_XEpo_GQGbIaonKrOwMro}OCDCPcdJRDHR__KC@O{QD`cZFAKdIAgmKBKtLb_zNb|FQC|P__WFAPOVEPo\\FqOeIqwpKbOtLb[xMrx@PSXJRdAC@ogKB`WWF@s]GQOfJQwnKBSxMbk|NsHHQsxR`?WGBP?QDpo\\FaGdHaomJrCqLBW{NS@BQctNSWKFAOgMC@KTEpwbHQWhIaspLBSwMro|PshLTTaC@OoLD@WVE`k^GQOjJBKtLrgzNBs}NsHBPs`LRtTV_ocJB?wOCPs^GAKcHaomJrCrLr_xMbpAQCtVUDdY`O_IB?{PD@S\\FaKeIqosLb_xMbs}Ns@CPSdJSdPU_o_IBOwQEPg\\FqKcIAgjKBGwMRk|Ns@@QctOTTXXVGSE@okLBp_ZFPwbHa_lJr?pKrSuMRx?PspRTT\\WUTmC@_gJB@KSEPk_HQcjJQwpKbSwMRgzOCPGSTHTTdd[VWGGAOgJC@GRE@{dHqgnKBCqLrcyNb|?OsTJRtLVUDl]";
DIGRAPHS_NamedDigraphs.("fostercage") := ".]_____iBcd`aKLdIJL`HI`GJbELcFLaFIPaEJObGKRTcHKQSdEHPRdFGOQaGHN`EFKYbFHJMcEGIM}_____iBcd`aKLdIJL`HI`GJbELcFLaFIPaEJObGKRTcHKQSdEHPRdFGOQaGHN`EFKYbFHJMcEGIM";
DIGRAPHS_NamedDigraphs.("(2,1)-generalizeddodecagon") := ".~?A|_C??K?_?B_oD`?F`WH`SH`SK`[L`sO`{QaKSa[UakWa{XbSZbc]bk^cCacKcCsbCkiDKhdSkd[ldsneCpeSGEcFE[wE{vfCyfKzfc|fs~gCEGSDGKD?sF@@DdGicofcggHCH@[I@`I`GL`OM`_N`WOaOTaGUa_VaWWb__bW^bO]bG\\c_fcWgcOdcGedOkIShDXRdooHklDxKeOsJKpEXW_wuI{GEhU`oQ`gPa?R`wSbo`Kk\\CPcc?cKs^CXfd_ldWme_teWuaoXagYb?ZMKVB`oe?qdwpewwG{wFPOewxIKyFXnfG{LszFhNf_}Hs}G@tfg~Md?GPYfx@J\\AGXlgHCLcDG`T_pBIdDGtIHYDgxuh@HO|LJtKJ|NN\\MNdXKAIj@`PLZOAKjP~P\\SOdTO\\VKTUK\\PNAQi@vQLRJiRiP[QdmNTnNLfLYYkpiRLlOLkOTcLI]khgRlsNttNlpMQam@rSMDOuFPAdsIcsQbS}IPmHPuJQEKP}QQiiqIUTMSRAjqYVTeYR]XRe]Ryqri_UK@Sie_IfTAt_QpUcAUQrU{CTQl_YhTsBTaoVSCTYnVM|_C??K?_?B_oD`?F`WH`SH`SK`[L`sO`{QaKSa[UakWa{XbSZbc]bk^cCacKcCsbCkiDKhdSkd[ldsneCpeSGEcFE[wE{vfCyfKzfc|fs~gCEGSDGKD?sF@@DdGicofcggHCH@[I@`I`GL`OM`_N`WOaOTaGUa_VaWWb__bW^bO]bG\\c_fcWgcOdcGedOkIShDXRdooHklDxKeOsJKpEXW_wuI{GEhU`oQ`gPa?R`wSbo`Kk\\CPcc?cKs^CXfd_ldWme_teWuaoXagYb?ZMKVB`oe?qdwpewwG{wFPOewxIKyFXnfG{LszFhNf_}Hs}G@tfg~Md?GPYfx@J\\AGXlgHCLcDG`T_pBIdDGtIHYDgxuh@HO|LJtKJ|NN\\MNdXKAIj@`PLZOAKjP~P\\SOdTO\\VKTUK\\PNAQi@vQLRJiRiP[QdmNTnNLfLYYkpiRLlOLkOTcLI]khgRlsNttNlpMQam@rSMDOuFPAdsIcsQbS}IPmHPuJQEKP}QQiiqIUTMSRAjqYVTeYR]XRe]Ryqri_UK@Sie_IfTAt_QpUcAUQrU{CTQl_YhTsBTaoVSCTYnVN";
DIGRAPHS_NamedDigraphs.("(3,9)-generalizedquadrangle") := ".~?@o_G?@_?CA_GA?_GA?_GA?_?Q?@?u?Ag?H_?q?Aw?D_?Y?@w?G_?kQ_?oP_?gN_?cO_?SR_?_U_?WS_?[T_OSLCPGSDPcYFGCEBO{QCpWWE`y@@osOCPKUDpc\\_O_LBp?SDP[WEwCIB`CSD`[YEpy@AOwQDPWWEPk\\_OkMC@KSE@c[FgCKB_{RDP[YF@u@@?{OCPGZF@s]_OSEA_oLC@[[FaQ@@O[HAosNE@o\\GwCEA?cJBPCXEpwe_O[GA_oLC`gZFQU@AOgJB?sRD@SUHwCC@OkKD`cYFPwa_OOGAOgRDp_\\F`}@@?WIApSWE`k[GWCC@ocKD@[XEpo__OSE@o_MDp_XEa[k_OSGAOoMBpGSFaCn_OSGA_kMC@CTFQ?o_OWFAooMBp?UEp{m_OWFAOgMCPGRFAGl_OOD@_cNCPSUDqSj_OOD@ogOC`OUEAWi_OOEA?oOC`KTEQKh_OOFA?kNCPKSEaOg__OD@_[GAOgJB@{_GQGbHASeHwKKBP?QD@WXEpweIQwnKRKtLbcy_okLBpCTD`gZFQSgJb?pKbSvMBiBA_sOC`KTE@o\\GqglKBCqLBWxMgKHBO{PCpOVF@wcIqsnKRKsLr_y_oORD@SUDp_XEa[gIQgjKbKsLRiB@OwNC@WVE@s]FqgjJA{oLR_xMgKGB`CQCpcYFPwaIAckJr?sLb[y_oWMC@CTDpcZFA?hIqolJbGvMRiB@owNC`OWE`k[GQ_iJAsmKrWwMgGLDp_XE`k[FPwkKbKsLRWvMBcy__wNC@CQCpOTDa_hIaklJa{oKRhC_oSGA_kLD@_YFaCcHa[oLBSuMCHD_oSGAOoLDP[XFQ?bHQ[nLBSvMSLD_oWFAooLCpcYFAGbHA[mKbKuLs@D_oWFAOgLD`[WEp{dHa[lKbKwMSDD_oSE@o_LBp?PCaKcHQWkJQwnKB|C_ocIAooMEpo\\F`{_GQGpLb[wMR|DQgKC@OWHC`_XF@w`GaKeIqorLBc{PWKC@O[ICP[YF@s_GaOdIaoqLB_zPWKC@__KBp[YEpw^GQOdIQorLR[|PWKC@o_JC@_XEps^GAKeIAoqLRW}PWKD@_gKB`GTD`g`GaSfIQgmKBc}PC}B@O[HAowPD@WXGAGeHq_jJa{wNSPM_oWGAOkMBpKTE@{`Gq[gIqsoLrlCRWKFA?gKB`?RD@[^GAOfIQglJrW{PCqB@?SJB?{OD@S[GACbHAgjJbCtOSPH_oOEA_kOCPKUF`{aHAWhIr?pKcLCPwKC@ocKBpGRD`s^GaKdIAgnKRLAPCYB@?_HA`CQD@SZGACdHa_hJRCsOCPG__SHA_sPC`W\\FaGdHa{oKROwMR|AOs`MRtHRTGGGAooLBp?RFPw^GqOnKBCtLb[~OcLHRCtOST]A@_cKBO{QDPk[GQKdJQwpKr[xNs@@PctNSTLT__[IAosOCPOZFA?cHasmKRGuMB|?OS\\KRd@QTgGD@_kLC@SUE@cbHa[jJb?qLRc}OSLJRSxRTd]A@O[KBO{SD`[YHASfIawnKrSwNSDAQspNSdTV__WGA_sPCpSVEaOdHqclKBGsLrl?OslKRtDSTgGFA?cLC`KSE@cbHa[gJQ{rLBW{OCHJRSxOTDUA@OcIB`OTDp_[GACfIaklLB_xMrp@QChMRtTUUWGGAooMD@SXE`k_GQ[gIQwtLb[|Nc@HQcpLTTXW__WHB?wRD`[XF`{aHqcjJrKvMRo|OsXIRS|STtmA@ogJB`KUE@g\\FqGfIAgoKbWwMrxAPshKRdPVUgGCA_oQCpOYF@w`GaOhIbCrLBW{OCHFQShNSTH[__OHApCRDPc[FQ?aGq_jKRGsLrl?OsXHQcxOStuA@?cJBpOUE@k]FqCeIAkpKrSwNSDAPs`IRT@RVgGCA_oODPWVEps^GAShIbCqLRc}OSLEQChKSTH^__SEAowNCPg[FaCaHAkkJb?vMBk|Ns\\HRtLUTt|f__SFB?wOC`c[FQ?aGqgkJa{uMRo}NsXHRdHTTtxe__WGA_wOC`_ZF`{`HackJR?uMRo}Ns\\GRTDSTdtd__[GAOwNCP[ZFP{_HQ_kJQ{vMBk|NsXGRD@STTpc__ODA@?PD@[XFa?cHacjJA{sLRo|Os`HQt@QTdhb__ODA?{QDP_YFQCbHQ_iJB?sLRk}Oc`HQtDRTTla__OE@pCQD`cYEqGdHa_hJAwqKrs}OCXFQtHRTDd___OE@o{OCp[WF@{bHAgjJAsqKrk{OSXFQt@PTt``{GA??W?@?gA?_GA?_GA?_G?C_?OL_?i?AW?K_?m?@W?E_?]?AG?JCg?KCW?IBw?HCG?DCw?GDg?EDG?FDWCDBPCQD@SXE`q@@_sNC`KUE@g]_O[LC@CRD`[XFWCGBO{OD@SVE@m@A_wPD@WVE`k]_OcMC`SUE@cZFWCJB`?RD@_XF@y@B?wNCpSVE`o\\_OONC@CQEpo\\FgCD@_gKBP?VF@wc_OSFAOkLBp_[FQM@@__HAosPEPk]HgCFA?gKBPGYEpsd_OcIAooLCpOTDa]@@?SJB@WXE`s]GgCCA?cICp[WFPw^_OOEA_kTE@gZFAE@@?[HB@OVEPk[GGCD@_[GB`[WEPgfJGCDA?cKB_{QD@w`JwCDA?gJB`?PDPs_KGCE@okKB_{OD`k^JgCE@ocIB`CQCpoaJWCC@OWHBpCTD`[dIwCC@O[IC@GSD`_eIgCC@__KC@GRDPcbIWCC@o_JBpCRD@gcIGGC@OWFA?cIAoo^GACaGqOdHa]BB?sOC`OUEPk]HacmJrCrLRWxMgKJBO{PDPWYEpsdIAwoKRGtLr_y_ogLC@GRDP_[FQKiJR?pKbOuMRiBAOsNCPKSDpo]HAklJrCrLB[wMgKCCpOTD`[WEPgfIAciIrGrLBSy_oSMBp?UDp_\\F`{iIqonKBSwMRiBA?wPC`KXE`s]Ga_hJA{oLBWvMgKEB`?PDP[XEpo_IQkkJQwqLrcy_o[MBpGSE@gZFACgIaolJbKuMBiABP[WEPgZF@s]JBGrLBSuLr_xMgGMBp?PC`KSDPWgIQgjJQwnKBCyPGKDA?gJBPOWE`w`HAWfKBOtLb`APWKDA?cKBPSVEPs_GqSfJrOtLrdBPWKE@okKBPKXE`oaGqOfJbGrLb\\?PWKE@ocIBPWVE@k^HQWfJRGrMBd@PWKD@_[GBO{OCPGbHASeJAsmJr?~PGKHA_kKB`k[FPw^GACaKRWvMBc~PSiB@?SEAPGWEPo]GQGbHakkKrOxNCUB@?SFA`CVE`o\\GAGcHQgkKbOwMsUB@?WGB?{VE`k]FqCcHQckKrSvNSUB@?[GAp?WEPk\\Fq?bHa_kKbSuNcUB@OWIB?wQDPWYGQGdHqciJb?xNcPN_oSFAOkMCPOUEQ?aHa[gIqwnMBtCRgKEA?cJB_{RDP_^GQKfIAklKB[zPCuB@o_IB?wOCpOVFq?cHqciJQ{uNCPK_oODAooNC@OTFA?`GqOiIqwpLSDCQWKC@_gJC@CRD`w^GaOeIQkoKRHBPC]B@?[HB?{QCpW\\FqGbHQ_iJrCrOcPE_oOGAOgPC`OTEq?`HQWgIQspLC@CQGGDAOgLCPGUFPwaHQWnKBCsMBc~OcLGRc|QStQAA?kKBO{OCps]FqKcJr?pLRWvNsHBQSpLSDDV__WHB?sNC`SZFACbHQsmKRKvMR|?OSXLRtDRTWGFA_kLC@CSEpo_HAWlJbCqLb_~OCDFRCxOSdYA@OWJBP?TD`_XGqWfIqwoKbSxNcDBQstMStXV__SFB?sND@WVEaOdHqgmJrKtMBt@OclKRtHTTwGEA?gLCPKTDpgcHQ[hJR?qLB[zOCLJRC|PTDYA@o_HBPGRD@_XGqWfIAsnKrOuNC@AQstMSDPT__SHA_wSDP[WFA?`HqgjJROwMRk{OS`IRc|TTdeAA?kKB`OTEPgZGACfIAcmLRWvNRx?QShKRTTUUGGEAOoMCpWVEPw^Ga[hIq{rLrc{NSLEQctNTD\\Z__[IAowRD`_YFP{aHq_iKBGuMBk}Oc\\IRCxSTtiA@?gKC`KSE`o]GQGcIQgpKrOuNC@APsdIRtDQVGGCAOkPCpSXF@s_GaKgIrCqLB[zOCLEQShMSDL\\__OHAo{SD`_ZF`{`Ha_jKRKtMBt@Oc\\GQctOStyA@?gKC@SUDpk\\Fq?dIQgpKbSxNcDBPc`IRDDQVwGD@_kMBpCYF@w`GaOjJAwoLr_zNR|FQS|RTd\\^XwGD@ooMC@GXF@s_GaKiJAwnLbc{Nb|EQSxQTT\\]XgGEA?gMC@GWEpw^GQWhJAsoLbc{Nb|FQCtPTDX\\XWGFA?cMBpCVEps^GASgJAsnLr_zNR|EQCpOTDT[XGGC@O_OCPOVEPw_HAWhIqonLBS{NSLGQSlOSdXYWwGC@O_NC`SWE`s`GqSgIaooLBSzNcHGQSlPStTZWgGC@_[PC`WXE`kaHQWgIQomKbK|Nc@EPslQStPXWGGC@_[NC@KVE@o^GqOiIqolKbKzNCDEPslOST\\WW^";
DIGRAPHS_NamedDigraphs.("hundredtwentycell") := ".~?HW_@?A?C?GE_D_L?]@CEoCoI_X?mA{EgKOX`OA]C{HwRob`HAUCsJGUOk`ZAyDkJWVom`\\A}D{IGSOi@xAkFkIGS`?`RAeDOOgTojAFAgGkK?]Ow@xBCFgQGW?}`cC[H[K_aop@}CuEgOGY@?ARBOGKLO_or@yC}EWNwZ@BAdB[G{Loaow@|BcG_UG\\@EAnBWGoTG\\`CAlBcGOSW\\@B`uCGJsFwNo`_~AICCGGOOa`DAECKQ?oPHBAEqHOWGh`bB\\CkKSU?t@oa]EMHoXGi@hBfD?KG[gh@daaEWMsT_tpXBTDoLW]Wm`eamEaHgX?}P^BMFmJoXwl`hasE_OCR?p@waiESN{``AQDcSGaPKbPEQNC\\BAFCKWXOs`jBMEsLw[oz`tEoO_cGuaCCbEwOgcgwAEC`GGPWcw}aHCfFKOwcgxAGCjFSP?dGvaDClFOOwdg}AHCpFWPOdWzaIChF_PWeG{aKCrFkPgeW~ALCtGCPwex?ANCnFCOocW~aMCvGKPodw|AKCtAKQ[GpGObCdAOQkHPGodCrAOQsH@JoeCpAWRSHPIOeCvEAKKWgppcbTEaKsXwqphbVHoWCfP_q]E@H{WShP`qbEFIGWch`dQoEJIkWkh@_qgERI{XSi`dqfELIoW{jPcQhENJCXCgPaQ_EDJGXKkpdqmELJSXcm@equE[MQUgrx^BUFJJOZ{l@oBfdkLG\\KlpgqyE_MqVOsx[BSdqLS\\{lpjqzFANEVWu@vQ{EtJ{[W|H[bZdyLqVwv`zq{ErJs[O}X^B\\Fl@mB[EwMO[_x@uBsFgNO\\_z@[WCDp`oVEB@cXSE@`OWEJ@cWcE`boYEP@gXKE@doXEL@COcD`AoPCL@WO{D@DOSCP@GP[C`CoRCZ@KPcDPFoTC\\KoacrQIRMGjK{astaKrPGrLGa{vqLr_GtL?aksqJrSGlLSbkwQNrXGtLgb[vaNrZGxLobcvQKRWG|MKbstqKRaGzMO_sxP~ReGLM[_xMH|b|fkNu\\?~a]RkF~Ms^{yq@CyfQOI\\P@q`RsGFNS_[za?D?f]NofK{@}C|fcOE]H?Q^rrG?Se^@@AarvGHNg_hRX{cDIHO_cDAaOdOgOQI``GyDCcgQQSidBaRSKHPPKelCqVsHHQTI`XIajSPHZPCetBQTdRgYQqaPLqnsMHMTeaPJajsNHbP?ePWyFcrIzP?e@VIDcsIp@EAKDgIOU_d@IA[DGIoR_j?}@sCGGOM_\\?}ACBWEoL_^Q?f@YIOcxJRQGfPYiPczJPQkgxYYQC}JVQcg@ZIYDJJbQWgHYiRdAJZQ_gXZYVdCJ\\QwhxZiTDEJXQoh@ZyWc|J^RWhP[IQc~J`QshH[YZdMJdRSgh[yWDKJfRGhh[iXc{JVT?nDSQ{TQJqXaiX\\InDsigVSrLTQytYJqXykx\\ijDzimVgr\\Wa|TWJlTcmxhIldzL@T{nxfIoD~LFUKntVA}t\\JwYikH\\rOtcJ|USn`iYrD|LZOY^t@x}R|f~OA_T?y?sDgHW?r@kJ?eXLbWGrPkJ@eZLdWOr`kzFEfLfX[tplJFehLhWSrhkZGe]LjWWrxlZEeeLlW[s@ljIEcLpWct`lzCE`LnWgsPkjEEbLtWkshmJHEiLrX?tXmZHegLtXStxmzJElLvUalHtItTjjW\\YmL[rktlj[\\ml|[bmtoMnVanL\\Iyt}jzVmmt\\y|T~jxZoy`wJ]fTN@ZwypwZdfUNLZ{y`wj_FTND[OyxwzgF^NV[SzxwzbFWNB[[z@xjcFXNN[czPxzeFZNR[gzXyJfF[NP[sz`yZ`f\\NH[Czhyj`FVNV[{zHxZgf]NT\\KzpxJhFYNJ?ouCARWoKLn?cu[ARYOILr?gukAbZOJLp?ouSAr\\oJLt]p?d{CEgOnY_U}I@sHV}ON]x?dzsBGTn}_a~iACIVqOI`^?QBjzgJni_i}YAZyGIOz^X@YFJ{GKoE_xCD~CFvxOZ^h?qG{?GHOv_D@yEj|gKPD?W{C@rpOFNT?_{s@RtoDNH?S{kABroENB?[|KABsOENF`@DEDSWGsoaafAQIKHghogahFEESXgvomarBqK{MgkokanCaKc^W^PV`pEANKQgmPz`lDyOKKGlPv`hDmGSVHDpMbJCQK[OGnQNaXDQQKPgpPl?DEaLg@GvPocq?QMSaOAPsbm?YNC]hN_GcXHK@CahIQ\\?PGyQG?gtP|?LGAOSiO@P}clI[?s`HBQbda?C?G?O?_Y?U?s@wCOZ?R?i@cAwJoY_p@eD?HwRof`NAMCcHWROk`XAqDkJgUol`^AyDsJwVog`PAgFcIo]og`QCADKIWT@A`VAkG[I_aoo@xB_FcKO]`G`_ByEOPodoqAJBCFwRWY`?`gC?HKL?_otABBKFgRwX_~`kCKISLobovAJB_FsMOa@W`sCWI{L_b@S`uCOIsMO`@P`sCMFWO_nO^_~AEB{GgOO_`@AICSGWOpGB@CcKGZGd@_aeEKLsQopPWBSFAHwWwf@cagEcM[S?o`qacEUIGX_zPUBVDcLSV?t`xayEYIwYGe`cBxD{Kw]wn@fauEeJOY@?PKBCFaIgXO~qECHGUPOaHCqLCXG{PsKG[Op`dBQEkKwZOv`rBmFSZ@AAObYGOQKZ`AaQb_GWQC_`DaRbyGcQ[[pBaQbcG_Qk\\PCASb]GSQs\\@BaUbwGcRC\\`DATbmGgQc]@DaWbqGoRK]pEaXb{GsRS_PFaZc?G{Q{[PBAPb}GwR[_pFAVbsGoRSGpHobC`AKQSH@IodCbASRKH@JOcCnAWRCH`LOdChAWR[WGopabFEQLSYGrPfbJEeL[f@_Q\\EBHwWCfp`QdEFIKW[g`aQeETJ?WkipaqcEBI_XKjpdQiEVI[Wsj@bqlEPIcW{kPcQ`EHI?WSk`cqrEVIwWslPeQwEZJWXoxHYbNd{LW[kl@nqsF?M]Uos`sqvEbJgY?zH\\BRdoLQVGtPvqvEnJk[G{X\\bWF\\JoZSnppbsdqLmVgvH^b]FnJoZKnPpBxd{Ls]sEwLoZ_x@qBcFWNO]_|@uBkDp_OVEF@[WKEPdOWED@_WkEPaOYEN@gXCE`coWEV@cWsCPAOUCJ@COsD`BoSCT@OPCC`DoQCR@KPkCpEOTC^@SPsrAIRLGhKwakrqJRUGrLCbKsaJr^GvM?bSsAIrRGnLOastQMr`G~LcbSuaLr]G~LkbcvAMR\\GpL_bswqNRVGpMGbkxABRdF|MW_sxqBcwfuNu]o~XsB}HxMo^{zP~rjGCRi\\H?htCFIDNO_[|Q@rmG?SA\\w~A[roFwRu]P?Xwc@H~NK_@QX{CCIJN[_c}aAdLfqOSgdAAOSIHATA`@GiECbgSQQ`HHQiSMHLOodDCqYsRH^OcdHSiDciIlPCdlCQZSLHUTM`hJIHCvI~OwcxUYHCmInO{eLCAXDbg]RKjlCAWD[gUROjCCWGoU_h@YASCgHoS_j@MAkBwFOO_`?y@sBwGOL_Z?u@|GA[DghARclLGa\\DihERklDIqbdhhGRwl\\IQ_DkhgSkmLHa_dihMSGllIA`dlh]SOltJafdmhSSWldJAcDnhaRsl|LadDohIR{mDJQcdphmSwmTLQadrh_Som\\KaedqheRol\\SA{TPJpTGnHeIhdsi{VQi`\\RKtTJjTgnHfirduikVmix]bLtaJtT_mtUQzeciuVksDVq~e[j?V{s\\Wq~T[JzTsn`iiodvLBUOntXQ}EhjKVstl@h~SFfxNu^|?I@SBgBOU_d_BKEokAXcuL_bLEokEXkuT`BMErk[Y[u\\drVEsk]Ycud`RMepkaXwul`bNetkYYWut`rOEukgYOvDaRUEvkOYCu|abPEqkWYKvTarQewkcYgvLcBTexkeY_vTdRVezkkYsv\\YIsfSjTUml`tiwtrMrUulpuyvtqMzV?y|]I{tsjjVynl]yzTvjtV}ndnBiF_ly\\S{DnbjF`mU\\W{tnriFam?\\S{TpBjfbm_\\{|\\pRnfbmK\\_{LprkFemO\\c{|qRlFfmW\\k|Lqblfgm[\\o|DrRmFhmE\\s{doRmfimC\\[|\\rrkfdma\\w|TsrnFcmc\\g{kBBWOHLb?ou{ARXoHLh?gvKAbYoILl?kvCBBXOJLv?kvTzCAVoOY`A|i@ZwgFOd^x?|zcAVnOK`U~yAJ}gGOh^H?iD{@GMnm_m}iAjxgHng_hBl|cDg[no_r?YBcOV{O^^d@l}cBGbo?_dB]?SFgYnu_pCS@boOFND?[|SABrODNV?S{c@RqoGNN?W{K@rsoGNP?W{]CCSWTP_bRAII[HGgoeafAaIc[WXPeb^AyJKNGroyarAqI{QGqP|`|D]FCWG{pIaxFmEsVh?ooatF]EcUw`P[cVCyKkPGpp?a|G}HcTHGpEbDEs?SYGu_Cb\\FARG@GxQH?HFQMw@g{Pyc}?aPccoCQIchHs@CbhG_AbTFs?s_H@Qh?DFyQshoBQCcLIN";
DIGRAPHS_NamedDigraphs.("110-iofinovaivanov") := ".~?@m_GA?d?YC`WUC`GUE`w]F`g_SapeLEgo[a@uNFH?TaPUREwgUc`YRFgw]bp}QEGcYa`eMDHC\\cP}ODwkVbPaKEwaRchEOaweNaguKbjcymrqwNj_{nR~@O{@AncM~OJk|mSEyOi?`gaMiKYSehqahKiosjQynKykolR]cLkPOpTFKU{lYrTJMS{hUpd^GTK\\XrtbHTWD[VWD]VwH_WWLcXwLdXgHaW}y?_GAS@gQD`WQC`WYF`w]Ea@QJEWsYb@qGFW{[c@UHDXKZa`YQDhK]b`yNFxGWaPiIEWwScPuPFx?Vap]LEGoZaHMQcXAJaW}IbWqMmRizNJ_}mBq|N{DBoCI}Oz|?mruxOZhAgAEaGygphQYfIIcqjBQlJi{rirAtLyOupDBDS[pZqtjLSkxRqdZET{`SptfNUKdT_Tp\\_Tx^_e@`_uPf_uTe_eHb";
DIGRAPHS_NamedDigraphs.("4-largestcubicnonplanardiameter") := ".e_OGCSHFBPwsQJGfBXkeVKdPpkORJFqA[WbIFRaS{`OHsQSUW]_dRSCkV`Er{Kw\\`dBsGoXrOGCAICb`g{YHDcR`kuRJeQgwuGHdbx@MKPdBhpI]OgCyHIJKNOQhiAUJobX}E[Moq`yCWK~";
DIGRAPHS_NamedDigraphs.("(4,4)-latticecomplement") := ".O__@_@_@A_@B_AB_ADE_BCE_CDFG`CDEFG`ADEGHI`BCEFHI`ABFGHJKaBCDEHJKaBCDFGILMo__@_@_@A_@B_AB_ADE_BCE_CDFG`CDEFG`ADEGHI`BCEFHI`ABFGHJKaBCDEHJKaBCDFGILM";
DIGRAPHS_NamedDigraphs.("moores") := ".~?DT_A@?B_I@oE_gAWG_sBGN?yCOO`KCgT@QDoU`cEGZ@iFO[a?Fw]aKGgfAYHGdbWIgjaqJWma}KGpbILWsbMP`Cc_PxDcMR@AcmU`OdESxQdQTXMdeThVdaRxgE]Y`heYZ`kf?Zxlgg[xvfq]hufI\\Htfe]Hzfu_YAgM`YEfyaYCg]^yGgAR`_F[axODg]aLdGVPqG}T@]FOcHTD{\\QPdKV@uGyT`aFcchXEC]ARd[WpzHQSPZFobHWEO^QTc{XPrGi_aBH[eIDGWeQYf{_A^IE^a@HWeyCGcfA\\g[aA]IAO`KEWYxBCgXpgckRPhEiP@EEoZhDCcZPpc[Q@nFAF_^A?GWcASI?haGGoeA]I_jBWLwkAsM?xawJoyBmK?pBoNWrBOO@@bGLO}B}R`NE?XXODCU`ZdGSp[DuT@TDwVxVD_WpcdWUP`EIXaIHYXqJH]YAKHaYQLHeYaMHiYqNHmZAOH}ZQPIAZaQIEZqRHq[ATHu[QSHy[q@GgdbHf[_aJH[qh{GKbAWKm]aDGseRKfW`aMHgrXqFwbqZKy\\@~H?frNfc_AQICsXtG_cQ_LA]qFHOfbSf_aQRHosh|GOdQ\\LMFO}DgpG\\CCUrCMEEoxDspWZBkVBDMME?vEGqGXBWXRB`gM@]KYE_yD{pbf`cN@_KKxgWBsWRGMUF?~EKpw[C?XBFMmB?lFGugLAo\\BZ_wIpxLqBoiFKtWOAw\\R\\`CJpuLeC_oF[tgRBC]B^`WL@|MAD?qFguGVBK^BV`SLPzLyB?]Fwubl_sFp~LkzgMA?_B[M}Bo`GCtRo`?HAGLs{WPAS`bXNIC_aGGtbr`[GqBL[|wRA[aR^NQD?hGSuBu`SIAFLw}GUAW`B_NU??@CGOwE?[P@D__ApEC]AOIC_QWA?SQ`L_K@@JCq??@CwRrB_G@PODCpGB?OS`RKU@_FDOTRE__ApUDcqGH?gTpWK]I_uES[qlMW{GkB_V`sISxrmawM`^FShbgNEJozDo\\afMO{goBoW@vIGyRrbCNP`F_iRiNQIovEG]QgMSzwqBwU`yIKwRubSNpbFkibjNaL@?EO^QkMo|WlBcVPqIOwrlbKOPZFoiraN]F_`Fw_QpNc~G^A?^q?J?}bzaGGqAGKjb~OAH_fGOaQqOD@GdAc`QEI{~cAaOIAFG_kr|OMX`jHWeqpKcrhfE_dqWIwqbJecYaXHgjrKKuZpoHofQqLGsxlFCfa_JKsBSeoZa^ICkBNLE?@NGgjRHOm?PMGkgbIOm?`PGoirJOq@POGsgrKOq@@RGwhrLOu?pQG{hBMOu@`SH?hRNOy@pTHChbOOyA@UHGiBPO}A`VHOibSPAAPWHSjBRPAApXHKiRQO}?@AEWlBHOTIW@CKXqtKh@Si_GQ`gJSqsHQm@@JEglbLOhJWBCoYqsKxAcm_SRPhJWrCHQq@`CEolrNOXJwFCSZQwL@@co__P`mJ[sSFREApFE{mRQO\\LGICc[QwLPACq_cQ@oJcssGRMB?L@wFqyNc}gM?{G?`Jg}r{`?COcASmr|NyD_VAKHa{O@@GQ@KG_fJo~s@`ODOgAcmsAOMB_NAgIq|M{{GK?sJ?lJwzRm`?COmA{nrpNIC_RB?KR?NK|GS@SK_tKG|bw`WDorBOoRtN]E?XBWLq|MSxgY@kM?xJwwrf`_EO{BsoBhMiF?\\C?OR@MGzGY@kM_zJ{xBg`oFO}B{ob`MmO`BCgQpKCt@SHOiP@DCWPpGCd@cFObTW?_O?wA_[@gI?eAGL?qBoM`CCGR@IDOS`[DgX@aEoY`sFG_@}FgbAIHoeaQHWuAiIwkauJgnbAKWqbULGrcWPHGC]PXBcoOhJdgSHPdMShSdURhXdYTxWc}Y@fegYXeewZHoE}ZYIFM\\x{fi\\hqfQ\\Xxfa]x|gE_iBgU`h}ge`IFf}aI?cwW@vGmS@YFgbXQDs[aNdOV`sHATP^FScXRDo\\aMdWW`xHIUP`F_cxVEK]qSdCUp{GqU@cFsdXNES[qIgG_qVHa`QEHceh~G?fq`fw_QUHm`AHHofYFG_fa_cGR@eEmOpIE[YHJCsYPicOP`kEyPPHEs[XFC_Zpo`wFo_AEH?dA_IWaAKH_fagIouB]J?lB_MWmA{M_zb?KO{BuKosC?OWqBSN_~cwRp_EUS@PDgUxQDKV@\\dOTP]D}TpWEKXHUDcWPaeWaaUe[aqVe_bAWecbQXegbaYekbqZeocA^escQ_ewca`e{cq[f?dQ\\fCdA]fK_QIHWqXvGGaqVKi^ABGoeBJfg`QLHcrHuGWbaYKu[`}G{erMfO^qOH{rxxG?ca`LE\\QGHCgBOfk`qSHwtHwGccq[LI^QCHSfRR`sN`YKQFP@DkpB``kMP\\KUEozDopRb`_LpaKaEOuESowYB_VbE`gM`^KWxwXBoWBBMYE?|ECqBd`oNpbK]F@?EOprj_oJPqLiBOkFOuwMAk]R[_{I`rLUC?mFSvWPA{\\bX`GK@vLYCopF_vwUBO^R_`OK`yLaDorFotwTBS]r]_oF`}LgzWL@{^rZMyB__G?vBn_{GQ@LS{GOAOaB\\NECOdGWuRq`GGaALW{wVAK_rVN]CofGcvrs`OIQDL_|gTA_`r]NaD_eGOwBt_??PACM@_FCOPWG?kP`F_cA`GCe?_DCgRWB?OQpK_??PMC{owA?SS@PKQ?oCDGSrD_W@pSDSpgG?kT`XKaAOID[UBFagL`dFKjReNAJ?wDw\\AdM[zgmBgVptIWyBpa{Mp[FWhrcNIK?{E?\\qaMc{wpBsWPwIcybsakLpaFciBdM}K_}Dg]abMC|gtB{WpzIgyrwbOO@cFsjBkNUJOxDs[acMKzWrCCUp{Ikwbv`wGP}GCkRxNqFo_F{_AoNg}waAK_aBIw~s?aWHqCGckc@OQHOhGS`anNx?gcA_`qGJK~SBeWYqUHkkRHKyXpgH[eAmKgqxhEgeQYI{rBLe{[A[HskbQLMZPpHwgArL?tHkEwfq`J?rrP_?RqIIsqSJ_CRaJIGqcJ_GSQKIkqsK_SSALIKrCK_OSqMI[rSL_KSaNIOrcL_WTAOISrsM_[TQPIWsCM__TaQI_sSN_gTqSIgtCO_cUATIossO_kUQRIcscN_?O`eJOqSDQe?PBE[lRIOTIgACgYAtKlASj_OQpiJWrSIQu?pKEklBMOhJgDCsYQuKpASk_WP@kJ[rsEQ}@pDEsmBOOXKGGCWZavLD@sp_kPpnJcscFRQA`HFCmBSO`KgHC_[AxLLACr_oBO]@{mbxNiB_NA?GQyNk~GO@CH?dJk~R}`WDobAWnC?OQC_RAGHq{N|?WS@SI?hJl?cB_wBoiAknRnNAB?LAoJQ}MszgO@CJ_nJ{{Rq`GCooBCoBrNQD?TBGLRANW}GU@[KosKC|Rv`_EOuB[nRdMYE_ZB_MQ}MKxwW@cN?|K?yRi`oFP?CCoRaMqE_ZBgMq~MOyG[@sN_~KGwRjcGOpICkR@LOTASIcOPPEC[Q@HOX@sG";
DIGRAPHS_NamedDigraphs.("moorespartial") := ".~?DT_c@wG_{BgL_yBWD@A@?N_WCWE@I@WC`gEWL@AB_N_wCO[_sC_ZaKFw_aEGg~AuJhBBAJxFBYLXDcYNg|cIOXPAeIhOc}TojaqVOrbQKWqdqUxYdeMgxbaLw{bmZPkem]pufU[HngC^y?fQ[xyfe[hpfa\\xUdUHwghKcYQh[diThgeY[Hmfa\\i?fyaIEhAbiWhYgI]cINaCHPYEMG_eDcXG^AKV@`a?H@ZEII_jDGTwhAoSPWasKO~D}J_qC?WGoBOOp]a{KpCDuLOwC_XwuB[PpgbcN@eEeM_zESYhvF_aQIfK\\ABGQ^q?GC_hnF?^P}ekZ@lEy\\PuFk^H@CwRpScGRPODMN`ECgQw|CSQPKfC[aFGa]PyGS`gfDWaqLa_TQKGy?OI@e??H@i?_KHA?oJG}?oFA[qgA?_IBH_??OX@gprG__A_gGwprK_[AOfGsqBJ_kHQNIgqgKAWcAhKeIAKH?pbHKqHqJG{pRIKm@_T@WeQYekZAXHgnXdEgZ@lJaX`hEkZavesZaZHonXnF?eq[Jq[PqHsfbBfK\\A\\HwmhtFWfq_Jy\\pwH{gAxdCU@uFkkHQD[\\P{I}]PyICgbCfk^A`IGnhFE_[@|JYQ@fE{^atfs^abIOnH~G?gqcJmO@_FO_qqb{VprGOkXCDs[QGJQOp]FG`qrcGS@yGSoH@C{]QEJ}N`EG?_b@bsPP~GCoi@GGhQeJm_qCIShaygS`afI_pIFG_hqgKMQ`JDoWR@ccR@ZEGohLDKUPcKAR`SDgWq~dS\\qIGwphUF_aQLKUEOYGcaax`[FqTHaE?_HWdw`AKcaSaGHAPHMLPEC_QguCSPpHawN`?CmJO|B{RGnCGP@Lb?OPBCyI`ODGSwhC{SPSaoT`WGmIpTD[bGiAkiQnacJAiJAK_sDwWGpBKVP^bGLAQHPAWpBKcQROiJ_qHGkcJasKQPJDBGoBOdArOmJorHKlCKb[MPeEaM?yESXwuB[daubSMATJUM?yHSeCQPQLoxHWdsPPMM_zH_mCTbcNAVJ\\DgN@OF?{PaC?R@kMsV`kF_zEhEg[@sN@hPeC_U@weRSPmCOT@sebSPqH@aHCjb{OyGp`HGjRzOuG`cHKjB{PAGPbHOirzO}Fp[HSjRxPQG@ZHWjbyPM@OR@[eBxP\\EgC@OEAVNhECXaSUaiIksCGaWUQhIosSFaoaqNIgssDOaIqKH?iRQOX@wYGcbRGK{|rw`caaMK[rbuNaIPSDg|SCO`HwiDKURsOL@sgasR@aNL?CMP}J`JEC{b~OtGGoCwWrtOHBsaa{RPcNP?SOQELPIDo{b|PPGwuCcUrrNxCscbs_QeMszc?bw_adMozb~b{`AeM[zs?RUOABISxbnN|LhBG[iBhND?hCG_hrgND?X@GWiBjN@?cxcG`QfMg{C@RiQ@}IKwrcNuPp|IOwbcNyPP~IOxRlNxNhEG?grdMo~S|c{]QaM?ysCd?]a`M?ycBdG^A`L{wSBSISPzIGvb`OPOXVFSgB[L|@hWFWfr[Lx@XUF_fr\\N\\@TEdS\\q_Ls|cESUV`qHsubhOeVPpHwubgOiVprHwurfOhQh_FOfRZMXATHeWZaZL[uCPeSZQ[LWuCQe[Zq[LcwsQSyY@oHkuRaPDRXhEkebTL\\FC]egZAXLStcZPuAoV@{GObASsBxNmB?WA?G_cAWsRyNrTWH?]AGN?yBWM_u@OO_OBwE@E@_Q_U@GY@eBOO_wBwM@CFGL@GEwb@}GG`aINolayOooa}PoubUPXEbyNXAcESOhaiSHNd[IwkdsKwsbEKh[dmUhXbiMWwb]NGzesZHjfk\\htfAZy@F}_HsfM]hxfI[Xwf]ThTa]IIRHEciVHYdYYHefAZhwfY_H}ga`iOgyeIUiAfhAbw`ASU`baGH`XEQFobDoWW_AOUpaagIpQD]IOkDCUGlBCNp^awK`?EAK?sCKVgnBKP@\\bSM@GE]L_vC[YGxBoX`hbgMpdEi\\pwGcahrFO_qCf{_A@GIZpoFs^hjEoZPmfS\\`zFqOPMC{THACsS@RbwP`ICmNPDCcRHpFG`qGfc]aDGYHpUGkbWgDSbAM_CA_X_?AOY_GBAO_KAqN_K@ofKi?_GA_qW??CEOYK[qGG?gIAMK[rGF?cHqLK_qwJASbqiKiB?eH?iRHa_bAOKWqRKa[aqNKSqbJ_WDOUHcehjEoeQYJuXPiEoZQweWYPjEwlxlEweq[JuZpoHkfA{fC[a\\HwoxrFOfQ]Ji\\PuH{gA}f[]A^I?mXPD_\\`zJAS`VFS^Anfc]a`IGpHzFogQaJyPpgF?^Quc_XpnFwlX|FwgqcJq^q?IKhAzc?W@sGKkg~D{[qCJEP@\\FCaAscKV`qG[kxAD?]aDKAOPNFc`a~bwPa?GGoW|CS^q@KI_QAIShazgK`AdIWmiDGWhqgKQ`qGI[iBBcgQp[ECoXHCoUpaKIRPRDcXB?cwT@YEKnxTF[aaMKYT`wGcbRD`cEaHGgmWV@{dQW`_GAUH]GObHGdGaAOcQRbSP`GCiL`DC[QWmBwO@JasNO~CqJpACORWoCCOpMagS@QDMIPNDCTGkDWUAJakTPVGqI_jIcjwhAoiaobGL@]EAKOrDsVwqBOcaSOeKOrHCcsIawKaQJHAwlBCcQpOqK?sHOksJa{KqRJPBGvBcX`gb_M`dE]L_vHWlgtB_dQtb_MaTH`CcSb[MQUH\\CSRbgMqWJ`DWxBodqvPYBoS@oNCW`?CoZBlDwZ@wMpiPiF?\\BoYSX`GD_]HctCZ`CDO\\HgtC[aOWaPIw~CMaKWQQIs}sLaGXARIo~COaCWqSIk}sN`{VATIs}SSa?UqUIw}cR_SCoVH_}SVPi@?S@_dryP`EWdDgiajL@AGeDciQkLD@wkGkbqiLL@SGakbAOIcscEO]EaHGsqBNN[}GXGgbbFKw|bwacT@YNT@CGQ]I`RDc|CBO\\IGlCoWbrO@Bc^awQp`NG~sLQAK@MEK|SAO|GgnCsXBsODCC`bSQ`[NG~SSQML`HDk{r}PLHG|GChblMx?G}GGhRkMw~w~GOhbfM|?Ctc?_qdMWzr~RYOqFI_yRpOIPAGI[yBpOEOQEI_yroOHMXAGShriN@?Syc_^abMKxB|c[^QcMGxB}cS^qcMSzR}RyPa?IKxRkNtNXNFcgb_Ml@HOFggR_Mh?xQFogR^MD?tAdC]qaLwwSCSETptI?vB^OYU@uH{vB]OUT`wH{vRvOTPhTF[gB\\NX@dDdw[a\\LgySHds[Q]LgyCId{[q]LkxsISiW@sHsureOdQXeEwerVL`CXdEsfBUL`ChfE{fBXMLCdMe_[AZLcwcPSuYPjHgtRVPpFhiEoeRTLXEs\\_kDo^ACGodL?}Rz_oE?_AGH?eLC}b{";
DIGRAPHS_NamedDigraphs.("truncatedwitt") := ".~?Fy_A?G?_A?G?_A?G?_A?G?_A?G?bOHog`{GWdBIJ_paOHgiA}Iol`{J_w`oGwdA{NGkBKMg\\BMG_h`wIw]BAF?hBYG?kCEH_gBUG_bBeI_qCAGOpBQH?fB]JOoB}FO_CIX?[CqGpMEQF`QEQHP^aGVGiDmJpLa?UGcDQJ`TbKU@gaCSwqDWYW]DAI@RaCTwhCsYGnC{XwqCyJPOEeJPWa{TggDUF`PbKSGbCuI@Sa?SgqC}H@VasSWmD]J`RaKRwhDYKpQa_TwmDQGPTaOSwhC}JPQbKSWbDYG@ObGRW]DaJpMbCU`nFuFP_F}I`[F_cw[D{]qEaSR@~HEI`KEgbg[DkYqKaGVqLG}K@_Es]GjDsX`}`sWP{GeJ@]FiK@\\GkbWjE?[qQa[UQJG}FpaFQH`YG?aGpEKZQRaWUPqa[WpdFyKPXFK]WoECZ@zaGRAAaSUpxHIJ@_G[aG[Do^QO`{UPuGMHP[FC\\g\\Dw_QM`{UaCGqJ@\\EyHpaFS]gaDk^ADaoWPnHAFpbF?agfDgYa@akV`vGiH`aHEH`bEo`gjEC[aAagVq?G]FP\\Ek`G_DCXq^aOTQTJeGPSIKkghCwhaibCWaDGeK@]FC_w{CGQWtCKPGsBcNgsBgPgzB{OG|CcQwxBgQGwB{PwvBoQwwC?Qh@CKPWvBsOgzC[QgtBWOWuCOPW}CWQGyCmN@Fb[PW~CELPHcOQgxCMM@GbsOG}CIL?zbWPgPEO[`~IOtgREOZ@|HwtWSFCbqgIcvWQGoba]IOtwZFWbQ[IWuWOEgYqSHStGRFKcQkI{vwXFG`QZJSvwUG{fQsK?vwOFs`qqJ[vWOEW[A`JgtwZEw\\QkKGvWQES\\qdJktGWEo`qZJcuwPF[`AmJOvgXE{`aYJkugUF?_Q{KKtWTFOaqiJsvGYEw_qUJCvgVGGeQyJ{uwUFS_qjJ?vGTES`A_IstWXE[]Q[J{vGZF{`QrJstGOF_`a_JWuWTEgaaUH{ugWEscAnKGtgSFocQbJ?twVGccaYIStgWFgbQfJovgYGkdqvJwuGPEW_QWIGuGYE_ZpwI_uwZHGfQmKCtwSFOlQxJwuWQFkcqfKKvWTFW]aWJ_vwYE[]qGIWtgWFCjQuK?uGUFwbAVJGugRE_aQhKCvGPEc[p{HOuwXEcZQ?HStWVFwbaiJKvgQG?cA^JCuWSFc_aaJ_tGVEkaabIkuGRG_cqXICugTDSVbHLAD`SDspBG`KT@`KwssO`WTp]KWsWZCwURFK}C`MEKrbO`ORPaKcssT`kRpaK_sgYD_VBDK}DpKDCqRM`_S@[L?ssX`CR@QK[qsW`GRpYKWrGRD[WBJK}E@QD{pBQ`CS@ZKosWXDSWBILHCWSDWURLLDFGODWWrCKhDwVD_UrGKhFgYDCVrEKtFwXDKWRKKtCwTDKVRDK\\FWOCsUbDKlDG@@?COQ@KD?T@WDoW@cE_Z_{F?\\@{G_dAWHoiAkJ?oBDIGM@wG?`AKH?gAcJOmA{K_rQa?RSLStbVL_uRYLkvB\\Lwvw@B{P@cEs[p}GkdaZIlDc[_CN`JEcZ@qGKaaVIgjsVQ]?OyCGYACG_eAYJonr?PTGW@BkQ@dFS`aPHSfQ^IdEC^_CL?wFW]P|GogQtJsosXPy?PDCc[@sGScq[HwkawPlHG@BWMPjFk^ALIGkQyKHCC\\_CN@?F[`qQIOiArJWncQQY?PBCW[PwF{badIwlqxPDHW@BSLpiG?bq_IokAzKDEcb_CNPFEWZaAH?dAbIWhsRQA?P@CgXpnFg_QHHcjQsPPGgNCoUPYDkV@\\DwVp_ECW`bQhIwMCsR`ND?SPQDKT@TDWTpWQdIwKBCKpcJ{oRBL|BCMQHHCe_sK?qEOnb?KGvcLO|GsdQ]Ao^BK]aNH[gR[MS}cTPlHWCAoK`kF_hqsLWxRsPdGSe_KF_^FSaq]JovrkO@ES_Q]@__AWYPxGcgB^MdAsRPTEGJA[JPsGKdQlL{ycDPHFCb__HOgEwbqdJsuRoNLCc^Q]A_hAk^AUJKnrUMG|SRPxHwKAWJPrGSea^Lo~sBPDEcf_cG_mFc^qfIotrnOXCCcQ]BOhAo_qLHciBWMC|sTP|GGI@sK`qHGgalLkyrxPDDSc_GGOdGG`QhJ[trqOhDS]QY@oaAO[PtHOmBXM@@sTPtGwJ@wH`gEs^QoLg~SLP\\Fse_cF?`G[cqTIGuRmO`CsVPe@?hB?aAOIklrZM_zCOP\\EwDAkKaCGwdAZL_|cBPHDsW__H?iESYqgJGtsCOTDsYQU@_^AsY`}H_lRTMG~CVPtHGAAGIAPHGfApLOwR}PLEca_GF?cF__A`IKvRfOlCCSP}?o_A[\\aUJkoB[MKzS\\P|GgDA{K@nG[iapLWycCPDDc^_[HOmFObQ\\IOvRxOdESaQUAo_BC`aGHkmBTNG}sPPPEWIAKK@hEw]axL_zs@PdFC\\_gJ?nFCaqYIWvbzOhCcZQI?_iAw]p|JKmrXMXBcPPlGWBAsKPfEo_A|Lh?SEPLDcZ__GOaFWkAyJwvRtNpEs[QA@o`AgXaKI?mRSNO}cQPPHGDAKJ@lFkeAuLkwrpPLDCb_oFo_ES_Q[IgubjOdCc]QE@_fBK[ACIcirYM@?cSPhGGE@wKPqHCeQqLozbwP@FC]_cH?dFolQuKCtB|O|CSWPq@?\\A{Y@vFwmbWMXBCSPxHWHA_IaEH?famLs|b~PXGse_sFObFK_amJotcAO\\CCWQI@o[A_Y`vKGorVMs{SWQ@GWCAKIpjGChAnLw}CGPXGCc_KH_rE{cq\\I{tRvNxCC`QMBOjA{^qHH[grZMO{CYPpHgK@wHqIGojAsLSxBrPXECd_SFOhEWXqIISvbfMdEc\\QEA?[Aw[AMH{hbSM`?CUPtFgAAWIPPECqbNMSyrmN@@CEQq?omB?R`[KWsbdM[{rzN|@sr_sF_dD[URGLKxRlM{|SJOpKWE@sH@KCsqbPMOyBnNG}cBRM@O[A?SpYKgsBaMW{rsNt?Sp__HonDCVbDK_wRfN_}c@OdJgEACIpUDkqRJMCyrpNt@SKRPYwIA?HPTEGprLMCybmO@?sFRU@?iBKTpYKOrbcM[{R{OXBSnVEA?^AcU@\\KWqR_MWzSBOhBSv_KI?kC{VrNL?ybtO`ASLOxLDl_sG_rDOWbNLCzBwNc~sDOhKDm_kH?oDWVrDLKwreMw|rxNpLgHAKKPOECqrQMOybsNc~cJQx[wIAGJPRDcqRQMKyRoNhBcNQ}@OiAsTPbKSrB_M_yrzNxACoUq@obA[S`\\L?sRuNk~C?OpBslUi?_nBCU@_KsrbbM_|R|N|?cl_kGOkCsVBCKswBkN?|c@OlKgC@oF`SEKpbJNG|bvOH@CHRT]GKA_IpKCwqBKMc|C?OH@cIRH\\gHAWK`QE?rBRMGzBpNH@sMR\\ZwF@{K`ODwpBFMczrrN\\@SGQp]wK@sJ`NDkprMMGzRwNx@CNRX^GMBOLPcE[Y@hMGwrcMTCCP_{L?vEW]@{JwxbnN?{STP]BozCSYppGG`r`M_|bxPPFGNBcO@lFG_R?MGybvOpECX_{L_{E{^aBJ{wrjN`BSYPmBo}CCXPsFc_BmNK}b}PDFWNBsOpkFK]aCMOyRkNlGsc_{LOwEw\\pzF{xrsNTBsUQIBo~CGY`uGS`b_Nt?CMPHCwNBgP@oFS^R@Ms{b{N|CCd_{P`FG_aQIGkxS@OH?s]P}BpGCcbQMH?ccCO\\ACIQDHwNCgQqKG{cQROT@cHOlGCe_wPPGIgiqlJo|rwNl?s\\QUB_vB_daVH_lBhODBCLPHGGMBkQQXHgeqnMgyrkOHCse_wMO{HofQoJSzRmOTBcTQ]B_|CWfa^I?gR_NG{sEP\\GWMBwQaaIKhAdMCzrsOPEC]_wM_~IWlavKGxbtNX@sXP}B`@CkhqgJCmRfN?}SGPlGwMCKPqhIomA|Ng~s?OlFCa_wO`CHOjarJgyBpOhBsYQQB_uC?dQqJkor{Ns~cHPPDgH@cJPUEOjqoJCkarKwrrZWqA_ZAwU@cJSlqwJgnBPLKvEM_kE_nD[XAsJWmQzJssBQLtbWF@_J@aGC_aCGWcaRKkrB^WIA?VAkWqBGS`qJH?cRLK{ueA_sEogDGdAUHgiQoJWqRLL|`wI@OGPOH[eQ[IOhq|K_rr^WmAOR@wR`hHkfqbJ[nRIKsteE_oCOqD[YAVI?hQrJSqRIL\\`GC@GJ`OE[gQaIgkaxKSqbTWhfwK@[JpTHkhAhIsjaqKksRSWM?oWAKTATHchajJCmBDKsvUBYI?_RBKRPfHggAfIwmBKLGuuD__DOlCwdAVHwiqkJcpBPLx`WA@cG@NE_eQ\\IGlbBKksrUW`hwG@WF`UHSdacIglRAK[sBWW`hgJ@_K`THgfQ]I_iavKWrrXWe?oYAcSphH_gQfJ?obCLKuUCYQ@?OACUAWHkfaeJKorKL?ueF_sD?cDCYQ\\ISjAlJCprQLpaek_[D_rC{eA[I_jQyJkpRHLxbE`_WCObDKXqUIKjAyKKqBMLPbUi_SC?cDGYA^IkjazKGpbMLTbe__[DO_CsgQdIWiQsJopbGL``eY_WDohDOdA[H{ganJoprKL\\aU[_SC_gDCdQ_IKiAnJOpBJLhauZ_cCOjDcY`kEsZaKJwpBOL|bwA@[FPaESZpoFCaALKSpb[X@lGH@_K@bEk[`rFO\\QMK[sRTXA?_YAoU`eFW\\qHG{nrGKcueNZYA_T@{V@eEo]QDG_bbIKouUX_kD?dDwXPqF_`qHGoqbJL`eGC@KKP[F?]`~GGaqKK_sR\\XI@OTA[R@mFC[q?GWaRLLGtEQ_gEOpDk\\@uFk^a@G[pROLPcwL@WHp^EkZPnF[cR@Kkrb\\XLmGB@kG`\\Eg[@rFkcA~Korb]XPmWJ@?I`_FC^aCGSnb@KcsRZXQ@_SAGWPeEk]a?GKcrCKSuuT_oC_iDwZ@nFO^qNHGrrQLxdWE@?F@\\Es\\PuFo_aGKssrWXXoWK@kHP_F[^QEGkbb?KWsBUXXoGD@cH`^ESY`{GKcb?K[qRXX\\mgC@WFpZFG^QLHKna~K{srVX\\mwL@KH`KFS]ACGgbqOKOpbVXdnGB@GF@`Ew]Q@GgbQPK[qBUX`nwG@gK@XFo^P}F{_AIKgrb[XDfWF@CFPYF_]PyFkoB@LGsrTXDfgC@_M`?CcRp_EOZpzIw}RzR|OWB@WM?xCkS@XEO\\@yIo}B{S\\RGK@KOpHCgS@aFG]Q\\JoxRqTdUwI@gM?{CCRp]GshQiK?xReTtWWD@[L`AC_Rp\\F[_QUIWwSARpPGF@OLoxCCSp^Ew\\aTJWwSJSLQwD@gLO~CWR`_EocAYJK{CLTLVgK@WO@CC_S`bEg`A]I|@SLT@TWA@OL?{CgTpZF{kayKC}RyRxRGK@SM@ACWU@YE[[@}Jk}bzSTQgH@?L?|B{Sp[Fk_AwJcyB{STVGL@gLoyC[RP\\FC]ajJgwrfShVGE@KL@CC[U@XGSdqoJ{zCETHSwE@WL_vCgSPYESYAIICzRvTXWWE@cMOzCGS`aFKjQwKCwr}RtOWA@?L_yBkT@^H?cqgJ?wBpTTXwA@kLPAC[R@RFc^A^ISzSFS@XGH@OMo}CKR@TGScaSI?{CDTDWGG@kN?~C_TP^FC\\ApJk~sNRtNwF@?O@ACSTPZESbaaJSxbqT`XgC@[PPECgRP]EW`AVI_ysITDVwB@SOPFC_SPaGKbqYHowBhT|WGF@GNPGCgR@SEk[AkIwxr}SHQWJ@KMpECkSPbE{_AjJGxB~SHWwI@[N`FCcR``E[[p~JCyBwSdWwC@CM?|COR`\\Ggba{KGwbnS@RwC@gL?uCkT``Es`qXIK|S@SLQGG@?LpCCWR@VF[bAbIc|b|SXXWI@CL@ACKRP_GGaQZJW|CKStXWI@_MO~CST`\\GKdAWJwzCGT@XwD@CLo}C?T`]Ek^alJcxCNRxPwF@kM?yCKTp[G{fafJxAcMTHVgG@GL_wBwSpZEWYa[JL@cGTPWgH@GLP?CkTp^F_^Q\\KG{sCTXVWH@kMPCCcT@ZGGcQcKK|S?RpYGJ@WM_{CSU@aFWaqXIdASKSPYGJ@cL@?CCS@YEocqWI?ysMUHYWJ@SLo~CKS`XEc\\QtK?{sBS|XGK@cLO{BwSPXGccQUJsyb|S`QwL@CLOxCgRp`HGhqnJ{yRpTPYWL@[Mo{COT`_E_[`{J\\?sCT`UgL@_OpGCkR`]EwaqcJOybuSxTwB@KLOyBsS`YEs`asKL?cJSXRWB@cL_~CcU@bFsaA^IgwbmThXgD@_Mo|CCRP`Ec]AGIG|sFRhUWE@SNO}CSS@bGCbATHl?C@RlTwA@GOPECcTP[GW`qeJs|CHRlRgG@OPPFCkT@[FSbQ`J[zbnRhU~y_A?G?_A?G?_A?G?_A?G?_A?G?bOHog`{GWdBIJ_paOHgiA}Iol`{J_w`oGwdA{NGkBKMg\\BMG_h`wIw]BAF?hBYG?kCEH_gBUG_bBeI_qCAGOpBQH?fB]JOoB}FO_CIX?[CqGpMEQF`QEQHP^aGVGiDmJpLa?UGcDQJ`TbKU@gaCSwqDWYW]DAI@RaCTwhCsYGnC{XwqCyJPOEeJPWa{TggDUF`PbKSGbCuI@Sa?SgqC}H@VasSWmD]J`RaKRwhDYKpQa_TwmDQGPTaOSwhC}JPQbKSWbDYG@ObGRW]DaJpMbCU`nFuFP_F}I`[F_cw[D{]qEaSR@~HEI`KEgbg[DkYqKaGVqLG}K@_Es]GjDsX`}`sWP{GeJ@]FiK@\\GkbWjE?[qQa[UQJG}FpaFQH`YG?aGpEKZQRaWUPqa[WpdFyKPXFK]WoECZ@zaGRAAaSUpxHIJ@_G[aG[Do^QO`{UPuGMHP[FC\\g\\Dw_QM`{UaCGqJ@\\EyHpaFS]gaDk^ADaoWPnHAFpbF?agfDgYa@akV`vGiH`aHEH`bEo`gjEC[aAagVq?G]FP\\Ek`G_DCXq^aOTQTJeGPSIKkghCwhaibCWaDGeK@]FC_w{CGQWtCKPGsBcNgsBgPgzB{OG|CcQwxBgQGwB{PwvBoQwwC?Qh@CKPWvBsOgzC[QgtBWOWuCOPW}CWQGyCmN@Fb[PW~CELPHcOQgxCMM@GbsOG}CIL?zbWPgPEO[`~IOtgREOZ@|HwtWSFCbqgIcvWQGoba]IOtwZFWbQ[IWuWOEgYqSHStGRFKcQkI{vwXFG`QZJSvwUG{fQsK?vwOFs`qqJ[vWOEW[A`JgtwZEw\\QkKGvWQES\\qdJktGWEo`qZJcuwPF[`AmJOvgXE{`aYJkugUF?_Q{KKtWTFOaqiJsvGYEw_qUJCvgVGGeQyJ{uwUFS_qjJ?vGTES`A_IstWXE[]Q[J{vGZF{`QrJstGOF_`a_JWuWTEgaaUH{ugWEscAnKGtgSFocQbJ?twVGccaYIStgWFgbQfJovgYGkdqvJwuGPEW_QWIGuGYE_ZpwI_uwZHGfQmKCtwSFOlQxJwuWQFkcqfKKvWTFW]aWJ_vwYE[]qGIWtgWFCjQuK?uGUFwbAVJGugRE_aQhKCvGPEc[p{HOuwXEcZQ?HStWVFwbaiJKvgQG?cA^JCuWSFc_aaJ_tGVEkaabIkuGRG_cqXICugTDSVbHLAD`SDspBG`KT@`KwssO`WTp]KWsWZCwURFK}C`MEKrbO`ORPaKcssT`kRpaK_sgYD_VBDK}DpKDCqRM`_S@[L?ssX`CR@QK[qsW`GRpYKWrGRD[WBJK}E@QD{pBQ`CS@ZKosWXDSWBILHCWSDWURLLDFGODWWrCKhDwVD_UrGKhFgYDCVrEKtFwXDKWRKKtCwTDKVRDK\\FWOCsUbDKlDG@@?COQ@KD?T@WDoW@cE_Z_{F?\\@{G_dAWHoiAkJ?oBDIGM@wG?`AKH?gAcJOmA{K_rQa?RSLStbVL_uRYLkvB\\Lwvw@B{P@cEs[p}GkdaZIlDc[_CN`JEcZ@qGKaaVIgjsVQ]?OyCGYACG_eAYJonr?PTGW@BkQ@dFS`aPHSfQ^IdEC^_CL?wFW]P|GogQtJsosXPy?PDCc[@sGScq[HwkawPlHG@BWMPjFk^ALIGkQyKHCC\\_CN@?F[`qQIOiArJWncQQY?PBCW[PwF{badIwlqxPDHW@BSLpiG?bq_IokAzKDEcb_CNPFEWZaAH?dAbIWhsRQA?P@CgXpnFg_QHHcjQsPPGgNCoUPYDkV@\\DwVp_ECW`bQhIwMCsR`ND?SPQDKT@TDWTpWQdIwKBCKpcJ{oRBL|BCMQHHCe_sK?qEOnb?KGvcLO|GsdQ]Ao^BK]aNH[gR[MS}cTPlHWCAoK`kF_hqsLWxRsPdGSe_KF_^FSaq]JovrkO@ES_Q]@__AWYPxGcgB^MdAsRPTEGJA[JPsGKdQlL{ycDPHFCb__HOgEwbqdJsuRoNLCc^Q]A_hAk^AUJKnrUMG|SRPxHwKAWJPrGSea^Lo~sBPDEcf_cG_mFc^qfIotrnOXCCcQ]BOhAo_qLHciBWMC|sTP|GGI@sK`qHGgalLkyrxPDDSc_GGOdGG`QhJ[trqOhDS]QY@oaAO[PtHOmBXM@@sTPtGwJ@wH`gEs^QoLg~SLP\\Fse_cF?`G[cqTIGuRmO`CsVPe@?hB?aAOIklrZM_zCOP\\EwDAkKaCGwdAZL_|cBPHDsW__H?iESYqgJGtsCOTDsYQU@_^AsY`}H_lRTMG~CVPtHGAAGIAPHGfApLOwR}PLEca_GF?cF__A`IKvRfOlCCSP}?o_A[\\aUJkoB[MKzS\\P|GgDA{K@nG[iapLWycCPDDc^_[HOmFObQ\\IOvRxOdESaQUAo_BC`aGHkmBTNG}sPPPEWIAKK@hEw]axL_zs@PdFC\\_gJ?nFCaqYIWvbzOhCcZQI?_iAw]p|JKmrXMXBcPPlGWBAsKPfEo_A|Lh?SEPLDcZ__GOaFWkAyJwvRtNpEs[QA@o`AgXaKI?mRSNO}cQPPHGDAKJ@lFkeAuLkwrpPLDCb_oFo_ES_Q[IgubjOdCc]QE@_fBK[ACIcirYM@?cSPhGGE@wKPqHCeQqLozbwP@FC]_cH?dFolQuKCtB|O|CSWPq@?\\A{Y@vFwmbWMXBCSPxHWHA_IaEH?famLs|b~PXGse_sFObFK_amJotcAO\\CCWQI@o[A_Y`vKGorVMs{SWQ@GWCAKIpjGChAnLw}CGPXGCc_KH_rE{cq\\I{tRvNxCC`QMBOjA{^qHH[grZMO{CYPpHgK@wHqIGojAsLSxBrPXECd_SFOhEWXqIISvbfMdEc\\QEA?[Aw[AMH{hbSM`?CUPtFgAAWIPPECqbNMSyrmN@@CEQq?omB?R`[KWsbdM[{rzN|@sr_sF_dD[URGLKxRlM{|SJOpKWE@sH@KCsqbPMOyBnNG}cBRM@O[A?SpYKgsBaMW{rsNt?Sp__HonDCVbDK_wRfN_}c@OdJgEACIpUDkqRJMCyrpNt@SKRPYwIA?HPTEGprLMCybmO@?sFRU@?iBKTpYKOrbcM[{R{OXBSnVEA?^AcU@\\KWqR_MWzSBOhBSv_KI?kC{VrNL?ybtO`ASLOxLDl_sG_rDOWbNLCzBwNc~sDOhKDm_kH?oDWVrDLKwreMw|rxNpLgHAKKPOECqrQMOybsNc~cJQx[wIAGJPRDcqRQMKyRoNhBcNQ}@OiAsTPbKSrB_M_yrzNxACoUq@obA[S`\\L?sRuNk~C?OpBslUi?_nBCU@_KsrbbM_|R|N|?cl_kGOkCsVBCKswBkN?|c@OlKgC@oF`SEKpbJNG|bvOH@CHRT]GKA_IpKCwqBKMc|C?OH@cIRH\\gHAWK`QE?rBRMGzBpNH@sMR\\ZwF@{K`ODwpBFMczrrN\\@SGQp]wK@sJ`NDkprMMGzRwNx@CNRX^GMBOLPcE[Y@hMGwrcMTCCP_{L?vEW]@{JwxbnN?{STP]BozCSYppGG`r`M_|bxPPFGNBcO@lFG_R?MGybvOpECX_{L_{E{^aBJ{wrjN`BSYPmBo}CCXPsFc_BmNK}b}PDFWNBsOpkFK]aCMOyRkNlGsc_{LOwEw\\pzF{xrsNTBsUQIBo~CGY`uGS`b_Nt?CMPHCwNBgP@oFS^R@Ms{b{N|CCd_{P`FG_aQIGkxS@OH?s]P}BpGCcbQMH?ccCO\\ACIQDHwNCgQqKG{cQROT@cHOlGCe_wPPGIgiqlJo|rwNl?s\\QUB_vB_daVH_lBhODBCLPHGGMBkQQXHgeqnMgyrkOHCse_wMO{HofQoJSzRmOTBcTQ]B_|CWfa^I?gR_NG{sEP\\GWMBwQaaIKhAdMCzrsOPEC]_wM_~IWlavKGxbtNX@sXP}B`@CkhqgJCmRfN?}SGPlGwMCKPqhIomA|Ng~s?OlFCa_wO`CHOjarJgyBpOhBsYQQB_uC?dQqJkor{Ns~cHPPDgH@cJPUEOjqoJCkarKwrrZWqA_ZAwU@cJSlqwJgnBPLKvEM_kE_nD[XAsJWmQzJssBQLtbWF@_J@aGC_aCGWcaRKkrB^WIA?VAkWqBGS`qJH?cRLK{ueA_sEogDGdAUHgiQoJWqRLL|`wI@OGPOH[eQ[IOhq|K_rr^WmAOR@wR`hHkfqbJ[nRIKsteE_oCOqD[YAVI?hQrJSqRIL\\`GC@GJ`OE[gQaIgkaxKSqbTWhfwK@[JpTHkhAhIsjaqKksRSWM?oWAKTATHchajJCmBDKsvUBYI?_RBKRPfHggAfIwmBKLGuuD__DOlCwdAVHwiqkJcpBPLx`WA@cG@NE_eQ\\IGlbBKksrUW`hwG@WF`UHSdacIglRAK[sBWW`hgJ@_K`THgfQ]I_iavKWrrXWe?oYAcSphH_gQfJ?obCLKuUCYQ@?OACUAWHkfaeJKorKL?ueF_sD?cDCYQ\\ISjAlJCprQLpaek_[D_rC{eA[I_jQyJkpRHLxbE`_WCObDKXqUIKjAyKKqBMLPbUi_SC?cDGYA^IkjazKGpbMLTbe__[DO_CsgQdIWiQsJopbGL``eY_WDohDOdA[H{ganJoprKL\\aU[_SC_gDCdQ_IKiAnJOpBJLhauZ_cCOjDcY`kEsZaKJwpBOL|bwA@[FPaESZpoFCaALKSpb[X@lGH@_K@bEk[`rFO\\QMK[sRTXA?_YAoU`eFW\\qHG{nrGKcueNZYA_T@{V@eEo]QDG_bbIKouUX_kD?dDwXPqF_`qHGoqbJL`eGC@KKP[F?]`~GGaqKK_sR\\XI@OTA[R@mFC[q?GWaRLLGtEQ_gEOpDk\\@uFk^a@G[pROLPcwL@WHp^EkZPnF[cR@Kkrb\\XLmGB@kG`\\Eg[@rFkcA~Korb]XPmWJ@?I`_FC^aCGSnb@KcsRZXQ@_SAGWPeEk]a?GKcrCKSuuT_oC_iDwZ@nFO^qNHGrrQLxdWE@?F@\\Es\\PuFo_aGKssrWXXoWK@kHP_F[^QEGkbb?KWsBUXXoGD@cH`^ESY`{GKcb?K[qRXX\\mgC@WFpZFG^QLHKna~K{srVX\\mwL@KH`KFS]ACGgbqOKOpbVXdnGB@GF@`Ew]Q@GgbQPK[qBUX`nwG@gK@XFo^P}F{_AIKgrb[XDfWF@CFPYF_]PyFkoB@LGsrTXDfgC@_M`?CcRp_EOZpzIw}RzR|OWB@WM?xCkS@XEO\\@yIo}B{S\\RGK@KOpHCgS@aFG]Q\\JoxRqTdUwI@gM?{CCRp]GshQiK?xReTtWWD@[L`AC_Rp\\F[_QUIWwSARpPGF@OLoxCCSp^Ew\\aTJWwSJSLQwD@gLO~CWR`_EocAYJK{CLTLVgK@WO@CC_S`bEg`A]I|@SLT@TWA@OL?{CgTpZF{kayKC}RyRxRGK@SM@ACWU@YE[[@}Jk}bzSTQgH@?L?|B{Sp[Fk_AwJcyB{STVGL@gLoyC[RP\\FC]ajJgwrfShVGE@KL@CC[U@XGSdqoJ{zCETHSwE@WL_vCgSPYESYAIICzRvTXWWE@cMOzCGS`aFKjQwKCwr}RtOWA@?L_yBkT@^H?cqgJ?wBpTTXwA@kLPAC[R@RFc^A^ISzSFS@XGH@OMo}CKR@TGScaSI?{CDTDWGG@kN?~C_TP^FC\\ApJk~sNRtNwF@?O@ACSTPZESbaaJSxbqT`XgC@[PPECgRP]EW`AVI_ysITDVwB@SOPFC_SPaGKbqYHowBhT|WGF@GNPGCgR@SEk[AkIwxr}SHQWJ@KMpECkSPbE{_AjJGxB~SHWwI@[N`FCcR``E[[p~JCyBwSdWwC@CM?|COR`\\Ggba{KGwbnS@RwC@gL?uCkT``Es`qXIK|S@SLQGG@?LpCCWR@VF[bAbIc|b|SXXWI@CL@ACKRP_GGaQZJW|CKStXWI@_MO~CST`\\GKdAWJwzCGT@XwD@CLo}C?T`]Ek^alJcxCNRxPwF@kM?yCKTp[G{fafJxAcMTHVgG@GL_wBwSpZEWYa[JL@cGTPWgH@GLP?CkTp^F_^Q\\KG{sCTXVWH@kMPCCcT@ZGGcQcKK|S?RpYGJ@WM_{CSU@aFWaqXIdASKSPYGJ@cL@?CCS@YEocqWI?ysMUHYWJ@SLo~CKS`XEc\\QtK?{sBS|XGK@cLO{BwSPXGccQUJsyb|S`QwL@CLOxCgRp`HGhqnJ{yRpTPYWL@[Mo{COT`_E_[`{J\\?sCT`UgL@_OpGCkR`]EwaqcJOybuSxTwB@KLOyBsS`YEs`asKL?cJSXRWB@cL_~CcU@bFsaA^IgwbmThXgD@_Mo|CCRP`Ec]AGIG|sFRhUWE@SNO}CSS@bGCbATHl?C@RlTwA@GOPECcTP[GW`qeJs|CHRlRgG@OPPFCkT@[FSbQ`J[zbnRhU~";
DIGRAPHS_NamedDigraphs.("truncatedheawood") := ".i_OG?IICb`gcULFcqPcaOMdQxPAUPfriHU_ShshheWWiDyhYqYkEzlq}`nGKANAa_hWIPLBA@WOWNi_OG?IICb`gcULFcqPcaOMdQxPAUPfriHU_ShshheWWiDyhYqYkEzlq}`nGKANAa_hWIPLBA@WOWN";
DIGRAPHS_NamedDigraphs.("truncatedmapbius-kantor") := ".o_OG?IMCa@wkYPFdBHGmHDbQpsoZNfdILAaUgDylQmYjEzDee^hFjXus[nGZyBFaqh{aZKhriGImAiUOogeCED[GCA?Ab`G_]JEcPxOqQJaPWsk\\KErxxQbOgdi@]jSjeipmpXhfyPyu\\lFJqE}_pwki^GerI[yaAj_idcKIH`@`V";
DIGRAPHS_NamedDigraphs.("truncatedpappus") := ".u_OG?IMEcaGcOTDabGs]RGdbh_}YOdCWkQ[QfcyHU_SjTbd]m]kUZ\\iy_nGZyIwboHKIVEfpHlElXhtJlav[n_kWJ@dBB@wO_Ru_OG?IMEcaGcOTDabGs]RGdbh_}YOdCWkQ[QfcyHU_SjTbd]m]kUZ\\iy_nGZyIwboHKIVEfpHlElXhtJlav[n_kWJ@dBB@wO_R";
DIGRAPHS_NamedDigraphs.("truncatedgreatdodecahedron") := ".{_OGC_NICpWomJEcBgwuGGa@xKiHDbU``UUTiEIXecSgcxx]_UdDxHYpXkuHq]YeqxDTlHYQwD]DT^PvcTtR[SX{rQApYoUBQElVodayEr`KfVq@?_Q?{gRDbAwkYOMbbW_aGFcqgcULYEDXXUgWheYPQaRfdy@YSVcdjDerWfHxiZJcTUsdhJ_TwTT|F]PVTLpPfrLGJDj@WLGYt^AUJgZMCq\\^";
DIGRAPHS_NamedDigraphs.("truncateddesargues") := ".{_OG?IICa@Wc]MGcqHgq]MdQaCmUQeRX{ycSgdYXY_VbPY`aiXjeypix[PVZiQ|^owKaZIfpiKQfKiri|IvOkv[UU~_sxK}bUAv_lfCKWL`B`~q@?_?ggQGDaPwwaRGebHwyTIGQxXIXLfriPQaThdi@]LDiEIhemZjEjdpD\\mhJt~B_qHki^EgpI[qjMjsj\\Ar\\pXZ}BRcryLWJ]Au[Op_uCMF~";
DIGRAPHS_NamedDigraphs.("truncatednauru") := ".~?@G_GA??gSC`?UH@waFAW_IcPANbguOCW{QdHMUEHSV``[Y`P_XbGmFaGiHcAANFxCacaEMFgs\\`HwffQa_Gx{cgaU`HiehKIukjyypLZ?sibMjKjKwkbetLjOvmzi|Nzo}nsA}OWDB_SHC_osS__wRqGA?_?ID@GODaO]G`oeGAhCObwyLc@ENChQRd`aTDwWVEgSWEWqJ`waIaX?_bp}PGhG`b`yLFWQ]HxsggAM^HIGdgQYhiRAljI}mkRUoLIgrirIrMJGxlRYsLzmynR}{Nj|?ncE@OwDAPGKLDGGMC~";
DIGRAPHS_NamedDigraphs.("truncatedf26a") := ".~?@M_GA??gSC`?UG@wgGAWoJb_uLBg[JBG[GcpERd`UWDxIRDHSUdpaXe`u[ep}EFPyDFqAQaWyNbWmKHxWbdAYVHH_gdQUCgqicJI_lhQyfIYWjjy{ukJOwkZKykj[{lZW}mBeyMzw~nBv?osR@ocZCP[XF_Se@QCiBAOiAC`fM_GA??gSC`?UG@wgGAWoJb_uLBg[JBG[GcpERd`UWDxIRDHSUdpaXe`u[ep}EFPyDFqAQaWyNbWmKHxWbdAYVHH_gdQUCgqicJI_lhQyfIYWjjy{ukJOwkZKykj[{lZW}mBeyMzw~nBv?osR@ocZCP[XF_Se@QCiBAOiAC`f";
DIGRAPHS_NamedDigraphs.("truncatedtutte-coxeter") := ".~?@Y_GA??gSC`oYC`W]EaWcKaggMawkOaG_QcxIMbxEObWqSd@qTdPyUdxeWdaAVGX_beQIYeaaZeqi^fHu]gAq`JYKngaychYYfibQhLYkuiB]kKIspjrImKzOwlRevMzWynzy{n[@CoSVBP{HEpCbDQ[\\IpcnKrTBNrdJOS\\HR_TU@TDYB@?UA@_^Y_GA??gSC`oYC`W]EaWcKaggMawkOaG_QcxIMbxEObWqSd@qTdPyUdxeWdaAVGX_beQIYeaaZeqi^fHu]gAq`JYKngaychYYfibQhLYkuiB]kKIspjrImKzOwlRevMzWynzy{n[@CoSVBP{HEpCbDQ[\\IpcnKrTBNrdJOS\\HR_TU@TDYB@?UA@_^";
DIGRAPHS_NamedDigraphs.("truncateddyck") := ".~?@__GA??gSC`?UG@w[GaoiIAw{LbgoLBwkMCHSIdHMTdPaVdhcYe@mFepuYFHy]GY[_Hh{dgaM`HIOigqeeHySgjImmKIsn`a{q`R?pbWyObxQRaRaGLyKyhBeeLiSt`Jg|mRyuNzT?mBmvNKF@QKVCqSnGQk^Eoc~BRk|OrdFJRKhLs|JTT|PUtdfVUGDZ_Th[_pKV_`OUwGA?_?ID@GODa?]FAGkIa_mNBWyKBO}JB`ATAhQRdXSWdxYXEh_Z`xk\\e`q]faEfGAY^HYGbgQQcIiKhha]dIIqjjbAlJwWnKgSoKWuMcG}ScwcwaB]bMiOxhbYdLWQyNZc}lb}tOJ_zlrr@oSbDpKdJqCjFpkHNoszNSKxPqsrIR\\NQtT^STlXXtta@UwDYVGKRDwGSDn";
DIGRAPHS_NamedDigraphs.("truncatedcubicklein") := ".~?Ag_C?_?A_gC__D`?F_wG`_I`[H@OK`?J@kFaCOAKVAOTaGUacRBKSBKRBSUBCTA{Zbc]B{\\CC_CS^CKhC_fcWgcsdd?icwheWkESjEKmEClD{oEknEspEcqE[|F?{ewzfSxf|?GL?GS~GKzFs{FlCg\\EHDDG|FHTGHLNH`MhXL_pLICDHpN`SJ`kKjHVjCSJKRJSUakgc{eJsdJkoIsnIkpKCqJ{{JCzI|MIdLI[Ck@djxejp`jhai`hiXiipbihcixgj@fjP[jHZl\\jNDpmDxnDvNduNltmdrmTmN\\nNTlOLkOEBoQGnYCnQDoYGoQHoAFoIEnh~n`}p]IqIRqAQpuNpiVpaUqqXqyWqYSqQTr]Yri^ra]rq`ry__Ib_IaScBCoj_OdDeg_C?_?A_gC__D`?F_wG`_I`[H@OK`?J@kFaCOAKVAOTaGUacRBKSBKRBSUBCTA{Zbc]B{\\CC_CS^CKhC_fcWgcsdd?icwheWkESjEKmEClD{oEknEspEcqE[|F?{ewzfSxf|?GL?GS~GKzFs{FlCg\\EHDDG|FHTGHLNH`MhXL_pLICDHpN`SJ`kKjHVjCSJKRJSUakgc{eJsdJkoIsnIkpKCqJ{{JCzI|MIdLI[Ck@djxejp`jhai`hiXiipbihcixgj@fjP[jHZl\\jNDpmDxnDvNduNltmdrmTmN\\nNTlOLkOEBoQGnYCnQDoYGoQHoAFoIEnh~n`}p]IqIRqAQpuNpiVpaUqqXqyWqYSqQTr]Yri^ra]rq`ry__Ib_IaScBCoj_OdDf";
DIGRAPHS_NamedDigraphs.("truncatedfoster") := ".~?CM_A?G??I@OC_[@gH?aAoI_O@WE?]A?H_gAwL_oCGN_wCgQ@MC?P`UDGX@aEoY`YD_[`]Do]`aEWZ`iFG\\`}Fg_a?IG`aCIgaaGJGbaKJgcaOKGdaSKgeaWLGfa[LggaeJGla}JgjaiKGpbMLGvbULgqbaM@GbeMPIbiM`KbmMpMbqN@ObuNPQbyN`Sb}NpUcAO@WcEOPYcIO`[c]Pp]cYOxBEEPHCEMPXEEAPPdcaQXKcuRxOdMSXQcyTHTdYTxJciU@geGYXXEiXpjdo[h\\EqX`pdwZX`EyWpreOZxdFQU`odk\\X^F]W@u_aAgC_Y@wD_eAxgFaYPxeg]hjFm[`{eo^XpFyZP~ew_HoGQ[q@fO_xtGUZqAfW`xvGYaIGIAaYHIIaiIIQayJIYbIKIabYLIibiMIqbyNIycIPhIcyShUdiVh}eIXhiey]hqfY`JAhqpiKkiiJMgAsiwlYdJYgavismIcJejqyiWmyhJqiA|iknykJykB?jCoYqKIkrBjOpItKUlbEj[pywKamRHjgqizKmnBKjsrY}K}nrMlAtb_lEtralIvRcl}vbeliurglMtBilUvBklauRmm?wZaMMxBdmWxzgMeybjmozZmM}{JsNa{ZtNi{jrNq|zuNy}Bxng}z}N}~B|oB?sCoF?cEoP@[EO]?SH_DACI_KE?X_GE_Zoy?G?_??gD?Q@oE_cAGJ?i@?D_W@wG?eA_J_uB?O_}B_Q`GCwO@EDWS`cEGZ@iDgU@qDwV@yEGX`mEg[`uFw]aAG?gaEGOiaIG_kaMGomaQH?oaUHOqaYH_sa]HouaaIWkauJwmamIgobEKwsb]LWubIMGwCaMWxCiMgyCqMwzCyNG{DANW|DINg}DQNw~DYOH?DaOX@DiOhADqPxFDyPhBcKWXCcOWxDcWWHDEUQHHcqRXNdASxPdIRhSdUThVcmQhWEaW`hdcYhfEmV@qdsZHeFEV`leCZhbFMX@neS\\HYFAUptd{\\x_FYAGI_Q@gF_UAWJe_]HhFeY`yek]xqFqZ@|fC^hlF}Za?f?`HrGE\\ABfS`XnGI\\aFf[`iGg_gIHgcgiIgghIJgkhiKgoiILgsiiMgwjINg{jiOhEciRhQdYUh]fyWheeiZhyfI\\iCkIfJEgqqigky_JQjatiSliaJ]jQwiOmYnJihazicnIgJuiq~ionioKAkR@jGoirKMlBCjSpYuKYlrFj_qIxKembIjkqy{KqnRLjwry~KysJUMAsZVMIsj\\MQvz]MYujZMaszSMitZ[MquJXMywB`mGwzcMUxbfm_yZiMmzBlmwzzonO}JpnS}jqnK~JvnW~jwNe}bznw~z{Nv?KBOR?[AOZ@CDoX@w@Oe?SGOi?oW@e?_Y@n";
DIGRAPHS_NamedDigraphs.("truncatedbiggs-smith") := ".~?Cq_A?G??I@OC_[@gH?aBOJ_iBGG_eAgJ_qBWM_}CGP`MCgT`]DGU__CGH@EA_Q_kCwK@QBOT_{DwM@YEWW`iEw[`uFw]`cIGWAeE_i`kIw[AqFOl`wJw^AyIggaeIwkauJwmbAKWtb]KwubILG_aEGgcaUHgfaMMGxbiMw{buNg~c?Sx@DQO`OcKShCDUPPVcWSXFDYQHHciQpacqRPccwWXNEA@GD_]@gwDAMPPbgSgzDMN@SbsTW~D]N`Uc?UH@DeO`YcKUxCDqPP\\c[VxEDyQ@_ccWXIEIQpbcoXHLEURpfcwXhge_`Hhec`hiegaHjekahkeuZhnfE[hofM\\Hufe\\Xyf]]HzgQ`iGg]aYDgmah{Hq^Q\\fwfh~H}_A_gCgYBII_abgqbYMg{jiOhCkIQIucqkhQdQshYdquhaeQwhieqygojILIubamg{jyOJAcQphKkyQJIdAshSlYUJYdqvh_mIXJeeqzhgmi[IQfQdhwhi^I]gAgiCiYbImgaijqnj?juoi~kEozDkYpJGk]qzHkirB[ksvZNLyrb^lCwJOMEsrblGwjSlUtbll]uBnleurplgzJSMqtRllWzjVM}uBolc{ZYNMurqloxJ\\MUvbel{xz_MawRhmKyjaMm|Zvna}Zunm}jsnp@J|OU~cEn|@{?Ob?SHoLA{AOj@CKoTB[EOz@sNo`CKHPFAcRolCkSplFKTphFkUp\\GKWpdGk[PvFc^qHG{_QFHKfQbH[eQjIChqhIw@Qu?SkQy?oC?U?_E?^Kg?_A??A_S@GF?YAOG_sAwI_qAGH_iAwK_uBgN`ACWR`IDWV`QDgG@AAOP_gCgJ@MB?S_sDWN@]B_U`eEGY`mFG\\`}FgXAaE?h`gIgZAmF?k`sJW]A}FomaiIGhamJGla}JgobELWvbMLgqbQGG`aIHGdaYHwbbaMWybmNG|byNx?DMOPScGSHBDIP@TcSTxEDEPpUcaQXIckWhKcsXHMEERp__Q@WF_YM@ObcSWyDIMpRboTG|DUNpVbwTh?DaOPXcGUhBDmP@[cSVXFD}P`]c_WHHEEQ`ackWxKEQRPdc{XxMEYYHgGQYXhGYYhiGaYxjGiZHleyZxpfI[HrfQ\\hxfU]hvfa]yCgYaIFge`YJgi^A[fsfX}Hy^q^g?gI@IE_qagGgyKgubiNIycIPJAcalhKjIShSlIUh[liWhcmIYhkmiKIqbQlgwjiNI}cAohCkYRJMcaqhOlITJUdauh[lyWJaeQxhkmyYJifAchshY]IYfqfi?iI`IegqjiGii{jyoI|kInz@kMpZEkQqJFkmqZIkovJLLurr]kwvzPMAsB`lKwzQMItJTlWzZVl_zzXlk{ZYMqtBklSzZUMytrnl_{JXNEubrlk{j[MQvRdlwxj^M]wBgmCyZbMiwbjnU|zwne|jzni|J{OQ~SDnx@j~O^?CGoDA[BOn?cIoPBKDOv@cMo\\B{GPBASPohC{JPJDKZPrD[YPzDkVQBEKXQJFC\\pxF{aQNGC`qRHsgqVHciq`I[iQm?Sl_DJCm_K@?D_G@_F";
DIGRAPHS_NamedDigraphs.("latinsquaregraphfromz4") := ".O__@_@A_@_@_ACD_ACD_BCE_BDFG`BCDFG`BCDEH`ADEGHI`ACFGHJKaBDEFGIJLaBCEFHIJKo__@_@A_@_@_ACD_ACD_BCE_BDFG`BCDFG`BCDEH`ADEGHI`ACFGHJKaBDEFGIJLaBCEFHIJK";
DIGRAPHS_NamedDigraphs.("orthogonalarraygraphoa(4,5)") := ".X__@_@A_@A_@B_@C_@DE_@ABEF_@ACDF_@BCDE_ABCDEF_ABEFHIJ_ACDFGIJ_BCDEGHJ_ADEGHIKLM_BCFGHIKLM`ADEGHIJKLMO`BCFGHIJKLMN`ABEFHIJLMNOPQ`ACDFGIJKMNOPQ`BCDEGHJKLNOPQ`ABCDEFKLMNORSTaBCDEFGHIJNORSTaBCDEFGHIKLMPQUVx__@_@A_@A_@B_@C_@DE_@ABEF_@ACDF_@BCDE_ABCDEF_ABEFHIJ_ACDFGIJ_BCDEGHJ_ADEGHIKLM_BCFGHIKLM`ADEGHIJKLMO`BCFGHIJKLMN`ABEFHIJLMNOPQ`ACDFGIJKMNOPQ`BCDEGHJKLNOPQ`ABCDEFKLMNORSTaBCDEFGHIJNORSTaBCDEFGHIKLMPQUV";
DIGRAPHS_NamedDigraphs.("dodecadodecahedron") := ".]____hDgHdgJefLefNqCHMPbJLPaIOP`KNP`EHaFJUaGLcDNWbDO`GMYcFKbEI[}____hDgHdgJefLefNqCHMPbJLPaIOP`KNP`EHaFJUaGLcDNWbDO`GMYcFKbEI[";
