/*
 * Decompiled with CFR 0.152.
 */
package io.sentry.transport;

import io.sentry.DataCategory;
import io.sentry.Hint;
import io.sentry.SentryEnvelope;
import io.sentry.SentryEnvelopeItem;
import io.sentry.SentryLevel;
import io.sentry.SentryOptions;
import io.sentry.clientreport.DiscardReason;
import io.sentry.hints.Retryable;
import io.sentry.hints.SubmissionResult;
import io.sentry.transport.CurrentDateProvider;
import io.sentry.transport.ICurrentDateProvider;
import io.sentry.util.CollectionUtils;
import io.sentry.util.HintUtils;
import io.sentry.util.StringUtils;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public final class RateLimiter
implements Closeable {
    private static final int HTTP_RETRY_AFTER_DEFAULT_DELAY_MILLIS = 60000;
    @NotNull
    private final ICurrentDateProvider currentDateProvider;
    @NotNull
    private final SentryOptions options;
    @NotNull
    private final @NotNull Map<DataCategory, @NotNull Date> sentryRetryAfterLimit = new ConcurrentHashMap<DataCategory, Date>();
    @NotNull
    private final List<IRateLimitObserver> rateLimitObservers = new CopyOnWriteArrayList<IRateLimitObserver>();
    @Nullable
    private Timer timer = null;
    @NotNull
    private final Object timerLock = new Object();

    public RateLimiter(@NotNull ICurrentDateProvider currentDateProvider, @NotNull SentryOptions options) {
        this.currentDateProvider = currentDateProvider;
        this.options = options;
    }

    public RateLimiter(@NotNull SentryOptions options) {
        this(CurrentDateProvider.getInstance(), options);
    }

    @Nullable
    public SentryEnvelope filter(@NotNull SentryEnvelope envelope, @NotNull Hint hint) {
        ArrayList<SentryEnvelopeItem> dropItems = null;
        for (SentryEnvelopeItem item : envelope.getItems()) {
            if (!this.isRetryAfter(item.getHeader().getType().getItemType())) continue;
            if (dropItems == null) {
                dropItems = new ArrayList<SentryEnvelopeItem>();
            }
            dropItems.add(item);
            this.options.getClientReportRecorder().recordLostEnvelopeItem(DiscardReason.RATELIMIT_BACKOFF, item);
        }
        if (dropItems != null) {
            this.options.getLogger().log(SentryLevel.INFO, "%d items will be dropped due rate limiting.", dropItems.size());
            ArrayList<SentryEnvelopeItem> toSend = new ArrayList<SentryEnvelopeItem>();
            for (SentryEnvelopeItem item : envelope.getItems()) {
                if (dropItems.contains(item)) continue;
                toSend.add(item);
            }
            if (toSend.isEmpty()) {
                this.options.getLogger().log(SentryLevel.INFO, "Envelope discarded due all items rate limited.", new Object[0]);
                RateLimiter.markHintWhenSendingFailed(hint, false);
                return null;
            }
            return new SentryEnvelope(envelope.getHeader(), toSend);
        }
        return envelope;
    }

    public boolean isActiveForCategory(@NotNull DataCategory dataCategory) {
        Date currentDate = new Date(this.currentDateProvider.getCurrentTimeMillis());
        Date dateAllCategories = this.sentryRetryAfterLimit.get((Object)DataCategory.All);
        if (dateAllCategories != null && !currentDate.after(dateAllCategories)) {
            return true;
        }
        if (DataCategory.Unknown.equals((Object)dataCategory)) {
            return false;
        }
        Date dateCategory = this.sentryRetryAfterLimit.get((Object)dataCategory);
        if (dateCategory != null) {
            return !currentDate.after(dateCategory);
        }
        return false;
    }

    public boolean isAnyRateLimitActive() {
        Date currentDate = new Date(this.currentDateProvider.getCurrentTimeMillis());
        for (DataCategory dataCategory : this.sentryRetryAfterLimit.keySet()) {
            Date dateCategory = this.sentryRetryAfterLimit.get((Object)dataCategory);
            if (dateCategory == null || currentDate.after(dateCategory)) continue;
            return true;
        }
        return false;
    }

    private static void markHintWhenSendingFailed(@NotNull Hint hint, boolean retry) {
        HintUtils.runIfHasType(hint, SubmissionResult.class, result -> result.setResult(false));
        HintUtils.runIfHasType(hint, Retryable.class, retryable -> retryable.setRetry(retry));
    }

    private boolean isRetryAfter(@NotNull String itemType) {
        DataCategory dataCategory = this.getCategoryFromItemType(itemType);
        return this.isActiveForCategory(dataCategory);
    }

    @NotNull
    private DataCategory getCategoryFromItemType(@NotNull String itemType) {
        switch (itemType) {
            case "event": {
                return DataCategory.Error;
            }
            case "session": {
                return DataCategory.Session;
            }
            case "attachment": {
                return DataCategory.Attachment;
            }
            case "profile": {
                return DataCategory.Profile;
            }
            case "statsd": {
                return DataCategory.MetricBucket;
            }
            case "transaction": {
                return DataCategory.Transaction;
            }
            case "check_in": {
                return DataCategory.Monitor;
            }
            case "replay_video": {
                return DataCategory.Replay;
            }
        }
        return DataCategory.Unknown;
    }

    public void updateRetryAfterLimits(@Nullable String sentryRateLimitHeader, @Nullable String retryAfterHeader, int errorCode) {
        if (sentryRateLimitHeader != null) {
            for (String limit : sentryRateLimitHeader.split(",", -1)) {
                String limitNamespaces;
                String[] rateLimit = (limit = limit.replace(" ", "")).split(":", -1);
                String string = limitNamespaces = rateLimit.length > 4 ? rateLimit[4] : null;
                if (rateLimit.length <= 0) continue;
                String retryAfter = rateLimit[0];
                long retryAfterMillis = this.parseRetryAfterOrDefault(retryAfter);
                if (rateLimit.length <= 1) continue;
                String allCategories = rateLimit[1];
                Date date = new Date(this.currentDateProvider.getCurrentTimeMillis() + retryAfterMillis);
                if (allCategories != null && !allCategories.isEmpty()) {
                    String[] categories;
                    for (String catItem : categories = allCategories.split(";", -1)) {
                        String[] namespaces;
                        DataCategory dataCategory = DataCategory.Unknown;
                        try {
                            String catItemCapitalized = StringUtils.camelCase(catItem);
                            if (catItemCapitalized != null) {
                                dataCategory = DataCategory.valueOf(catItemCapitalized);
                            } else {
                                this.options.getLogger().log(SentryLevel.ERROR, "Couldn't capitalize: %s", catItem);
                            }
                        }
                        catch (IllegalArgumentException e) {
                            this.options.getLogger().log(SentryLevel.INFO, e, "Unknown category: %s", catItem);
                        }
                        if (DataCategory.Unknown.equals((Object)dataCategory) || DataCategory.MetricBucket.equals((Object)dataCategory) && limitNamespaces != null && !limitNamespaces.equals("") && (namespaces = limitNamespaces.split(";", -1)).length > 0 && !CollectionUtils.contains(namespaces, "custom")) continue;
                        this.applyRetryAfterOnlyIfLonger(dataCategory, date);
                    }
                    continue;
                }
                this.applyRetryAfterOnlyIfLonger(DataCategory.All, date);
            }
        } else if (errorCode == 429) {
            long retryAfterMillis = this.parseRetryAfterOrDefault(retryAfterHeader);
            Date date = new Date(this.currentDateProvider.getCurrentTimeMillis() + retryAfterMillis);
            this.applyRetryAfterOnlyIfLonger(DataCategory.All, date);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void applyRetryAfterOnlyIfLonger(@NotNull DataCategory dataCategory, @NotNull Date date) {
        Date oldDate = this.sentryRetryAfterLimit.get((Object)dataCategory);
        if (oldDate == null || date.after(oldDate)) {
            this.sentryRetryAfterLimit.put(dataCategory, date);
            this.notifyRateLimitObservers();
            Object object = this.timerLock;
            synchronized (object) {
                if (this.timer == null) {
                    this.timer = new Timer(true);
                }
                this.timer.schedule(new TimerTask(){

                    @Override
                    public void run() {
                        RateLimiter.this.notifyRateLimitObservers();
                    }
                }, date);
            }
        }
    }

    private long parseRetryAfterOrDefault(@Nullable String retryAfterHeader) {
        long retryAfterMillis = 60000L;
        if (retryAfterHeader != null) {
            try {
                retryAfterMillis = (long)(Double.parseDouble(retryAfterHeader) * 1000.0);
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        return retryAfterMillis;
    }

    private void notifyRateLimitObservers() {
        for (IRateLimitObserver observer : this.rateLimitObservers) {
            observer.onRateLimitChanged(this);
        }
    }

    public void addRateLimitObserver(@NotNull IRateLimitObserver observer) {
        this.rateLimitObservers.add(observer);
    }

    public void removeRateLimitObserver(@NotNull IRateLimitObserver observer) {
        this.rateLimitObservers.remove(observer);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        Object object = this.timerLock;
        synchronized (object) {
            if (this.timer != null) {
                this.timer.cancel();
                this.timer = null;
            }
        }
        this.rateLimitObservers.clear();
    }

    public static interface IRateLimitObserver {
        public void onRateLimitChanged(@NotNull RateLimiter var1);
    }
}

