/*
    BLUES - BD-Java emulation server

    Copyright (C) 2007-2025 GuinpinSoft inc <blues@makemkv.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
package blues;

import java.io.IOException;
import java.lang.reflect.Method;
import java.security.AccessController;

public class Main {

    public static final String VERSION = "1.4";

    public static void main(String[] args) {
        if (args.length < 1) {
            System.out.println("Missing command");
            usage();
            return;
        }

        try {
            setupJavaRuntime();
        } catch (Exception e1) {
            System.out.println("Unable to set up Java runtime, check command-line options");
            e1.printStackTrace();
            return;
        }

        if (args[0].equals("connect")) {
            if (args.length < 3) {
                System.out.println("Invalid arguments");
                return;
            }
            String addr = args[1];
            String port = args[2];
            runConnectSession(addr, port);
            System.exit(0);
            return;
        }

        if (args[0].equals("listen")) {
            if (args.length < 2) {
                System.out.println("Invalid arguments");
                return;
            }
            String port = args[1];
            runListenSession(port);
            System.exit(0);
            return;
        }

        if (args[0].equals("loop")) {
            if (args.length < 2) {
                System.out.println("Invalid arguments");
                return;
            }
            String port = args[1];
            while (true) {
                System.out.println("--- Session start ---");
                runListenSession(port);
                System.out.println("--- Session end ---\n\n\n");
                Server.reset();
                Cache.clear();
            }
        }

        if (args[0].equals("test")) {
            if (args.length < 2) {
                System.out.println("Invalid arguments");
                return;
            }
            try {
                String className = args[1];
                Class c = Class.forName(className);
                Method m = c.getMethod("testMain", new Class[]{ args.getClass() });
                String[] testArgs = new String[args.length-2];
                System.arraycopy(args, 2, testArgs , 0, testArgs.length);
                StringBuffer buf = new StringBuffer(256);
                buf.append("--- ");
                buf.append(className);
                buf.append(".main ");
                for (int i=0;i<testArgs.length;i++) {
                    if (i!=0) {
                        buf.append(',');
                    }
                    buf.append('"');
                    buf.append(testArgs[i]);
                    buf.append('"');
                }
                buf.append(" ---");
                System.out.println(buf.toString());
                m.invoke(null, new Object[]{ testArgs });
                System.out.println("--- end ---");
            } catch(Throwable e) {
                e.printStackTrace(System.out);
            }
        }
    }

    public static void usage() {
        System.out.println("\n\nblues.Main - blues server v"+VERSION+"\n\n");
        System.out.println("Usage:");
        System.out.println("\n" + Main.class.getName() + " Command [command args] \n");
        System.out.println("\n" + Main.class.getName() + " connect <ip_addr> <port>\n");
        System.out.println("\n" + Main.class.getName() + " listen <port>\n");
        System.exit(1);
    }

    static void runListenSession(String port) {
        Comm comm = new Comm(Integer.parseInt(port));
        out("blues.Main: Listening");
        try {
            comm.listenAndAccept();
        } catch (IOException e) {
            e.printStackTrace();
            try {
                comm.closeSession();
            } catch (IOException e1) {
            }
            return;
        }
        runSession(comm);
    }

    static void runConnectSession(String addr, String port) {
        Comm comm = new Comm(Integer.parseInt(port));
        out("blues.Main: Connecting to " + addr);
        try {
            comm.connect(addr);
        } catch (IOException e) {
            e.printStackTrace();
            return;
        }
        runSession(comm);
    }

    private static void runSession(Comm comm) {
        out("blues.Main: Connected, starting session");
        try {
            comm.runSession();
        } catch (IOException e) {
            e.printStackTrace();
        }
        try {
            comm.closeSession();
        } catch (IOException e) {
            e.printStackTrace();
        }
        out("blues.Main: Session is done!");
        out("blues.Main: exiting.");
        Server.close();
    }

    public static void out(String str) {
        System.out.println(str);
    }

    private static void setupJavaRuntime() throws Exception {

        // -Djava.security.policy=blues.policy
        AccessController.getContext().checkPermission(new java.security.AllPermission());

        Class clsModule;
        try {
            clsModule = Class.forName("java.lang.Module");
        } catch(ClassNotFoundException x) {
            clsModule = null;
        }

        if (null!=clsModule) {
            // --add-opens=java.base/java.lang=ALL-UNNAMED
            Method implAddOpens = clsModule.getDeclaredMethod("implAddOpensToAllUnnamed",String.class);
            implAddOpens.setAccessible(true);

            Method getModule =  Class.class.getMethod("getModule",null);
            Object javabase =  getModule.invoke(Object.class, null);

            // ObjectStreamClassPeer
            // --add-opens=java.base/java.io=ALL-UNNAMED
            implAddOpens.invoke(javabase, new Object[]{"java.io"});

            // JarEntryPeer
            // --add-opens=java.base/java.util=ALL-UNNAMED
            implAddOpens.invoke(javabase, new Object[]{"java.util"});
        }

        SManager.init();
    }
}
