/*
 * Decompiled with CFR 0.152.
 */
package org.tinymediamanager.core;

import com.madgag.gif.fmsware.GifDecoder;
import java.awt.Point;
import java.awt.image.BufferedImage;
import java.awt.image.BufferedImageOp;
import java.awt.image.ColorConvertOp;
import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.util.Collection;
import java.util.HexFormat;
import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageWriteParam;
import javax.imageio.ImageWriter;
import javax.imageio.stream.FileImageOutputStream;
import org.apache.commons.lang3.StringUtils;
import org.imgscalr.Scalr;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tinymediamanager.Globals;
import org.tinymediamanager.core.EmptyFileException;
import org.tinymediamanager.core.ImageUtils;
import org.tinymediamanager.core.InvalidFileTypeException;
import org.tinymediamanager.core.Settings;
import org.tinymediamanager.core.Utils;
import org.tinymediamanager.core.entities.MediaEntity;
import org.tinymediamanager.core.entities.MediaFile;
import org.tinymediamanager.core.tasks.ImageCacheTask;
import org.tinymediamanager.core.threading.ThreadUtils;
import org.tinymediamanager.core.threading.TmmTaskManager;
import org.tinymediamanager.scraper.http.Url;
import org.tinymediamanager.scraper.util.UrlUtil;

public class ImageCache {
    private static final Logger LOGGER = LoggerFactory.getLogger(ImageCache.class);
    private static final Path CACHE_DIR = Paths.get(Globals.CACHE_FOLDER + "/image", new String[0]);
    private static final char[] HEX_DIGITS = "0123456789ABCDEF".toCharArray();

    public static void init() {
        ImageCache.createSubdirs();
    }

    private ImageCache() {
        throw new IllegalAccessError();
    }

    private static void createSubdirs() {
        if (!Files.exists(CACHE_DIR, new LinkOption[0])) {
            try {
                Files.createDirectories(CACHE_DIR, new FileAttribute[0]);
            }
            catch (IOException e) {
                LOGGER.warn("Could not create cache dir '{}' - '{}'", (Object)CACHE_DIR, (Object)e.getMessage());
            }
        }
        for (char sub : HEX_DIGITS) {
            try {
                Path p = CACHE_DIR.resolve(Character.toString(sub));
                Files.createDirectories(p, new FileAttribute[0]);
            }
            catch (FileAlreadyExistsException p) {
            }
            catch (IOException e) {
                LOGGER.warn("Could not create cache sub dir '{}' - '{}'", (Object)Character.valueOf(sub), (Object)e.getMessage());
            }
        }
    }

    public static Path getCacheDir() {
        return CACHE_DIR;
    }

    public static String getMD5(String path) {
        try {
            if (path == null) {
                return null;
            }
            MessageDigest md = MessageDigest.getInstance("MD5");
            byte[] key = md.digest(path.getBytes());
            return HexFormat.of().withUpperCase().formatHex(key);
        }
        catch (Exception e) {
            LOGGER.debug("Failed to create cached filename for image: {} - {}", (Object)path, (Object)e.getMessage());
            return "";
        }
    }

    public static String getMD5WithSubfolder(String path) {
        String md5 = ImageCache.getMD5(path);
        if (StringUtils.isBlank((CharSequence)md5)) {
            return null;
        }
        return Paths.get(md5.substring(0, 1), md5).toString();
    }

    public static Path getAbsolutePath(MediaFile mf) {
        return CACHE_DIR.resolve(ImageCache.getMD5WithSubfolder(mf.getFileAsPath().toString()) + "." + mf.getExtension());
    }

    public static Path cacheImage(MediaFile mediaFile) throws Exception {
        return ImageCache.cacheImage(mediaFile, false);
    }

    public static void cacheImageAsync(MediaFile mediaFile) {
        if (!Settings.getInstance().isImageCache() || !mediaFile.isGraphic()) {
            return;
        }
        TmmTaskManager.getInstance().addImageCacheTask(new ImageCacheTask(mediaFile));
    }

    private static Path cacheImage(MediaFile mediaFile, boolean overwrite) throws Exception {
        if (!mediaFile.isGraphic()) {
            throw new InvalidFileTypeException(mediaFile.getFileAsPath());
        }
        Path originalFile = mediaFile.getFileAsPath();
        Path cachedFile = ImageCache.getCacheDir().resolve(ImageCache.getMD5WithSubfolder(originalFile.toString()) + "." + mediaFile.getExtension());
        if (overwrite || !Files.exists(cachedFile, new LinkOption[0])) {
            if (!Files.exists(originalFile, new LinkOption[0])) {
                throw new FileNotFoundException("unable to cache file: " + originalFile + "; file does not exist");
            }
            if (Files.size(originalFile) == 0L) {
                throw new EmptyFileException(originalFile);
            }
            BufferedImage originalImage = null;
            int retries = 5;
            do {
                try {
                    byte[] bytes = Files.readAllBytes(originalFile);
                    BasicFileAttributes view = Files.readAttributes(mediaFile.getFileAsPath(), BasicFileAttributes.class, new LinkOption[0]);
                    if (view.size() != (long)bytes.length) {
                        throw new IOException("File '{}' is not completely written");
                    }
                    GifDecoder decoder = new GifDecoder();
                    int status = decoder.read((InputStream)new ByteArrayInputStream(bytes));
                    if (status == 0 && decoder.getFrameCount() > 1) {
                        return ImageCache.writeAnimatedGif(originalFile, cachedFile);
                    }
                    originalImage = ImageUtils.createImage(bytes);
                    break;
                }
                catch (OutOfMemoryError e) {
                    LOGGER.debug("hit memory cap: {}", (Object)e.getMessage());
                }
                catch (IOException e) {
                    LOGGER.debug("{}", (Object)e.getMessage());
                }
                ThreadUtils.sleep(500L);
            } while (--retries > 0);
            if (originalImage == null) {
                throw new IOException("could not open original image to scale; probably due to memory limits");
            }
            int desiredWidth = ImageCache.calculateCacheImageWidth(originalImage);
            Point size = ImageUtils.calculateSize(desiredWidth, originalImage.getHeight(), originalImage.getWidth(), originalImage.getHeight(), true);
            BufferedImage scaledImage = null;
            retries = 5;
            while (true) {
                try {
                    switch (Settings.getInstance().getImageCacheType()) {
                        case BALANCED: {
                            scaledImage = Scalr.resize((BufferedImage)originalImage, (Scalr.Method)Scalr.Method.BALANCED, (Scalr.Mode)Scalr.Mode.FIT_EXACT, (int)size.x, (int)size.y, (BufferedImageOp[])new BufferedImageOp[0]);
                            break;
                        }
                        case QUALITY: {
                            scaledImage = Scalr.resize((BufferedImage)originalImage, (Scalr.Method)Scalr.Method.QUALITY, (Scalr.Mode)Scalr.Mode.FIT_EXACT, (int)size.x, (int)size.y, (BufferedImageOp[])new BufferedImageOp[0]);
                            break;
                        }
                        case ULTRA_QUALITY: {
                            scaledImage = Scalr.resize((BufferedImage)originalImage, (Scalr.Method)Scalr.Method.ULTRA_QUALITY, (Scalr.Mode)Scalr.Mode.FIT_EXACT, (int)size.x, (int)size.y, (BufferedImageOp[])new BufferedImageOp[0]);
                        }
                    }
                }
                catch (OutOfMemoryError e) {
                    LOGGER.debug("hit memory cap: {}", (Object)e.getMessage());
                    ThreadUtils.sleep(500L);
                    if (--retries > 0) continue;
                }
                break;
            }
            originalImage.flush();
            if (scaledImage == null) {
                throw new IOException("could not scale image; probably due to memory limits");
            }
            ImageWriter imgWrtr = null;
            ImageWriteParam imgWrtrPrm = null;
            if (ImageUtils.hasTransparentPixels(scaledImage)) {
                imgWrtr = ImageIO.getImageWritersByFormatName("png").next();
                imgWrtrPrm = imgWrtr.getDefaultWriteParam();
            } else {
                BufferedImage rgb = new BufferedImage(scaledImage.getWidth(), scaledImage.getHeight(), 1);
                ColorConvertOp xformOp = new ColorConvertOp(null);
                xformOp.filter(scaledImage, rgb);
                imgWrtr = ImageIO.getImageWritersByFormatName("jpg").next();
                imgWrtrPrm = imgWrtr.getDefaultWriteParam();
                imgWrtrPrm.setCompressionMode(2);
                imgWrtrPrm.setCompressionQuality(0.8f);
                scaledImage = rgb;
            }
            FileImageOutputStream output = new FileImageOutputStream(cachedFile.toFile());
            imgWrtr.setOutput(output);
            IIOImage image = new IIOImage(scaledImage, null, null);
            imgWrtr.write(null, image, imgWrtrPrm);
            imgWrtr.dispose();
            output.flush();
            output.close();
            scaledImage.flush();
            ThreadUtils.sleep(200L);
            if (!Files.exists(cachedFile, new LinkOption[0])) {
                throw new IOException("unable to cache file: " + originalFile);
            }
        }
        return cachedFile;
    }

    private static Path writeAnimatedGif(Path originalFile, Path cachedFile) throws IOException {
        Files.copy(originalFile, cachedFile, StandardCopyOption.REPLACE_EXISTING);
        return cachedFile;
    }

    private static int calculateCacheImageWidth(BufferedImage originalImage) {
        int desiredWidth = originalImage.getWidth();
        switch (Settings.getInstance().getImageCacheSize()) {
            case ORIGINAL: {
                break;
            }
            case BIG: {
                if ((float)originalImage.getWidth() / (float)originalImage.getHeight() > 1.0f) {
                    if (originalImage.getWidth() <= 1000) break;
                    desiredWidth = 1000;
                    break;
                }
                if (originalImage.getHeight() <= 1000) break;
                desiredWidth = 1000 * originalImage.getWidth() / originalImage.getHeight();
                break;
            }
            case SMALL: {
                if ((float)originalImage.getWidth() / (float)originalImage.getHeight() > 1.0f) {
                    if (originalImage.getWidth() <= 400) break;
                    desiredWidth = 400;
                    break;
                }
                if (originalImage.getHeight() <= 400) break;
                desiredWidth = 400 * originalImage.getWidth() / originalImage.getHeight();
            }
        }
        return desiredWidth;
    }

    public static void cacheImageSilently(Path path) {
        ImageCache.cacheImageSilently(new MediaFile(path));
    }

    public static void cacheImageSilently(MediaFile mediaFile) {
        ImageCache.cacheImageSilently(mediaFile, true);
    }

    public static void cacheImageSilently(MediaFile mediaFile, boolean overwrite) {
        if (!Settings.getInstance().isImageCache()) {
            return;
        }
        if (!mediaFile.isGraphic()) {
            return;
        }
        try {
            ImageCache.cacheImage(mediaFile, overwrite);
        }
        catch (Exception e) {
            LOGGER.debug("could not cache image: {}", (Object)e.getMessage());
        }
    }

    public static void invalidateCachedImage(Path path) {
        ImageCache.invalidateCachedImage(new MediaFile(path));
    }

    public static void invalidateCachedImage(MediaFile mediaFile) {
        if (!mediaFile.isGraphic()) {
            return;
        }
        Path path = mediaFile.getFileAsPath();
        Path cachedFile = ImageCache.getCacheDir().resolve(ImageCache.getMD5WithSubfolder(path.toAbsolutePath().toString()) + "." + Utils.getExtension(path));
        if (Files.exists(cachedFile, new LinkOption[0])) {
            Utils.deleteFileSafely(cachedFile);
        }
    }

    public static Path getCachedFile(String url) {
        Path cachedFile;
        if (url == null || url.isEmpty()) {
            return null;
        }
        String ext = UrlUtil.getExtension(url);
        if (ext.isEmpty()) {
            ext = "jpg";
        }
        if (Files.exists(cachedFile = ImageCache.getCacheDir().resolve(ImageCache.getMD5WithSubfolder(url) + "." + ext), new LinkOption[0])) {
            return cachedFile;
        }
        if (!Settings.getInstance().isImageCache()) {
            return null;
        }
        try {
            LOGGER.trace("downloading image to the image cache: {}", (Object)url);
            Url u = new Url(url);
            boolean ok = u.download(cachedFile);
            if (ok) {
                return cachedFile;
            }
        }
        catch (Exception e) {
            LOGGER.trace("Problem getting cached file for url {}", (Object)e.getMessage());
        }
        return null;
    }

    public static Path getCachedFile(Path path) {
        return ImageCache.getCachedFile(new MediaFile(path));
    }

    public static Path getCachedFile(MediaFile mediaFile) {
        if (mediaFile == null || !mediaFile.isGraphic()) {
            return null;
        }
        Path path = mediaFile.getFileAsPath().toAbsolutePath();
        Path cachedFile = ImageCache.getCacheDir().resolve(ImageCache.getMD5WithSubfolder(path.toString()) + "." + Utils.getExtension(path));
        if (Files.exists(cachedFile, new LinkOption[0])) {
            return cachedFile;
        }
        if (path.startsWith(CACHE_DIR.toAbsolutePath())) {
            return path;
        }
        if (!Settings.getInstance().isImageCache()) {
            return null;
        }
        try {
            return ImageCache.cacheImage(mediaFile);
        }
        catch (EmptyFileException e) {
            LOGGER.debug("failed to cache file (file is empty): {}", (Object)path);
        }
        catch (FileNotFoundException e) {
        }
        catch (Exception e) {
            LOGGER.debug("problem caching file: {}", (Object)e.getMessage());
        }
        return null;
    }

    public static boolean isImageCached(Path path) {
        if (!Settings.getInstance().isImageCache()) {
            return false;
        }
        Path cachedFile = CACHE_DIR.resolve(ImageCache.getMD5WithSubfolder(path.toString()) + "." + Utils.getExtension(path));
        return Files.exists(cachedFile, new LinkOption[0]);
    }

    public static void clearImageCacheForMediaEntity(MediaEntity entity) {
        ImageCache.clearImageCache(entity.getMediaFiles());
    }

    public static void clearImageCache(Collection<MediaFile> mediaFiles) {
        if (mediaFiles == null || mediaFiles.isEmpty()) {
            return;
        }
        mediaFiles.parallelStream().filter(MediaFile::isGraphic).forEach(mediaFile -> {
            Path file = ImageCache.getCachedFile(mediaFile);
            if (file != null) {
                Utils.deleteFileSafely(file);
            }
        });
    }

    public static enum CacheType {
        BALANCED,
        QUALITY,
        ULTRA_QUALITY;

    }

    public static enum CacheSize {
        SMALL,
        BIG,
        ORIGINAL;

    }
}

