﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/docdb/DocDBRequest.h>
#include <aws/docdb/DocDB_EXPORTS.h>
#include <aws/docdb/model/Tag.h>

#include <utility>

namespace Aws {
namespace DocDB {
namespace Model {

/**
 * <p>Represents the input to <a>CreateDBSubnetGroup</a>.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/docdb-2014-10-31/CreateDBSubnetGroupMessage">AWS
 * API Reference</a></p>
 */
class CreateDBSubnetGroupRequest : public DocDBRequest {
 public:
  AWS_DOCDB_API CreateDBSubnetGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateDBSubnetGroup"; }

  AWS_DOCDB_API Aws::String SerializePayload() const override;

 protected:
  AWS_DOCDB_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name for the subnet group. This value is stored as a lowercase
   * string.</p> <p>Constraints: Must contain no more than 255 letters, numbers,
   * periods, underscores, spaces, or hyphens. Must not be default.</p> <p>Example:
   * <code>mySubnetgroup</code> </p>
   */
  inline const Aws::String& GetDBSubnetGroupName() const { return m_dBSubnetGroupName; }
  inline bool DBSubnetGroupNameHasBeenSet() const { return m_dBSubnetGroupNameHasBeenSet; }
  template <typename DBSubnetGroupNameT = Aws::String>
  void SetDBSubnetGroupName(DBSubnetGroupNameT&& value) {
    m_dBSubnetGroupNameHasBeenSet = true;
    m_dBSubnetGroupName = std::forward<DBSubnetGroupNameT>(value);
  }
  template <typename DBSubnetGroupNameT = Aws::String>
  CreateDBSubnetGroupRequest& WithDBSubnetGroupName(DBSubnetGroupNameT&& value) {
    SetDBSubnetGroupName(std::forward<DBSubnetGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description for the subnet group.</p>
   */
  inline const Aws::String& GetDBSubnetGroupDescription() const { return m_dBSubnetGroupDescription; }
  inline bool DBSubnetGroupDescriptionHasBeenSet() const { return m_dBSubnetGroupDescriptionHasBeenSet; }
  template <typename DBSubnetGroupDescriptionT = Aws::String>
  void SetDBSubnetGroupDescription(DBSubnetGroupDescriptionT&& value) {
    m_dBSubnetGroupDescriptionHasBeenSet = true;
    m_dBSubnetGroupDescription = std::forward<DBSubnetGroupDescriptionT>(value);
  }
  template <typename DBSubnetGroupDescriptionT = Aws::String>
  CreateDBSubnetGroupRequest& WithDBSubnetGroupDescription(DBSubnetGroupDescriptionT&& value) {
    SetDBSubnetGroupDescription(std::forward<DBSubnetGroupDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon EC2 subnet IDs for the subnet group.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSubnetIds() const { return m_subnetIds; }
  inline bool SubnetIdsHasBeenSet() const { return m_subnetIdsHasBeenSet; }
  template <typename SubnetIdsT = Aws::Vector<Aws::String>>
  void SetSubnetIds(SubnetIdsT&& value) {
    m_subnetIdsHasBeenSet = true;
    m_subnetIds = std::forward<SubnetIdsT>(value);
  }
  template <typename SubnetIdsT = Aws::Vector<Aws::String>>
  CreateDBSubnetGroupRequest& WithSubnetIds(SubnetIdsT&& value) {
    SetSubnetIds(std::forward<SubnetIdsT>(value));
    return *this;
  }
  template <typename SubnetIdsT = Aws::String>
  CreateDBSubnetGroupRequest& AddSubnetIds(SubnetIdsT&& value) {
    m_subnetIdsHasBeenSet = true;
    m_subnetIds.emplace_back(std::forward<SubnetIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to be assigned to the subnet group.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateDBSubnetGroupRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateDBSubnetGroupRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_dBSubnetGroupName;

  Aws::String m_dBSubnetGroupDescription;

  Aws::Vector<Aws::String> m_subnetIds;

  Aws::Vector<Tag> m_tags;
  bool m_dBSubnetGroupNameHasBeenSet = false;
  bool m_dBSubnetGroupDescriptionHasBeenSet = false;
  bool m_subnetIdsHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace DocDB
}  // namespace Aws
