/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.deployment.util;

import io.smallrye.common.os.OS;
import io.smallrye.common.process.ProcessBuilder;
import io.smallrye.config.SmallRyeConfig;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.regex.Pattern;
import org.eclipse.microprofile.config.ConfigProvider;
import org.jboss.logging.Logger;

public final class ContainerRuntimeUtil {
    private static final Logger log = Logger.getLogger(ContainerRuntimeUtil.class);
    private static final String CONTAINER_EXECUTABLE = ((SmallRyeConfig)ConfigProvider.getConfig().unwrap(SmallRyeConfig.class)).getOptionalValue("quarkus.native.container-runtime", String.class).orElse(null);
    private static final Pattern PODMAN_PATTERN = Pattern.compile("^podman(?:\\.exe)? version.*", 32);
    private static final String CONTAINER_RUNTIME_SYS_PROP = "quarkus-local-container-runtime";
    private static final int MAX_ANTICIPATED_CHARACTERS_IN_DOCKER_INFO = 3000;

    private ContainerRuntimeUtil() {
    }

    public static ContainerRuntime detectContainerRuntime() {
        return ContainerRuntimeUtil.detectContainerRuntime(true, new ContainerRuntime[0]);
    }

    public static ContainerRuntime detectContainerRuntime(ContainerRuntime ... orderToCheckRuntimes) {
        return ContainerRuntimeUtil.detectContainerRuntime(true, orderToCheckRuntimes);
    }

    public static ContainerRuntime detectContainerRuntime(boolean required, ContainerRuntime ... orderToCheckRuntimes) {
        return ContainerRuntimeUtil.detectContainerRuntime(required, orderToCheckRuntimes != null && orderToCheckRuntimes.length > 0 ? Arrays.asList(orderToCheckRuntimes) : List.of(ContainerRuntime.DOCKER, ContainerRuntime.PODMAN));
    }

    public static ContainerRuntime detectContainerRuntime(boolean required, boolean silent, ContainerRuntime ... orderToCheckRuntimes) {
        return ContainerRuntimeUtil.detectContainerRuntime(required, silent, orderToCheckRuntimes != null && orderToCheckRuntimes.length > 0 ? Arrays.asList(orderToCheckRuntimes) : List.of(ContainerRuntime.DOCKER, ContainerRuntime.PODMAN));
    }

    public static ContainerRuntime detectContainerRuntime(boolean required, List<ContainerRuntime> orderToCheckRuntimes) {
        return ContainerRuntimeUtil.detectContainerRuntime(required, false, orderToCheckRuntimes);
    }

    public static ContainerRuntime detectContainerRuntime(boolean required, boolean silent, List<ContainerRuntime> orderToCheckRuntimes) {
        ContainerRuntime containerRuntime = ContainerRuntimeUtil.loadContainerRuntimeFromSystemProperty();
        if (containerRuntime != null && orderToCheckRuntimes.contains((Object)containerRuntime)) {
            return containerRuntime;
        }
        ContainerRuntime containerRuntimeEnvironment = ContainerRuntimeUtil.getContainerRuntimeEnvironment(orderToCheckRuntimes);
        if (containerRuntimeEnvironment == ContainerRuntime.UNAVAILABLE) {
            ContainerRuntimeUtil.storeContainerRuntimeInSystemProperty(ContainerRuntime.UNAVAILABLE);
            if (required) {
                throw new IllegalStateException("No container runtime was found. Make sure you have either Docker or Podman installed in your environment.");
            }
            return ContainerRuntime.UNAVAILABLE;
        }
        containerRuntime = ContainerRuntimeUtil.fullyResolveContainerRuntime(containerRuntimeEnvironment, silent);
        ContainerRuntimeUtil.storeContainerRuntimeInSystemProperty(containerRuntime);
        return containerRuntime;
    }

    private static ContainerRuntime getContainerRuntimeEnvironment(List<ContainerRuntime> orderToCheckRuntimes) {
        ArrayList runtimesToCheck = new ArrayList(orderToCheckRuntimes.stream().distinct().toList());
        runtimesToCheck.retainAll(List.of(ContainerRuntime.DOCKER, ContainerRuntime.PODMAN));
        if (CONTAINER_EXECUTABLE != null) {
            Optional<ContainerRuntime> runtime = runtimesToCheck.stream().filter(containerRuntime -> CONTAINER_EXECUTABLE.trim().equalsIgnoreCase(containerRuntime.getExecutableName())).findFirst().filter(r -> {
                String versionOutput = ContainerRuntimeUtil.getVersionOutputFor(r);
                return switch (r.ordinal()) {
                    case 0, 1 -> versionOutput.contains("Docker version");
                    case 4, 5 -> PODMAN_PATTERN.matcher(versionOutput).matches();
                    default -> false;
                };
            });
            if (runtime.isPresent()) {
                return runtime.get();
            }
            log.warn((Object)"quarkus.native.container-runtime config property must be set to either podman or docker and the executable must be available. Ignoring it.");
        }
        for (ContainerRuntime runtime : runtimesToCheck) {
            String versionOutput = ContainerRuntimeUtil.getVersionOutputFor(runtime);
            switch (runtime.ordinal()) {
                case 0: 
                case 1: {
                    boolean dockerAvailable = versionOutput.contains("Docker version");
                    if (!dockerAvailable) break;
                    return PODMAN_PATTERN.matcher(versionOutput).matches() ? ContainerRuntime.PODMAN : ContainerRuntime.DOCKER;
                }
                case 4: 
                case 5: {
                    if (!PODMAN_PATTERN.matcher(versionOutput).matches()) break;
                    return ContainerRuntime.PODMAN;
                }
            }
        }
        return ContainerRuntime.UNAVAILABLE;
    }

    private static ContainerRuntime fullyResolveContainerRuntime(ContainerRuntime containerRuntimeEnvironment, boolean silent) {
        String execName = containerRuntimeEnvironment.getExecutableName();
        try {
            return (ContainerRuntime)((Object)ProcessBuilder.newBuilder((String)execName).arguments(new String[]{"info"}).output().gatherOnFail(true).processWith(br -> {
                String line;
                boolean rootless = false;
                boolean isInWindowsWSL = false;
                boolean isDocker = containerRuntimeEnvironment.isDocker();
                while ((line = br.readLine()) != null) {
                    if (isDocker) {
                        if (line.trim().equals("rootless") || line.contains("Docker Desktop") || line.contains("desktop-linux")) {
                            rootless = true;
                        }
                    } else if (line.trim().equals("rootless: true")) {
                        rootless = true;
                    }
                    if (OS.current() != OS.LINUX || !isDocker || !line.trim().contains("WSL")) continue;
                    isInWindowsWSL = true;
                }
                if (rootless) {
                    if (isInWindowsWSL) {
                        return ContainerRuntime.WSL_ROOTLESS;
                    }
                    return containerRuntimeEnvironment == ContainerRuntime.DOCKER ? ContainerRuntime.DOCKER_ROOTLESS : ContainerRuntime.PODMAN_ROOTLESS;
                }
                if (isInWindowsWSL) {
                    return ContainerRuntime.WSL;
                }
                return containerRuntimeEnvironment;
            }).run());
        }
        catch (Exception e) {
            if (!silent) {
                log.warnf("Command \"%s\" failed. Rootless container runtime detection might not be reliable or the container service is not running at all.", (Object)execName);
            }
            return ContainerRuntime.UNAVAILABLE;
        }
    }

    private static ContainerRuntime loadContainerRuntimeFromSystemProperty() {
        String runtime = System.getProperty(CONTAINER_RUNTIME_SYS_PROP);
        if (runtime == null) {
            return null;
        }
        ContainerRuntime containerRuntime = ContainerRuntime.of(runtime);
        if (containerRuntime == null) {
            log.warnf("System property %s contains an unknown value %s. Ignoring it.", (Object)CONTAINER_RUNTIME_SYS_PROP, (Object)runtime);
        }
        return containerRuntime;
    }

    private static void storeContainerRuntimeInSystemProperty(ContainerRuntime containerRuntime) {
        System.setProperty(CONTAINER_RUNTIME_SYS_PROP, containerRuntime.name());
    }

    private static String getVersionOutputFor(ContainerRuntime containerRuntime) {
        String execName = containerRuntime.getExecutableName();
        try {
            return (String)ProcessBuilder.newBuilder((String)execName).arguments(new String[]{"--version"}).output().gatherOnFail(true).toSingleString(16384).run();
        }
        catch (Throwable t) {
            log.debugf(t, "Failure to read version output from %s", (Object)execName);
            return "";
        }
    }

    public static enum ContainerRuntime {
        DOCKER("docker", false),
        DOCKER_ROOTLESS("docker", true),
        WSL("docker", false),
        WSL_ROOTLESS("docker", false),
        PODMAN("podman", false),
        PODMAN_ROOTLESS("podman", true),
        UNAVAILABLE(null, false);

        private final String executableName;
        private final boolean rootless;

        private ContainerRuntime(String executableName, boolean rootless) {
            this.executableName = executableName;
            this.rootless = rootless;
        }

        public String getExecutableName() {
            if (this == UNAVAILABLE) {
                throw new IllegalStateException("Cannot get an executable name when no container runtime is available");
            }
            return this.executableName;
        }

        public boolean isDocker() {
            return this == DOCKER || this == DOCKER_ROOTLESS || this == WSL || this == WSL_ROOTLESS;
        }

        public boolean isPodman() {
            return this == PODMAN || this == PODMAN_ROOTLESS;
        }

        public boolean isInWindowsWSL() {
            return this == WSL || this == WSL_ROOTLESS;
        }

        public boolean isRootless() {
            return this.rootless;
        }

        public boolean isUnavailable() {
            return this == UNAVAILABLE;
        }

        public static ContainerRuntime of(String value) {
            for (ContainerRuntime containerRuntime : ContainerRuntime.values()) {
                if (!containerRuntime.name().equalsIgnoreCase(value)) continue;
                return containerRuntime;
            }
            return null;
        }
    }
}

