"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PRIVATE_TENANTS = exports.GLOBAL_TENANTS = void 0;
exports.addTenantParameterToResolvedShortLink = addTenantParameterToResolvedShortLink;
exports.isValidTenant = isValidTenant;
exports.resolve = resolve;
exports.resolveTenant = resolveTenant;
var _lodash = require("lodash");
var _std = require("@osd/std");
var _common = require("../../common");
var _router = require("../../../../src/core/server/http/router");
var _short_url_routes = require("../../../../src/plugins/share/common/short_url_routes");
/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */

const PRIVATE_TENANTS = exports.PRIVATE_TENANTS = [_common.PRIVATE_TENANT_SYMBOL, 'private', 'Private'];
const GLOBAL_TENANTS = exports.GLOBAL_TENANTS = ['global', _common.GLOBAL_TENANT_SYMBOL, 'Global'];
/**
 * Resovles the tenant the user is using.
 *
 * @param request OpenSearchDashboards request.
 * @param username
 * @param roles
 * @param availableTenants
 * @param config security plugin config.
 * @param cookie cookie extracted from the request. The cookie should have been parsed by AuthenticationHandler.
 * pass it as parameter instead of extracting again.
 * @param multitenancyEnabled
 * @param privateTenantEnabled
 * @param defaultTenant
 *
 * @returns user preferred tenant of the request.
 */
function resolveTenant({
  request,
  username,
  roles,
  availableTenants,
  config,
  cookie,
  multitenancyEnabled,
  privateTenantEnabled,
  defaultTenant
}) {
  var _request$url, _request$url2, _request$url3, _config$multitenancy, _config$multitenancy2;
  const DEFAULT_READONLY_ROLES = ['kibana_read_only'];
  let selectedTenant;
  const securityTenant_ = request === null || request === void 0 || (_request$url = request.url) === null || _request$url === void 0 || (_request$url = _request$url.searchParams) === null || _request$url === void 0 ? void 0 : _request$url.get('securityTenant_');
  const securitytenant = request === null || request === void 0 || (_request$url2 = request.url) === null || _request$url2 === void 0 || (_request$url2 = _request$url2.searchParams) === null || _request$url2 === void 0 ? void 0 : _request$url2.get('securitytenant');
  // eslint-disable-next-line @typescript-eslint/naming-convention
  const security_tenant = request === null || request === void 0 || (_request$url3 = request.url) === null || _request$url3 === void 0 || (_request$url3 = _request$url3.searchParams) === null || _request$url3 === void 0 ? void 0 : _request$url3.get('security_tenant');
  if (securityTenant_) {
    selectedTenant = securityTenant_;
  } else if (securitytenant) {
    selectedTenant = securitytenant;
  } else if (security_tenant) {
    selectedTenant = security_tenant;
  } else if (request.headers.securitytenant || request.headers.securityTenant_) {
    selectedTenant = request.headers.securitytenant ? request.headers.securitytenant : request.headers.securityTenant_;
  } else if (isValidTenant(cookie.tenant)) {
    selectedTenant = cookie.tenant;
  } else if (defaultTenant && multitenancyEnabled) {
    selectedTenant = defaultTenant;
  } else {
    selectedTenant = undefined;
  }
  const isReadonly = roles === null || roles === void 0 ? void 0 : roles.some(role => {
    var _config$readonly_mode;
    return ((_config$readonly_mode = config.readonly_mode) === null || _config$readonly_mode === void 0 ? void 0 : _config$readonly_mode.roles.includes(role)) || DEFAULT_READONLY_ROLES.includes(role);
  });
  const preferredTenants = (_config$multitenancy = config.multitenancy) === null || _config$multitenancy === void 0 ? void 0 : _config$multitenancy.tenants.preferred;
  const globalTenantEnabled = (_config$multitenancy2 = config.multitenancy) === null || _config$multitenancy2 === void 0 ? void 0 : _config$multitenancy2.tenants.enable_global;
  return resolve(username, selectedTenant, preferredTenants, availableTenants, globalTenantEnabled, multitenancyEnabled, privateTenantEnabled);
}
function resolve(username, requestedTenant, preferredTenants, availableTenants,
// is an object like { tenant_name_1: true, tenant_name_2: false, ... }
globalTenantEnabled, multitenancyEnabled, privateTenantEnabled) {
  const availableTenantsClone = (0, _lodash.cloneDeep)(availableTenants);
  delete availableTenantsClone[username];
  if (!globalTenantEnabled && !privateTenantEnabled && (0, _lodash.isEmpty)(availableTenantsClone)) {
    return undefined;
  }
  if (!multitenancyEnabled) {
    if (!globalTenantEnabled) {
      return undefined;
    }
    return _common.GLOBAL_TENANT_SYMBOL;
  }
  if (isValidTenant(requestedTenant)) {
    requestedTenant = requestedTenant;
    if (requestedTenant in availableTenants) {
      return requestedTenant;
    }
    if (privateTenantEnabled && username in availableTenants && PRIVATE_TENANTS.indexOf(requestedTenant) > -1) {
      return _common.PRIVATE_TENANT_SYMBOL;
    }
    if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(requestedTenant) > -1) {
      return _common.GLOBAL_TENANT_SYMBOL;
    }
  }
  if (preferredTenants && !(0, _lodash.isEmpty)(preferredTenants)) {
    for (const element of preferredTenants) {
      const tenant = element.toLowerCase();
      if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(tenant) > -1) {
        return _common.GLOBAL_TENANT_SYMBOL;
      }
      if (privateTenantEnabled && PRIVATE_TENANTS.indexOf(tenant) > -1 && username in availableTenants) {
        return _common.PRIVATE_TENANT_SYMBOL;
      }
      if (tenant in availableTenants) {
        return tenant;
      }
    }
  }
  if (globalTenantEnabled) {
    return _common.GLOBAL_TENANT_SYMBOL;
  }
  if (privateTenantEnabled) {
    return _common.PRIVATE_TENANT_SYMBOL;
  }

  /**
   * Fall back to the first tenant in the available tenants
   * Under the condition of enabling multitenancy, if the user has disabled both 'Global' and 'Private' tenants:
   * it will remove the default global tenant key for custom tenant.
   */
  if (Object.keys(availableTenantsClone).length > 1 && availableTenantsClone.hasOwnProperty(_common.globalTenantName)) {
    delete availableTenantsClone[_common.globalTenantName];
  }
  return (0, _lodash.findKey)(availableTenantsClone, () => true);
}

/**
 * Return true if tenant parameter is a valid tenant.
 *
 * Note: empty string '' is valid, which means global tenant.
 *
 * @param tenant
 */
function isValidTenant(tenant) {
  return tenant !== undefined && tenant !== null;
}

/**
 * If multitenancy is enabled & the URL entered starts with /goto,
 * We will modify the rawResponse to add a new parameter to the URL, the security_tenant (or value for tenant when in multitenancy)
 * With the security_tenant added, the resolved short URL now contains the security_tenant information (so the short URL retains the tenant information).
 **/

function addTenantParameterToResolvedShortLink(request) {
  if (request.url.pathname.startsWith(`${_short_url_routes.GOTO_PREFIX}/`)) {
    const rawRequest = (0, _router.ensureRawRequest)(request);
    const rawResponse = rawRequest.response;
    const responsePath = rawResponse.headers.location;

    // Make sure the request really should redirect
    if (rawResponse.headers.location && !responsePath.includes('security_tenant') && request.headers.securitytenant) {
      // Mutating the headers toolkit.next({headers: ...}) logs a warning about headers being overwritten
      rawResponse.headers.location = (0, _std.modifyUrl)(responsePath, parts => {
        if (parts.query.security_tenant === undefined) {
          parts.query.security_tenant = request.headers.securitytenant;
        }
      });
    }
  }
  return request;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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