/*
 * Decompiled with CFR 0.152.
 */
package org.jitsi.videobridge.util;

import java.time.Duration;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.LongAdder;
import org.jetbrains.annotations.NotNull;
import org.jitsi.utils.OrderedJsonObject;
import org.jitsi.utils.logging2.Logger;
import org.jitsi.utils.logging2.LoggerImpl;
import org.jitsi.utils.stats.RateTracker;
import org.jitsi.videobridge.util.ByteBufferPool;
import org.json.simple.JSONArray;

class PartitionedByteBufferPool {
    private static final int NUM_PARTITIONS = 8;
    private static final int INITIAL_SIZE = 10;
    private static final boolean ACCEPT_SMALL_BUFFERS = false;
    private static final Logger logger = new LoggerImpl(PartitionedByteBufferPool.class.getName());
    private final Partition[] partitions = new Partition[8];
    private boolean enableStatistics = false;
    private final int defaultBufferSize;

    PartitionedByteBufferPool(int defaultBufferSize) {
        this.defaultBufferSize = defaultBufferSize;
        for (int i = 0; i < 8; ++i) {
            this.partitions[i] = new Partition(i, 10);
        }
        logger.info((Object)("Initialized a new " + this.getClass().getSimpleName() + " with 8 partitions."));
    }

    void enableStatistics(boolean enable) {
        this.enableStatistics = enable;
    }

    private Partition getPartition() {
        return this.partitions[ThreadLocalRandom.current().nextInt(8)];
    }

    byte[] getBuffer(int size) {
        return this.getPartition().getBuffer(size);
    }

    void returnBuffer(byte[] buf) {
        this.getPartition().returnBuffer(buf);
    }

    OrderedJsonObject getStats() {
        OrderedJsonObject stats = new OrderedJsonObject();
        stats.put((Object)"default_size", (Object)this.defaultBufferSize);
        long requests = 0L;
        long returns = 0L;
        long allocations = 0L;
        long storedBytes = 0L;
        JSONArray partitionStats = new JSONArray();
        for (Partition p : this.partitions) {
            requests += p.numRequests.sum();
            returns += p.numReturns.sum();
            allocations += p.numAllocations.sum();
            storedBytes += p.storedBytes.get();
            partitionStats.add((Object)p.getStatsJson());
        }
        stats.put((Object)"num_requests", (Object)requests);
        stats.put((Object)"num_returns", (Object)returns);
        stats.put((Object)"num_allocations", (Object)allocations);
        stats.put((Object)"stored_bytes", (Object)storedBytes);
        stats.put((Object)"allocation_percent", (Object)(100.0 * (double)allocations / (double)Math.max(1L, requests)));
        stats.put((Object)"partitions", (Object)partitionStats);
        return stats;
    }

    long getNumAllocations() {
        long allocations = 0L;
        for (int i = 0; i < 8; ++i) {
            allocations += this.partitions[i].numAllocations.sum();
        }
        return allocations;
    }

    long getStoredBytes() {
        long storedBytes = 0L;
        for (int i = 0; i < 8; ++i) {
            storedBytes += this.partitions[i].storedBytes.get();
        }
        return storedBytes;
    }

    private class Partition {
        private final LinkedBlockingQueue<byte[]> pool = new LinkedBlockingQueue();
        private final int id;
        private final LongAdder numNoAllocationNeeded = new LongAdder();
        private final LongAdder numAllocations = new LongAdder();
        private final LongAdder numEmptyPoolAllocations = new LongAdder();
        private final LongAdder numWrongSizeAllocations = new LongAdder();
        private final LongAdder numRequests = new LongAdder();
        private final LongAdder numReturns = new LongAdder();
        private final LongAdder numSmallReturns = new LongAdder();
        private final LongAdder numSmallBuffersDiscarded = new LongAdder();
        private final LongAdder numLargeRequests = new LongAdder();
        private final AtomicLong storedBytes = new AtomicLong();
        private final RateTracker requestRate = new RateTracker(Duration.ofSeconds(10L), Duration.ofMillis(100L));
        private final RateTracker returnRate = new RateTracker(Duration.ofSeconds(10L), Duration.ofMillis(100L));

        Partition(int id, int initialSize) {
            this.id = id;
            for (int i = 0; i < initialSize; ++i) {
                this.pool.add(new byte[PartitionedByteBufferPool.this.defaultBufferSize]);
            }
        }

        private byte[] getBuffer(int requiredSize) {
            byte[] buf;
            if (PartitionedByteBufferPool.this.enableStatistics) {
                this.numRequests.increment();
                this.requestRate.update(1L);
                if (requiredSize > PartitionedByteBufferPool.this.defaultBufferSize) {
                    this.numLargeRequests.increment();
                }
            }
            if ((buf = this.pool.poll()) == null) {
                buf = new byte[Math.max(PartitionedByteBufferPool.this.defaultBufferSize, requiredSize)];
                if (PartitionedByteBufferPool.this.enableStatistics) {
                    this.numAllocations.increment();
                    this.numEmptyPoolAllocations.increment();
                }
            } else if (buf.length < requiredSize) {
                if (ByteBufferPool.bookkeepingEnabled()) {
                    logger.info((Object)("Needed buffer of size " + requiredSize + ", got size " + buf.length + " retrying"));
                }
                if (buf.length >= PartitionedByteBufferPool.this.defaultBufferSize) {
                    this.pool.offer(buf);
                } else if (PartitionedByteBufferPool.this.enableStatistics) {
                    this.storedBytes.getAndAdd(-buf.length);
                    this.numSmallBuffersDiscarded.increment();
                }
                buf = new byte[Math.max(PartitionedByteBufferPool.this.defaultBufferSize, requiredSize)];
                if (PartitionedByteBufferPool.this.enableStatistics) {
                    this.numAllocations.increment();
                    this.numWrongSizeAllocations.increment();
                }
            } else if (PartitionedByteBufferPool.this.enableStatistics) {
                this.storedBytes.getAndAdd(-buf.length);
                this.numNoAllocationNeeded.increment();
            }
            return buf;
        }

        private void returnBuffer(@NotNull byte[] buf) {
            if (PartitionedByteBufferPool.this.enableStatistics) {
                this.numReturns.increment();
                this.returnRate.update(1L);
            }
            if (buf.length < PartitionedByteBufferPool.this.defaultBufferSize) {
                if (PartitionedByteBufferPool.this.enableStatistics) {
                    this.numSmallReturns.increment();
                }
            } else {
                this.pool.offer(buf);
                if (PartitionedByteBufferPool.this.enableStatistics) {
                    this.storedBytes.getAndAdd(buf.length);
                }
            }
        }

        private OrderedJsonObject getStatsJson() {
            long now = System.currentTimeMillis();
            OrderedJsonObject stats = new OrderedJsonObject();
            stats.put((Object)"id", (Object)this.id);
            long numRequestsSum = this.numRequests.sum();
            long numAllocationsSum = this.numAllocations.sum();
            stats.put((Object)"num_requests", (Object)numRequestsSum);
            stats.put((Object)"num_returns", (Object)this.numReturns.sum());
            stats.put((Object)"requests_rate_rps", (Object)this.requestRate.getRate(now));
            stats.put((Object)"returns_rate_rps", (Object)this.returnRate.getRate(now));
            stats.put((Object)"current_size", (Object)this.pool.size());
            stats.put((Object)"stored_bytes", (Object)this.storedBytes.get());
            stats.put((Object)"num_allocations", (Object)numAllocationsSum);
            stats.put((Object)"num_allocations_empty_pool", (Object)this.numEmptyPoolAllocations.sum());
            stats.put((Object)"num_allocations_wrong_size", (Object)this.numWrongSizeAllocations.sum());
            stats.put((Object)"num_no_allocation_needed", (Object)this.numNoAllocationNeeded.sum());
            stats.put((Object)"allocation_percent", (Object)(100.0 * (double)numAllocationsSum / (double)Math.max(1L, numRequestsSum)));
            stats.put((Object)"num_small_returns", (Object)this.numSmallReturns.sum());
            stats.put((Object)"num_large_requests", (Object)this.numLargeRequests.sum());
            stats.put((Object)"num_small_discarded", (Object)this.numSmallBuffersDiscarded.sum());
            return stats;
        }
    }
}

