/*
    BLUES - BD-Java emulation server

    Copyright (C) 2007-2025 GuinpinSoft inc <blues@makemkv.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
package impl.java.io;

import java.io.InvalidClassException;
import java.io.ObjectStreamClass;
import java.io.ObjectStreamField;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.security.AccessController;
import java.security.PrivilegedAction;

import impl.java.lang.PeerAccess;
import impl.java.lang.PeerManager;

import java.lang.reflect.Field;

/*
 * This code assumes that java.io.ObjectStreamClass is implemented in a certain way.
 * To date this assumption holds for all implementations.
 */

public class ObjectStreamClassPeer {

    private static final byte[] bytearray = new byte[0];
    private static final Object[] objectarray = new Object[0];
    private static final ObjectStreamField[] osfarray = new ObjectStreamField[0];

    private static PeerManager<ObjectStreamClass,ObjectStreamClassPeer>	objectStreamClassPeerManager;
    private static PeerManager<Object,Reflector>	reflectorPeerManager;
    private static PeerAccess<ObjectStreamClassPeer,jail.java.io.ObjectStreamClass> objectStreamClassPeerAccess;

    public static class Reflector {
        private Object reflector;

        private Method getPrimFieldValuesMethod;
        private Method setPrimFieldValuesMethod;
        private Method getObjFieldValuesMethod;
        private Method setObjFieldValuesMethod;

        public Reflector(Object reflector) {
            this.reflector = reflector;

            getPrimFieldValuesMethod =  getMethod(reflector.getClass(),"getPrimFieldValues",new Class[]{Object.class,bytearray.getClass()},Void.TYPE);
            setPrimFieldValuesMethod =  getMethod(reflector.getClass(),"setPrimFieldValues",new Class[]{Object.class,bytearray.getClass()},Void.TYPE);
            getObjFieldValuesMethod =  getMethod(reflector.getClass(),"getObjFieldValues",new Class[]{Object.class,objectarray.getClass()},Void.TYPE);
            setObjFieldValuesMethod =  getMethod(reflector.getClass(),"setObjFieldValues",new Class[]{Object.class,objectarray.getClass()},Void.TYPE);
        }

        public void getPrimFieldValues(Object obj,byte[] buf) {
            try {
                getPrimFieldValuesMethod.invoke(reflector, obj,buf);
            } catch (Exception e) {
                throw new RuntimeException("Unable to call Reflector.getPrimFieldValues",e);
            }
        }

        public void setPrimFieldValues(Object obj,byte[] buf) {
            try {
                setPrimFieldValuesMethod.invoke(reflector, obj,buf);
            } catch (Exception e) {
                throw new RuntimeException("Unable to call Reflector.setPrimFieldValues",e);
            }
        }

        public void getObjFieldValues(Object obj,Object[] vals) {
            try {
                getObjFieldValuesMethod.invoke(reflector, obj, vals);
            } catch (Exception e) {
                throw new RuntimeException("Unable to call Reflector.getObjFieldValues",e);
            }
        }

        public void setObjFieldValues(Object obj,Object[] vals) {
            try {
                setObjFieldValuesMethod.invoke(reflector, obj, vals);
            } catch (Exception e) {
                throw new RuntimeException("Unable to call Reflector.setObjFieldValues",e);
            }
        }
    }

    public static class ObjectStreamFieldData
    {
        public String name;
        public String signature;
        public Field field;
        public int offset;

        public ObjectStreamFieldData(String name,String signature,Field field,int offset) {
            this.name = name;
            this.signature = signature;
            this.field = field;
            this.offset = offset;
        }
    }

    static boolean initialized = false;
    static Method newInstanceMethod;
    static Method getReflectorMethod;

    static Constructor ObjectStreamFieldConstructorNS;
    static Constructor ObjectStreamFieldConstructorF;
    static Method setOffsetMethod;
    static Method hasStaticInitializerMethod;

    ObjectStreamClass	peer;

    public static synchronized void setPeerAccess(PeerAccess<ObjectStreamClassPeer,jail.java.io.ObjectStreamClass> a) {
        objectStreamClassPeerAccess = a;
    }

    public static synchronized void initStatic() {
        if (initialized) return;

        objectStreamClassPeerManager = new PeerManager<ObjectStreamClass,ObjectStreamClassPeer>();
        objectStreamClassPeerManager.setPeerAccess(new PeerAccess<ObjectStreamClass, ObjectStreamClassPeer>() {
            public ObjectStreamClassPeer construct(ObjectStreamClass peer) {
                return new ObjectStreamClassPeer(peer);
            }
            public ObjectStreamClass getPeer(ObjectStreamClassPeer obj) {
                return obj.peer;
            }
        });

        reflectorPeerManager = new PeerManager<Object,Reflector>();
        reflectorPeerManager.setPeerAccess(new PeerAccess<Object,Reflector>(){
            public Reflector construct(Object peer) {
                return new Reflector(peer);
            }
            public Object getPeer(Reflector obj) {
                return obj.reflector;
            }
        });

        Class cl = java.io.ObjectStreamClass.class;

        newInstanceMethod = getMethod(cl,"newInstance",null,Object.class);
        getReflectorMethod = getMethod(cl,"getReflector",new Class[]{osfarray.getClass(),java.io.ObjectStreamClass.class},null);
        hasStaticInitializerMethod =  getMethod(cl,"hasStaticInitializer",new Class[]{Class.class},Boolean.TYPE);

        cl = java.io.ObjectStreamField.class;
        ObjectStreamFieldConstructorNS = getConstructor(cl,new Class[]{String.class,String.class,Boolean.TYPE});
        ObjectStreamFieldConstructorF = getConstructor(cl,new Class[]{Field.class,Boolean.TYPE,Boolean.TYPE});
        setOffsetMethod = getMethod(cl,"setOffset",new Class[]{Integer.TYPE},Void.TYPE);

        initialized = true;
    }

    static Method getMethod(final Class cl, final String name, final Class[] argTypes, Class returnType) {
        Method meth = (Method) AccessController.doPrivileged(new PrivilegedAction() {
            @Override
            public Object run() {
                try {
                    Method m = cl.getDeclaredMethod(name, argTypes);
                    m.setAccessible(true);
                    return m;
                } catch (NoSuchMethodException e) {
                    throw new RuntimeException("No method in Java implementation - " + cl.getName() + "." + name, e);
                } catch (SecurityException e) {
                    throw new RuntimeException(e);
                }
            }
        });

        if ((meth == null) || (returnType != null) && (meth.getReturnType() != returnType)) {
            throw new RuntimeException(
                    "Invalid method signature in Java implementation - " + cl.getName() + "." + name);
        }

        return meth;
    }

    static Constructor getConstructor(final Class cl, final Class[] argTypes) {
        Constructor meth = (Constructor) AccessController.doPrivileged(new PrivilegedAction() {
            @Override
            public Object run() {
                try {
                    Constructor m = cl.getDeclaredConstructor(argTypes);
                    m.setAccessible(true);
                    return m;
                } catch (NoSuchMethodException e) {
                    throw new RuntimeException("No method in Java implementation - " + cl.getName() + "." + ".<init>", e);
                } catch (SecurityException e) {
                    throw new RuntimeException(e);
                }
            }
        });

        return meth;
    }

    public static java.io.ObjectStreamClass fromJail(jail.java.io.ObjectStreamClass j) {
        ObjectStreamClassPeer p = objectStreamClassPeerAccess.getPeer(j);
        return p.peer;
    }

    public static ObjectStreamClassPeer lookup(Class cl) {
        initStatic();
        if (cl==null) {
            return null;
        }
        return objectStreamClassPeerManager.toJail(ObjectStreamClass.lookup(cl));
    }

    private ObjectStreamClassPeer(ObjectStreamClass peer) {
        this.peer = peer;
    }

    public Object newInstance() throws InstantiationException, InvocationTargetException, UnsupportedOperationException,
            IllegalAccessException {
        return newInstanceMethod.invoke(peer, new Object[0]);
    }

    private static java.io.ObjectStreamField constructObjectStreamField(ObjectStreamFieldData field) {
        java.io.ObjectStreamField f;

        try {
            if (field.field!=null) {
                f = (java.io.ObjectStreamField)ObjectStreamFieldConstructorF.newInstance(field.field,new Boolean(false),new Boolean(false));
            } else {
                f = (java.io.ObjectStreamField)ObjectStreamFieldConstructorNS.newInstance(field.name,field.signature,new Boolean(false));
            }
            setOffsetMethod.invoke(f, new Integer(field.offset));
        } catch (Exception e) {
            throw new RuntimeException("Unable to construct java.io.ObjectStreamField",e);
        }

        return f;
    }

    public Reflector getReflector(ObjectStreamFieldData[] fields) throws InvalidClassException {
        initStatic();

        java.io.ObjectStreamField[] fieldsN;
        if (fields == null) {
            fieldsN = null;
        } else {
            fieldsN = new java.io.ObjectStreamField[fields.length];
            for (int i=0;i<fields.length;i++) {
                fieldsN[i] = constructObjectStreamField(fields[i]);
            }
        }

        Object reflectorObj;
        try {
            reflectorObj = getReflectorMethod.invoke(null, fieldsN, peer);
        } catch (Exception e) {
            if (InvocationTargetException.class.isInstance(e)) {
                Throwable x = ((InvocationTargetException)e).getTargetException();
                if (InvalidClassException.class.isInstance(x)) {
                    throw (InvalidClassException)x;
                }
            }
            throw new RuntimeException("Unable to call java.io.ObjectStreamField.getReflector",e);
        }

        return reflectorPeerManager.toJail(reflectorObj);
    }

    public static boolean hasStaticInitializer(Class cl) {
        initStatic();

        Boolean r;

        try {
            r = (Boolean)hasStaticInitializerMethod.invoke(null, cl);
        } catch (Exception e) {
            throw new RuntimeException("Unable to call java.io.ObjectStreamField.hasStaticInitializer",e);
        }

        return r.booleanValue();
    }

    public static jail.java.io.ObjectStreamField toJailObjectStreamField(java.io.ObjectStreamField field) {
        if (field==null) return null;

        // called privileged

        String name;
        Class type;
        boolean unshared;

        name = field.getName();
        type = field.getType();
        unshared = field.isUnshared();

        return new jail.java.io.ObjectStreamField(name,type,unshared);
    }
}
