/*
    BLUES - BD-Java emulation server

    Copyright (C) 2007-2025 GuinpinSoft inc <blues@makemkv.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
package blues;

import java.io.PrintStream;
import java.util.Arrays;

public class Log {
    public static final int LOG_ALL = 0;
    public static final int LOG_ERROR = 1;
    public static final int LOG_OUTPUT = 2;
    public static final int LOG_STACK = 3;
    public static final int LOG_PSR = 4;
    public static final int LOG_GPR = 5;
    public static final int LOG_SYS = 6;
    public static final int LOG_FILE = 7;
    public static final int LOG_LIBBLURAY = 8;
    public static final int LOG_EXCEPTIONS = 9;
    public static final int LOG_JAIL = 10;
    public static final int LOG_OUT = 11;
    public static final int LOG_ERR = 12;
    public static final int LOG_NET = 13;
    public static final int LOG_REFLECT = 14;
    public static final int LOG_CLASSLOAD = 15;
    public static final int LOG_OBJECTSTREAM = 16;
    public static final int LOG_AWT = 17;
    public static final int LOG_TODO = 18;
    public static final int LOG_CALL = 19;
    public static final int LOG_HDMV = 20;
    public static final int LOG_MAX = 21;

    private static final String[] logNames = new String[] { " ERR:", " OUT:", " STK:", " PSR:", " GPR:", " SYS:", "FILE:",
            "LIBB:", "XCPT:", "JAIL:", "SOUT:", "SERR:", " NET:", "REFL:", "CLDR:", "OSTR:", " AWT:", "TODO:", "CALL:",
            "HDMV:"};

    private static final int FRAME_ALWAYS = 0x8000;

    private static final int MAX_FRAMES_COUNT = 4;

    private static boolean mask[] = new boolean[LOG_MAX];

    private int disabledCounter = 0;
    private PrintStream out = null;

    public static Log getInstance() {
        Container container = Container.my();
        if (null==container) return null;
        return container.log;
    }

    public static boolean enabled(int source) {
        if (source >= LOG_MAX)
            throw new IllegalArgumentException();

        if ((mask[LOG_ALL] == false) && (mask[source] == false)) {
            return false;
        }
        return true;
    }

    public synchronized void setOut(PrintStream aout) {
        out = aout;
    }

    public synchronized void close() {
        if (out != null) {
            out.flush();
            if (!out.equals(System.out)) {
                out.close();
            }
            out = null;
        }
    }

    public synchronized void disable() {
        disabledCounter++;
    }

    public synchronized void reenable() {
        disabledCounter--;
    }

    public boolean disabled() {
        return (disabledCounter>0);
    }

    public static synchronized void setEnabled(int source, boolean value) {
        if (source >= LOG_MAX)
            throw new IllegalArgumentException();
        mask[source] = value;
    }

    public static synchronized void enableAll() {
        Arrays.fill(mask, true);
    }

    public static synchronized void log(int source, String msg) {
        if (enabled(source) == false)
            return;
        String[] strings = new String[] { msg };
        log(mask[LOG_STACK] ? 1 : -1, source, strings);
    }

    public static synchronized void log(int frame, int source, String msg) {
        if (enabled(source) == false)
            return;
        String[] strings = new String[] { msg };
        log(mask[LOG_STACK] ? frame : -1, source, strings);
    }

    public static synchronized void logLF(int frame, int source, String msg) {
        if (enabled(source) == false)
            return;
        if ((msg.length()>0) && (msg.charAt(msg.length()-1)=='\n'))
        {
            msg = msg.substring(0,msg.length()-1);
        }
        msg = msg.replace('\n', ' ');
        String[] strings = new String[] { msg };
        log(mask[LOG_STACK] ? frame : -1, source, strings);
    }

    public static synchronized void logN(boolean error, String file, int line, String msg) {
        if (enabled(LOG_LIBBLURAY) == false)
            return;
        StringBuffer buf = new StringBuffer(msg.length() + file.length() + 32);
        if (error) {
            buf.append("ERR ");
        }
        buf.append(msg);
        buf.append(" ");
        buf.append(file);
        buf.append(":");
        buf.append(Integer.toString(line));
        log(4,LOG_LIBBLURAY, buf.toString());
    }

    public static synchronized void log(int source, String... strings) {
        log(mask[LOG_STACK] ? 1 : -1, source, strings);
    }

    public static synchronized void log(int frame, int source, String... strings) {
        if (enabled(source) == false)
            return;

        Log log = getInstance();
        if (null!=log) {
            log.log0(frame,source,strings);
        }
    }

    synchronized void log0(int frame, int source, String... strings) {

        if (disabled()) return;
        if (null==out) return;

        int size = 2 + 128;
        for (int i = 0; i < strings.length; i++) {
            if (strings[i] != null) {
                size += strings[i].length();
            } else {
                size += 7;
            }
        }
        StringBuffer buf = new StringBuffer(size);
        buf.append(logNames[source-1]);
        buf.append(' ');
        for (int i = 0; i < strings.length; i++) {
            if (strings[i] != null) {
                buf.append(strings[i]);
            } else {
                buf.append("<null>");
            }
        }

        boolean stack = mask[LOG_STACK];
        if (((frame & FRAME_ALWAYS) != 0) && (frame>=0)) {
            stack = true;
            frame -= FRAME_ALWAYS;
        }

        if (stack && (frame >= 0)) {
            StackTraceElement[] traces = Thread.currentThread().getStackTrace();
            int offset = 2 + frame;
            int count = traces.length - offset;
            if (count > MAX_FRAMES_COUNT)
                count = MAX_FRAMES_COUNT;
            for (int i = 0; i < count; i++) {
                if (i == 0) {
                    buf.append(" @ ");
                } else {
                    buf.append(" ");
                }
                buf.append(traces[i + offset].toString());
            }
        }

        out.println(buf.toString());
    }

    public static synchronized void log(int source, String[] tail, String... strings) {
        log(mask[LOG_STACK] ? 1 : -1, source, tail, strings);
    }

    public static synchronized void log(int frame, int source, String[] tail, String... strings) {
        if (enabled(source) == false)
            return;

        String[] all;
        if (tail != null) {
            all = new String[strings.length + tail.length];
            System.arraycopy(strings, 0, all, 0, strings.length);
            System.arraycopy(tail, 0, all, strings.length, tail.length);
        } else {
            all = strings;
        }
        log(frame, source, all);
    }

    public static void unimplemented() {
        log(FRAME_ALWAYS | 0, LOG_TODO, "unimplemented");
    }

    public static void fatal(String msg) throws RuntimeException {
        log(FRAME_ALWAYS | 0, LOG_TODO, "fatal error: ",msg);
        throw new RuntimeException("Blues fatal error: "+msg);
    }

    public static void hang() {
        log(FRAME_ALWAYS | 0, LOG_TODO, "hanging forever");
        Object h = "";
        while (true) {
            synchronized(h) {
                try {
                    h.wait(500);
                } catch (InterruptedException e) {
                }
            }
        }
    }
}
