/*
 * Copyright (c) 1997, 2011, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package jail.java.util.jar;

import java.io.IOException;
import java.io.InputStream;
import java.util.Enumeration;
import java.util.jar.Manifest;

import jail.java.io.File;
import jail.java.util.zip.ZipEntry;
import jail.java.util.zip.ZipFile;

/**
 * The <code>JarFile</code> class is used to read the contents of a jar file
 * from any file that can be opened with <code>java.io.RandomAccessFile</code>.
 * It extends the class <code>java.util.zip.ZipFile</code> with support for
 * reading an optional <code>Manifest</code> entry. The <code>Manifest</code>
 * can be used to specify meta-information about the jar file and its entries.
 *
 * <p>
 * Unless otherwise noted, passing a <tt>null</tt> argument to a constructor or
 * method in this class will cause a {@link NullPointerException} to be thrown.
 *
 * @author David Connelly
 * @see Manifest
 * @see java.util.zip.ZipFile
 * @see java.util.jar.JarEntry
 * @since 1.2
 */
public class JarFile extends ZipFile {

    private java.util.jar.JarFile jpeer;

    /**
     * The JAR manifest file name.
     */
    public static final String MANIFEST_NAME = "META-INF/MANIFEST.MF";

    /**
     * Creates a new <code>JarFile</code> to read from the specified file
     * <code>name</code>. The <code>JarFile</code> will be verified if it is
     * signed.
     * 
     * @param name
     *            the name of the jar file to be opened for reading
     * @throws IOException
     *             if an I/O error has occurred
     * @throws SecurityException
     *             if access to the file is denied by the SecurityManager
     */
    public JarFile(String name) throws IOException {
        this(new File(name), true, ZipFile.OPEN_READ);
    }

    /**
     * Creates a new <code>JarFile</code> to read from the specified file
     * <code>name</code>.
     * 
     * @param name
     *            the name of the jar file to be opened for reading
     * @param verify
     *            whether or not to verify the jar file if it is signed.
     * @throws IOException
     *             if an I/O error has occurred
     * @throws SecurityException
     *             if access to the file is denied by the SecurityManager
     */
    public JarFile(String name, boolean verify) throws IOException {
        this(new File(name), verify, ZipFile.OPEN_READ);
    }

    /**
     * Creates a new <code>JarFile</code> to read from the specified
     * <code>File</code> object. The <code>JarFile</code> will be verified if it
     * is signed.
     * 
     * @param file
     *            the jar file to be opened for reading
     * @throws IOException
     *             if an I/O error has occurred
     * @throws SecurityException
     *             if access to the file is denied by the SecurityManager
     */
    public JarFile(File file) throws IOException {
        this(file, true, ZipFile.OPEN_READ);
    }

    /**
     * Creates a new <code>JarFile</code> to read from the specified
     * <code>File</code> object.
     * 
     * @param file
     *            the jar file to be opened for reading
     * @param verify
     *            whether or not to verify the jar file if it is signed.
     * @throws IOException
     *             if an I/O error has occurred
     * @throws SecurityException
     *             if access to the file is denied by the SecurityManager.
     */
    public JarFile(File file, boolean verify) throws IOException {
        this(file, verify, ZipFile.OPEN_READ);
    }

    /**
     * Creates a new <code>JarFile</code> to read from the specified
     * <code>File</code> object in the specified mode. The mode argument must be
     * either <tt>OPEN_READ</tt> or <tt>OPEN_READ | OPEN_DELETE</tt>.
     *
     * @param file
     *            the jar file to be opened for reading
     * @param verify
     *            whether or not to verify the jar file if it is signed.
     * @param mode
     *            the mode in which the file is to be opened
     * @throws IOException
     *             if an I/O error has occurred
     * @throws IllegalArgumentException
     *             if the <tt>mode</tt> argument is invalid
     * @throws SecurityException
     *             if access to the file is denied by the SecurityManager
     * @since 1.3
     */
    public JarFile(File file, boolean verify, int mode) throws IOException {
        super(file, mode, new Boolean(verify));
        jpeer = (java.util.jar.JarFile) impl.sun.misc.SharedSecrets.getJavaUtilZipFileAccess().getPeer(this);
    }

    /**
     * Returns the jar file manifest, or <code>null</code> if none.
     *
     * @return the jar file manifest, or <code>null</code> if none
     *
     * @throws IllegalStateException
     *             may be thrown if the jar file has been closed
     */
    public Manifest getManifest() throws IOException {
        return jpeer.getManifest();
    }

    /**
     * Returns the <code>JarEntry</code> for the given entry name or
     * <code>null</code> if not found.
     *
     * @param name
     *            the jar file entry name
     * @return the <code>JarEntry</code> for the given entry name or
     *         <code>null</code> if not found.
     *
     * @throws IllegalStateException
     *             may be thrown if the jar file has been closed
     *
     * @see java.util.jar.JarEntry
     */
    public JarEntry getJarEntry(String name) {
        return (JarEntry) getEntry(name);
    }

    /**
     * Returns the <code>ZipEntry</code> for the given entry name or
     * <code>null</code> if not found.
     *
     * @param name
     *            the jar file entry name
     * @return the <code>ZipEntry</code> for the given entry name or
     *         <code>null</code> if not found
     *
     * @throws IllegalStateException
     *             may be thrown if the jar file has been closed
     *
     * @see java.util.zip.ZipEntry
     */
    public ZipEntry getEntry(String name) {
        return super.getEntry(name);
    }

    /**
     * Returns an enumeration of the zip file entries.
     */
    public Enumeration<JarEntry> entries() {
        return (Enumeration<JarEntry>) super.entries();
    }

    /**
     * Returns an input stream for reading the contents of the specified zip
     * file entry.
     * 
     * @param ze
     *            the zip file entry
     * @return an input stream for reading the contents of the specified zip
     *         file entry
     * @throws ZipException
     *             if a zip file format error has occurred
     * @throws IOException
     *             if an I/O error has occurred
     * @throws SecurityException
     *             if any of the jar file entries are incorrectly signed.
     * @throws IllegalStateException
     *             may be thrown if the jar file has been closed
     */
    public synchronized InputStream getInputStream(ZipEntry ze) throws IOException {
        java.util.zip.ZipEntry ne = new java.util.zip.ZipEntry(ze.getName());
        //ne.setTime(ze.getTime());
        //ne.setCrc(ze.getCrc());
        //ne.setSize(ne.getSize());
        //ne.setCompressedSize(ze.getCompressedSize());
        //ne.setMethod(ze.getMethod());
        //ne.setExtra(ze.getExtra());
        //ne.setComment(ze.getComment());

        return jpeer.getInputStream(ne);
    }

}
