﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/amp/PrometheusServiceRequest.h>
#include <aws/amp/PrometheusService_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace PrometheusService {
namespace Model {

/**
 * <p>Represents the input of a <code>ListScrapers</code> operation.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/amp-2020-08-01/ListScrapersRequest">AWS
 * API Reference</a></p>
 */
class ListScrapersRequest : public PrometheusServiceRequest {
 public:
  AWS_PROMETHEUSSERVICE_API ListScrapersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListScrapers"; }

  AWS_PROMETHEUSSERVICE_API Aws::String SerializePayload() const override;

  AWS_PROMETHEUSSERVICE_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>(Optional) A list of key-value pairs to filter the list of scrapers returned.
   * Keys include <code>status</code>, <code>sourceArn</code>,
   * <code>destinationArn</code>, and <code>alias</code>.</p> <p>Filters on the same
   * key are <code>OR</code>'d together, and filters on different keys are
   * <code>AND</code>'d together. For example,
   * <code>status=ACTIVE&amp;status=CREATING&amp;alias=Test</code>, will return all
   * scrapers that have the alias Test, and are either in status ACTIVE or
   * CREATING.</p> <p>To find all active scrapers that are sending metrics to a
   * specific Amazon Managed Service for Prometheus workspace, you would use the ARN
   * of the workspace in a query:</p> <p>
   * <code>status=ACTIVE&amp;destinationArn=arn:aws:aps:us-east-1:123456789012:workspace/ws-example1-1234-abcd-56ef-123456789012</code>
   * </p> <p>If this is included, it filters the results to only the scrapers that
   * match the filter.</p>
   */
  inline const Aws::Map<Aws::String, Aws::Vector<Aws::String>>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Map<Aws::String, Aws::Vector<Aws::String>>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Map<Aws::String, Aws::Vector<Aws::String>>>
  ListScrapersRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersKeyT = Aws::String, typename FiltersValueT = Aws::Vector<Aws::String>>
  ListScrapersRequest& AddFilters(FiltersKeyT&& key, FiltersValueT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace(std::forward<FiltersKeyT>(key), std::forward<FiltersValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>(Optional) The token for the next set of items to return. (You received this
   * token from a previous call.)</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListScrapersRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Optional) The maximum number of scrapers to return in one
   * <code>ListScrapers</code> operation. The range is 1-1000.</p> <p>If you omit
   * this parameter, the default of 100 is used.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListScrapersRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::Map<Aws::String, Aws::Vector<Aws::String>> m_filters;

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_filtersHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace PrometheusService
}  // namespace Aws
