﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ssm-incidents/SSMIncidentsRequest.h>
#include <aws/ssm-incidents/SSMIncidents_EXPORTS.h>
#include <aws/ssm-incidents/model/Filter.h>
#include <aws/ssm-incidents/model/SortOrder.h>
#include <aws/ssm-incidents/model/TimelineEventSort.h>

#include <utility>

namespace Aws {
namespace SSMIncidents {
namespace Model {

/**
 */
class ListTimelineEventsRequest : public SSMIncidentsRequest {
 public:
  AWS_SSMINCIDENTS_API ListTimelineEventsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListTimelineEvents"; }

  AWS_SSMINCIDENTS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Filters the timeline events based on the provided conditional values. You can
   * filter timeline events with the following keys:</p> <ul> <li> <p>
   * <code>eventReference</code> </p> </li> <li> <p> <code>eventTime</code> </p>
   * </li> <li> <p> <code>eventType</code> </p> </li> </ul> <p>Note the following
   * when deciding how to use Filters:</p> <ul> <li> <p>If you don't specify a
   * Filter, the response includes all timeline events.</p> </li> <li> <p>If you
   * specify more than one filter in a single request, the response returns timeline
   * events that match all filters.</p> </li> <li> <p>If you specify a filter with
   * more than one value, the response returns timeline events that match any of the
   * values provided.</p> </li> </ul>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  ListTimelineEventsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  ListTimelineEventsRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the incident that includes the timeline
   * event.</p>
   */
  inline const Aws::String& GetIncidentRecordArn() const { return m_incidentRecordArn; }
  inline bool IncidentRecordArnHasBeenSet() const { return m_incidentRecordArnHasBeenSet; }
  template <typename IncidentRecordArnT = Aws::String>
  void SetIncidentRecordArn(IncidentRecordArnT&& value) {
    m_incidentRecordArnHasBeenSet = true;
    m_incidentRecordArn = std::forward<IncidentRecordArnT>(value);
  }
  template <typename IncidentRecordArnT = Aws::String>
  ListTimelineEventsRequest& WithIncidentRecordArn(IncidentRecordArnT&& value) {
    SetIncidentRecordArn(std::forward<IncidentRecordArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results per page.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListTimelineEventsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pagination token for the next set of items to return. (You received this
   * token from a previous call.)</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListTimelineEventsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Sort timeline events by the specified key value pair.</p>
   */
  inline TimelineEventSort GetSortBy() const { return m_sortBy; }
  inline bool SortByHasBeenSet() const { return m_sortByHasBeenSet; }
  inline void SetSortBy(TimelineEventSort value) {
    m_sortByHasBeenSet = true;
    m_sortBy = value;
  }
  inline ListTimelineEventsRequest& WithSortBy(TimelineEventSort value) {
    SetSortBy(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Sorts the order of timeline events by the value specified in the
   * <code>sortBy</code> field.</p>
   */
  inline SortOrder GetSortOrder() const { return m_sortOrder; }
  inline bool SortOrderHasBeenSet() const { return m_sortOrderHasBeenSet; }
  inline void SetSortOrder(SortOrder value) {
    m_sortOrderHasBeenSet = true;
    m_sortOrder = value;
  }
  inline ListTimelineEventsRequest& WithSortOrder(SortOrder value) {
    SetSortOrder(value);
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Filter> m_filters;

  Aws::String m_incidentRecordArn;

  int m_maxResults{0};

  Aws::String m_nextToken;

  TimelineEventSort m_sortBy{TimelineEventSort::NOT_SET};

  SortOrder m_sortOrder{SortOrder::NOT_SET};
  bool m_filtersHasBeenSet = false;
  bool m_incidentRecordArnHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_sortByHasBeenSet = false;
  bool m_sortOrderHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSMIncidents
}  // namespace Aws
