/*
 * This file is part of libbluray
 * Copyright (C) 2010  William Hahne
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 */

package jail.javax.tv.graphics;

import jail.java.awt.Color;
import jail.org.dvb.ui.DVBColor;

public class AlphaColor extends Color {
    public AlphaColor(float r, float g, float b, float a) {
        super(r, g, b, a);
    }

    public AlphaColor(int r, int g, int b, int a) {
        super(r, g, b, a);
    }

    public AlphaColor(int rgba, boolean hasAlpha) {
        super(rgba, hasAlpha);
    }

    public AlphaColor(Color c) {
        super(c.getRGB(), true);
    }

    private static final double FACTOR = 0.7;
    private static final int FACTORn = 70;
    private static final int FACTORd = 100;

    /**
     * Creates a new <code>Color</code> that is a brighter version of this
     * <code>Color</code>.
     * <p>
     * This method applies an arbitrary scale factor to each of the three RGB
     * components of this <code>Color</code> to create a brighter version of
     * this <code>Color</code>. Although <code>brighter</code> and
     * <code>darker</code> are inverse operations, the results of a series of
     * invocations of these two methods might be inconsistent because of
     * rounding errors.
     * 
     * @return a new <code>Color</code> object that is a brighter version of
     *         this <code>Color</code>.
     * @see java.awt.Color#darker
     * @since JDK1.0
     */
    public Color brighter() {
        int r = getRed();
        int g = getGreen();
        int b = getBlue();

        /*
         * From 2D group: 1. black.brighter() should return grey 2. applying
         * brighter to blue will always return blue, brighter 3. non pure color
         * (non zero rgb) will eventually return white
         */
        int i = (int) (1.0 / (1.0 - FACTOR));
        if (r == 0 && g == 0 && b == 0) {
            return new Color(i, i, i);
        }
        if (r > 0 && r < i)
            r = i;
        if (g > 0 && g < i)
            g = i;
        if (b > 0 && b < i)
            b = i;

        return new DVBColor(Math.min((r * FACTORd) / FACTORn, 255), Math.min((g * FACTORd) / FACTORn, 255),
                Math.min((b * FACTORd) / FACTORn, 255), getAlpha());
    }

    /**
     * Creates a new <code>Color</code> that is a darker version of this
     * <code>Color</code>.
     * <p>
     * This method applies an arbitrary scale factor to each of the three RGB
     * components of this <code>Color</code> to create a darker version of this
     * <code>Color</code>. Although <code>brighter</code> and
     * <code>darker</code> are inverse operations, the results of a series of
     * invocations of these two methods might be inconsistent because of
     * rounding errors.
     * 
     * @return a new <code>Color</code> object that is a darker version of this
     *         <code>Color</code>.
     * @see java.awt.Color#brighter
     * @since JDK1.0
     */
    public Color darker() {
        return new DVBColor(Math.max((getRed() * FACTORn) / FACTORd, 0), Math.max((getGreen() * FACTORn) / FACTORd, 0),
                Math.max((getBlue() * FACTORn) / FACTORd, 0), getAlpha());
    }

    public boolean equals(Object obj) {
        if (!(obj instanceof AlphaColor))
            return false;
        return super.equals(obj);
    }

    private static final long serialVersionUID = -3466072971590811211L;
}
