/*
 * @(#)NullGraphics.java	1.13 06/10/10
 *
 * Copyright  1990-2008 Sun Microsystems, Inc. All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 only, as published by the Free Software Foundation. 
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License version 2 for more details (a copy is
 * included at /legal/license.txt). 
 * 
 * You should have received a copy of the GNU General Public License
 * version 2 along with this work; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA 
 * 
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa
 * Clara, CA 95054 or visit www.sun.com if you need additional
 * information or have any questions. 
 *
 */

package impl.sun.awt;

import jail.java.awt.AlphaComposite;
import jail.java.awt.Color;
import jail.java.awt.Component;
import jail.java.awt.Composite;
import jail.java.awt.Font;
import jail.java.awt.FontMetrics;
import jail.java.awt.Graphics;
import jail.java.awt.GraphicsConfiguration;
import jail.java.awt.Image;
import jail.java.awt.Polygon;
import jail.java.awt.Rectangle;
import jail.java.awt.Shape;
import jail.java.awt.image.ImageObserver;
import jail.org.dvb.ui.DVBAlphaComposite;
import jail.org.dvb.ui.DVBGraphics;
import jail.org.dvb.ui.UnsupportedDrawingOperationException;
import jail.java.awt.Toolkit;

import java.awt.AWTError;
import java.text.AttributedCharacterIterator;

/**
 * A Graphics object which ignores all draw requests. This is useful to ensure
 * that a displayable, but not visible, component can get a graphics object to
 * draw onto without drawing onto the screen.
 * 
 * @author Nicholas Allen
 * @version 1.7, 08/19/02
 */

public class NullGraphics extends DVBGraphics {
    private Color color;
    private Rectangle clipBounds;
    private Font font;
    private AlphaComposite composite;
    private GraphicsConfiguration graphicsConfiguration;

    public NullGraphics(Component component) {
        color = component.getForeground();
        if (color == null)
            color = Color.black;
        font = component.getFont();
        if (font == null)
            font = new Font("dialog", Font.PLAIN, 12);

        composite = AlphaComposite.SrcOver;
        graphicsConfiguration = component.getGraphicsConfiguration();
    }

    private NullGraphics(NullGraphics g) {
        color = g.color;
        font = g.font;
        clipBounds = g.clipBounds;
        composite = g.composite;
        graphicsConfiguration = g.graphicsConfiguration;
    }

    /**
     * Creates a new <code>Graphics</code> object that is a copy of this
     * <code>Graphics</code> object.
     * 
     * @return a new graphics context that is a copy of this graphics context.
     * @since JDK1.0
     */
    public Graphics create() {
        return new NullGraphics(this);
    }

    /**
     * Translates the origin of the graphics context to the point (<i>x</i>
     * ,&nbsp;<i>y</i>) in the current coordinate system. Modifies this graphics
     * context so that its new origin corresponds to the point (<i>x</i>,&nbsp;
     * <i>y</i>) in this graphics context's original coordinate system. All
     * coordinates used in subsequent rendering operations on this graphics
     * context will be relative to this new origin.
     * 
     * @param x
     *            the <i>x</i> coordinate.
     * @param y
     *            the <i>y</i> coordinate.
     * @since JDK1.0
     */
    public void translate(int x, int y) {
        if (clipBounds != null) {
            clipBounds.x -= x;
            clipBounds.y -= y;
        }
    }

    /**
     * Gets this graphics context's current color.
     * 
     * @return this graphics context's current color.
     * @see java.awt.Color
     * @see java.awt.Graphics#setColor
     * @since JDK1.0
     */
    public Color getColor() {
        return color;
    }

    /**
     * Sets this graphics context's current color to the specified color. All
     * subsequent graphics operations using this graphics context use this
     * specified color.
     * 
     * @param c
     *            the new rendering color.
     * @see java.awt.Color
     * @see java.awt.Graphics#getColor
     * @since JDK1.0
     */
    public void setColor(Color c) {
        if (c == null)
            color = Color.black;
        else
            color = c;
    }

    /**
     * Sets the paint mode of this graphics context to overwrite the destination
     * with this graphics context's current color. This sets the logical pixel
     * operation function to the paint or overwrite mode. All subsequent
     * rendering operations will overwrite the destination with the current
     * color.
     * 
     * @since JDK1.0
     */
    public void setPaintMode() {
    }

    /**
     * This method sets the graphics context to xor paint mode using the
     * "exclusive or" color xorcolor. This specifies that logical pixel
     * operations are performed in the XOR mode, which alternates pixels between
     * the current color and a specified XOR color.
     * <p>
     * When drawing operations are performed, pixels which are the current color
     * are changed to the specified color, and vice versa.
     * <p>
     * Pixels that are of colors other than those two colors are changed in an
     * unpredictable but reversible manner; if the same figure is drawn twice,
     * then all pixels are restored to their original values.
     * <h3>Compatibility</h3> Both PersonalJava and Personal Profile
     * implementations are not required to support this method.
     * <h3>System Properties</h3> The System Property
     * <code>java.awt.graphics.SupportsXorMode</code> is set to
     * <code>"true"</code> or <code>"false"</code> indicating if the platform
     * supports XOR rendering.
     * 
     * @param c1
     *            the XOR alternation color
     * @exception <code>UnsupportedOperationException</code>
     *                if the implementation does not support an XOR paint mode.
     * @since JDK1.0
     */
    public void setXORMode(Color c1) {
    }

    /**
     * Gets the current font.
     * 
     * @return this graphics context's current font.
     * @see java.awt.Font
     * @see java.awt.Graphics#setFont
     * @since JDK1.0
     */
    public Font getFont() {
        return font;
    }

    /**
     * Sets this graphics context's font to the specified font. All subsequent
     * text operations using this graphics context use this font.
     * 
     * @param font
     *            the font.
     * @see java.awt.Graphics#getFont
     * @see java.awt.Graphics#drawChars
     * @see java.awt.Graphics#drawString
     * @see java.awt.Graphics#drawBytes
     * @since JDK1.0
     */
    public void setFont(Font font) {
        this.font = font;
    }

    /**
     * Gets the font metrics for the specified font.
     * 
     * @return the font metrics for the specified font.
     * @param f
     *            the specified font
     * @see java.awt.Graphics#getFont
     * @see java.awt.FontMetrics
     * @see java.awt.Graphics#getFontMetrics()
     * @since JDK1.0
     */
    public FontMetrics getFontMetrics(Font f) {
        return Toolkit.getDefaultToolkit().getFontMetrics(f);
    }

    /**
     * Returns the bounding rectangle of the current clipping area. The
     * coordinates in the rectangle are relative to the coordinate system origin
     * of this graphics context.
     * 
     * @return the bounding rectangle of the current clipping area.
     * @see java.awt.Graphics#getClip
     * @see java.awt.Graphics#clipRect
     * @see java.awt.Graphics#setClip(int, int, int, int)
     * @see java.awt.Graphics#setClip(Shape)
     * @since JDK1.1
     */
    public Rectangle getClipBounds() {
        return clipBounds;
    }

    /**
     * Intersects the current clip with the specified rectangle. The resulting
     * clipping area is the intersection of the current clipping area and the
     * specified rectangle. This method can only be used to make the current
     * clip smaller. To set the current clip larger, use any of the setClip
     * methods. Rendering operations have no effect outside of the clipping
     * area.
     * 
     * @param x
     *            the x coordinate of the rectangle to intersect the clip with
     * @param y
     *            the y coordinate of the rectangle to intersect the clip with
     * @param width
     *            the width of the rectangle to intersect the clip with
     * @param height
     *            the height of the rectangle to intersect the clip with
     * @see #setClip(int, int, int, int)
     * @see #setClip(Shape)
     */
    public void clipRect(int x, int y, int width, int height) {
        Rectangle rect = new Rectangle(x, y, width, height);
        if (clipBounds == null)
            clipBounds = rect;
        else
            clipBounds = clipBounds.union(rect);
    }

    /**
     * Sets the current clip to the rectangle specified by the given
     * coordinates. Rendering operations have no effect outside of the clipping
     * area.
     * 
     * @param x
     *            the <i>x</i> coordinate of the new clip rectangle.
     * @param y
     *            the <i>y</i> coordinate of the new clip rectangle.
     * @param width
     *            the width of the new clip rectangle.
     * @param height
     *            the height of the new clip rectangle.
     * @see java.awt.Graphics#clipRect
     * @see java.awt.Graphics#setClip(Shape)
     * @since JDK1.1
     */
    public void setClip(int x, int y, int width, int height) {
        clipBounds = new Rectangle(x, y, width, height);
    }

    /**
     * Gets the current clipping area.
     * 
     * @return a <code>Shape</code> object representing the current clipping
     *         area.
     * @see java.awt.Graphics#getClipBounds
     * @see java.awt.Graphics#clipRect
     * @see java.awt.Graphics#setClip(int, int, int, int)
     * @see java.awt.Graphics#setClip(Shape)
     * @since JDK1.1
     */
    public Shape getClip() {
        return clipBounds;
    }

    /**
     * Sets the current clipping area to an arbitrary clip shape. Not all
     * objects which implement the <code>Shape</code> interface can be used to
     * set the clip. The only <code>Shape</code> objects which are guaranteed to
     * be supported are <code>Shape</code> objects which are obtained via the
     * <code>getClip</code> method and via <code>Rectangle</code> objects.
     * 
     * @see java.awt.Graphics#getClip()
     * @see java.awt.Graphics#clipRect
     * @see java.awt.Graphics#setClip(int, int, int, int)
     * @since JDK1.1
     */
    public void setClip(Shape clip) {
        if (!(clip instanceof Rectangle))
            throw new AWTError("Only rectangular clip regions supported");
        clipBounds = (Rectangle) clip;
    }

    /**
     * Copies an area of the component by a distance specified by
     * <code>dx</code> and <code>dy</code>. From the point specified by
     * <code>x</code> and <code>y</code>, this method copies downwards and to
     * the right. To copy an area of the component to the left or upwards,
     * specify a negative value for <code>dx</code> or <code>dy</code>. If a
     * portion of the source rectangle lies outside the bounds of the component,
     * or is obscured by another window or component, <code>copyArea</code> will
     * be unable to copy the associated pixels. The area that is omitted can be
     * refreshed by calling the component's <code>paint</code> method.
     * 
     * @param x
     *            the <i>x</i> coordinate of the source rectangle.
     * @param y
     *            the <i>y</i> coordinate of the source rectangle.
     * @param width
     *            the width of the source rectangle.
     * @param height
     *            the height of the source rectangle.
     * @param dx
     *            the horizontal distance to copy the pixels.
     * @param dy
     *            the vertical distance to copy the pixels.
     * @since JDK1.0
     */
    public void copyArea(int x, int y, int width, int height, int dx, int dy) {
    }

    /**
     * Draws a line, using the current color, between the points
     * <code>(x1,&nbsp;y1)</code> and <code>(x2,&nbsp;y2)</code> in this
     * graphics context's coordinate system.
     * 
     * @param x1
     *            the first point's <i>x</i> coordinate.
     * @param y1
     *            the first point's <i>y</i> coordinate.
     * @param x2
     *            the second point's <i>x</i> coordinate.
     * @param y2
     *            the second point's <i>y</i> coordinate.
     * @since JDK1.0
     */
    public void drawLine(int x1, int y1, int x2, int y2) {
    }

    /**
     * Fills the specified rectangle. The left and right edges of the rectangle
     * are at <code>x</code> and <code>x&nbsp;+&nbsp;width&nbsp;-&nbsp;1</code>.
     * The top and bottom edges are at <code>y</code> and
     * <code>y&nbsp;+&nbsp;height&nbsp;-&nbsp;1</code>. The resulting rectangle
     * covers an area <code>width</code> pixels wide by <code>height</code>
     * pixels tall. The rectangle is filled using the graphics context's current
     * color.
     * 
     * @param x
     *            the <i>x</i> coordinate of the rectangle to be filled.
     * @param y
     *            the <i>y</i> coordinate of the rectangle to be filled.
     * @param width
     *            the width of the rectangle to be filled.
     * @param height
     *            the height of the rectangle to be filled.
     * @see java.awt.Graphics#fillRect
     * @see java.awt.Graphics#clearRect
     * @since JDK1.0
     */
    public void fillRect(int x, int y, int width, int height) {
    }

    /**
     * Clears the specified rectangle by filling it with the background color of
     * the current drawing surface. This operation does not use the current
     * paint mode.
     * <p>
     * Beginning with Java&nbsp;1.1, the background color of offscreen images
     * may be system dependent. Applications should use <code>setColor</code>
     * followed by <code>fillRect</code> to ensure that an offscreen image is
     * cleared to a specific color.
     * 
     * @param x
     *            the <i>x</i> coordinate of the rectangle to clear.
     * @param y
     *            the <i>y</i> coordinate of the rectangle to clear.
     * @param width
     *            the width of the rectangle to clear.
     * @param height
     *            the height of the rectangle to clear.
     * @see java.awt.Graphics#fillRect(int, int, int, int)
     * @see java.awt.Graphics#drawRect
     * @see java.awt.Graphics#setColor(java.awt.Color)
     * @see java.awt.Graphics#setPaintMode
     * @see java.awt.Graphics#setXORMode(java.awt.Color)
     * @since JDK1.0
     */
    public void clearRect(int x, int y, int width, int height) {
    }

    /**
     * Draws an outlined round-cornered rectangle using this graphics context's
     * current color. The left and right edges of the rectangle are at
     * <code>x</code> and <code>x&nbsp;+&nbsp;width</code>, respectively. The
     * top and bottom edges of the rectangle are at <code>y</code> and
     * <code>y&nbsp;+&nbsp;height</code>.
     * 
     * @param x
     *            the <i>x</i> coordinate of the rectangle to be drawn.
     * @param y
     *            the <i>y</i> coordinate of the rectangle to be drawn.
     * @param width
     *            the width of the rectangle to be drawn.
     * @param height
     *            the height of the rectangle to be drawn.
     * @param arcWidth
     *            the horizontal diameter of the arc at the four corners.
     * @param arcHeight
     *            the vertical diameter of the arc at the four corners.
     * @see java.awt.Graphics#fillRoundRect
     * @since JDK1.0
     */
    public void drawRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight) {
    }

    /**
     * Fills the specified rounded corner rectangle with the current color. The
     * left and right edges of the rectangle are at <code>x</code> and
     * <code>x&nbsp;+&nbsp;width&nbsp;-&nbsp;1</code>, respectively. The top and
     * bottom edges of the rectangle are at <code>y</code> and
     * <code>y&nbsp;+&nbsp;height&nbsp;-&nbsp;1</code>.
     * 
     * @param x
     *            the <i>x</i> coordinate of the rectangle to be filled.
     * @param y
     *            the <i>y</i> coordinate of the rectangle to be filled.
     * @param width
     *            the width of the rectangle to be filled.
     * @param height
     *            the height of the rectangle to be filled.
     * @param arcWidth
     *            the horizontal diameter of the arc at the four corners.
     * @param arcHeight
     *            the vertical diameter of the arc at the four corners.
     * @see java.awt.Graphics#drawRoundRect
     * @since JDK1.0
     */
    public void fillRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight) {
    }

    /**
     * Draws a 3-D highlighted outline of the specified rectangle. The edges of
     * the rectangle are highlighted so that they appear to be beveled and lit
     * from the upper left corner.
     * <p>
     * The colors used for the highlighting effect are determined based on the
     * current color. The resulting rectangle covers an area that is
     * <code>width&nbsp;+&nbsp;1</code> pixels wide by
     * <code>height&nbsp;+&nbsp;1</code> pixels tall.
     * 
     * @param x
     *            the <i>x</i> coordinate of the rectangle to be drawn.
     * @param y
     *            the <i>y</i> coordinate of the rectangle to be drawn.
     * @param width
     *            the width of the rectangle to be drawn.
     * @param height
     *            the height of the rectangle to be drawn.
     * @param raised
     *            a boolean that determines whether the rectangle appears to be
     *            raised above the surface or sunk into the surface.
     * @see java.awt.Graphics#fill3DRect
     * @since JDK1.0
     */
    public void draw3DRect(int x, int y, int width, int height, boolean raised) {
    }

    /**
     * Paints a 3-D highlighted rectangle filled with the current color. The
     * edges of the rectangle will be highlighted so that it appears as if the
     * edges were beveled and lit from the upper left corner. The colors used
     * for the highlighting effect will be determined from the current color.
     * 
     * @param x
     *            the <i>x</i> coordinate of the rectangle to be filled.
     * @param y
     *            the <i>y</i> coordinate of the rectangle to be filled.
     * @param width
     *            the width of the rectangle to be filled.
     * @param height
     *            the height of the rectangle to be filled.
     * @param raised
     *            a boolean value that determines whether the rectangle appears
     *            to be raised above the surface or etched into the surface.
     * @see java.awt.Graphics#draw3DRect
     * @since JDK1.0
     */
    public void fill3DRect(int x, int y, int width, int height, boolean raised) {
    }

    /**
     * Draws the outline of an oval. The result is a circle or ellipse that fits
     * within the rectangle specified by the <code>x</code>, <code>y</code>,
     * <code>width</code>, and <code>height</code> arguments.
     * <p>
     * The oval covers an area that is <code>width&nbsp;+&nbsp;1</code> pixels
     * wide and <code>height&nbsp;+&nbsp;1<code> pixels tall.
     * 
     * @param x
     *            the <i>x</i> coordinate of the upper left corner of the oval
     *            to be drawn.
     * @param y
     *            the <i>y</i> coordinate of the upper left corner of the oval
     *            to be drawn.
     * @param width
     *            the width of the oval to be drawn.
     * @param height
     *            the height of the oval to be drawn.
     * @see java.awt.Graphics#fillOval
     * @since JDK1.0
     */
    public void drawOval(int x, int y, int width, int height) {
    }

    /**
     * Fills an oval bounded by the specified rectangle with the current color.
     * 
     * @param x
     *            the <i>x</i> coordinate of the upper left corner of the oval
     *            to be filled.
     * @param y
     *            the <i>y</i> coordinate of the upper left corner of the oval
     *            to be filled.
     * @param width
     *            the width of the oval to be filled.
     * @param height
     *            the height of the oval to be filled.
     * @see java.awt.Graphics#drawOval
     * @since JDK1.0
     */
    public void fillOval(int x, int y, int width, int height) {
    }

    /**
     * Draws the outline of a circular or elliptical arc covering the specified
     * rectangle.
     * <p>
     * The resulting arc begins at <code>startAngle</code> and extends for
     * <code>arcAngle</code> degrees, using the current color. Angles are
     * interpreted such that 0&nbsp;degrees is at the 3&nbsp;o'clock position. A
     * positive value indicates a counter-clockwise rotation while a negative
     * value indicates a clockwise rotation.
     * <p>
     * The center of the arc is the center of the rectangle whose origin is (
     * <i>x</i>,&nbsp;<i>y</i>) and whose size is specified by the
     * <code>width</code> and <code>height</code> arguments.
     * <p>
     * The resulting arc covers an area <code>width&nbsp;+&nbsp;1</code> pixels
     * wide by <code>height&nbsp;+&nbsp;1</code> pixels tall.
     * 
     * @param x
     *            the <i>x</i> coordinate of the upper-left corner of the arc to
     *            be drawn.
     * @param y
     *            the <i>y</i> coordinate of the upper-left corner of the arc to
     *            be drawn.
     * @param width
     *            the width of the arc to be drawn.
     * @param height
     *            the height of the arc to be drawn.
     * @param startAngle
     *            the beginning angle.
     * @param arcAngle
     *            the angular extent of the arc, relative to the start angle.
     * @see java.awt.Graphics#fillArc
     * @since JDK1.0
     */
    public void drawArc(int x, int y, int width, int height, int startAngle, int arcAngle) {
    }

    /**
     * Fills a circular or elliptical arc covering the specified rectangle.
     * <p>
     * The resulting arc begins at <code>startAngle</code> and extends for
     * <code>arcAngle</code> degrees. Angles are interpreted such that
     * 0&nbsp;degrees is at the 3&nbsp;o'clock position. A positive value
     * indicates a counter-clockwise rotation while a negative value indicates a
     * clockwise rotation.
     * <p>
     * The center of the arc is the center of the rectangle whose origin is (
     * <i>x</i>,&nbsp;<i>y</i>) and whose size is specified by the
     * <code>width</code> and <code>height</code> arguments.
     * <p>
     * The resulting arc covers an area <code>width&nbsp;+&nbsp;1</code> pixels
     * wide by <code>height&nbsp;+&nbsp;1</code> pixels tall.
     * 
     * @param x
     *            the <i>x</i> coordinate of the upper-left corner of the arc to
     *            be filled.
     * @param y
     *            the <i>y</i> coordinate of the upper-left corner of the arc to
     *            be filled.
     * @param width
     *            the width of the arc to be filled.
     * @param height
     *            the height of the arc to be filled.
     * @param startAngle
     *            the beginning angle.
     * @param arcAngle
     *            the angular extent of the arc, relative to the start angle.
     * @see java.awt.Graphics#drawArc
     * @since JDK1.0
     */
    public void fillArc(int x, int y, int width, int height, int startAngle, int arcAngle) {
    }

    /**
     * Draws a sequence of connected lines defined by arrays of <i>x</i> and
     * <i>y</i> coordinates. Each pair of (<i>x</i>,&nbsp;<i>y</i>) coordinates
     * defines a point. The figure is not closed if the first point differs from
     * the last point.
     * 
     * @param xPoints
     *            an array of <i>x</i> points
     * @param yPoints
     *            an array of <i>y</i> points
     * @param nPoints
     *            the total number of points
     * @see java.awt.Graphics#drawPolygon(int[], int[], int)
     * @since JDK1.1
     */
    public void drawPolyline(int xPoints[], int yPoints[], int nPoints) {
    }

    /**
     * Draws a closed polygon defined by arrays of <i>x</i> and <i>y</i>
     * coordinates. Each pair of (<i>x</i>,&nbsp;<i>y</i>) coordinates defines a
     * point.
     * <p>
     * This method draws the polygon defined by <code>nPoint</code> line
     * segments, where the first <code>nPoint&nbsp;-&nbsp;1</code> line segments
     * are line segments from
     * <code>(xPoints[i&nbsp;-&nbsp;1],&nbsp;yPoints[i&nbsp;-&nbsp;1])</code> to
     * <code>(xPoints[i],&nbsp;yPoints[i])</code>, for 1&nbsp;&le;&nbsp;<i>i</i>
     * &nbsp;&le;&nbsp;<code>nPoints</code>. The figure is automatically closed
     * by drawing a line connecting the final point to the first point, if those
     * points are different.
     * 
     * @param xPoints
     *            a an array of <code>x</code> coordinates.
     * @param yPoints
     *            a an array of <code>y</code> coordinates.
     * @param nPoints
     *            a the total number of points.
     * @see java.awt.Graphics#fillPolygon
     * @see java.awt.Graphics#drawPolyline
     * @since JDK1.0
     */
    public void drawPolygon(int xPoints[], int yPoints[], int nPoints) {
    }

    /**
     * Draws the outline of a polygon defined by the specified
     * <code>Polygon</code> object.
     * 
     * @param p
     *            the polygon to draw.
     * @see java.awt.Graphics#fillPolygon
     * @see java.awt.Graphics#drawPolyline
     * @since JDK1.0
     */
    public void drawPolygon(Polygon p) {
    }

    /**
     * Fills a closed polygon defined by arrays of <i>x</i> and <i>y</i>
     * coordinates.
     * <p>
     * This method draws the polygon defined by <code>nPoint</code> line
     * segments, where the first <code>nPoint&nbsp;-&nbsp;1</code> line segments
     * are line segments from
     * <code>(xPoints[i&nbsp;-&nbsp;1],&nbsp;yPoints[i&nbsp;-&nbsp;1])</code> to
     * <code>(xPoints[i],&nbsp;yPoints[i])</code>, for 1&nbsp;&le;&nbsp;<i>i</i>
     * &nbsp;&le;&nbsp;<code>nPoints</code>. The figure is automatically closed
     * by drawing a line connecting the final point to the first point, if those
     * points are different.
     * <p>
     * The area inside the polygon is defined using an even-odd fill rule, also
     * known as the alternating rule.
     * 
     * @param xPoints
     *            a an array of <code>x</code> coordinates.
     * @param yPoints
     *            a an array of <code>y</code> coordinates.
     * @param nPoints
     *            a the total number of points.
     * @see java.awt.Graphics#drawPolygon(int[], int[], int)
     * @since JDK1.0
     */
    public void fillPolygon(int xPoints[], int yPoints[], int nPoints) {
    }

    /**
     * Fills the polygon defined by the specified Polygon object with the
     * graphics context's current color.
     * <p>
     * The area inside the polygon is defined using an even-odd fill rule, also
     * known as the alternating rule.
     * 
     * @param p
     *            the polygon to fill.
     * @see java.awt.Graphics#drawPolygon(int[], int[], int)
     * @since JDK1.0
     */
    public void fillPolygon(Polygon p) {
    }

    /**
     * Draws the text given by the specified string, using this graphics
     * context's current font and color. The baseline of the first character is
     * at position (<i>x</i>,&nbsp;<i>y</i>) in this graphics context's
     * coordinate system.
     * 
     * @param str
     *            the string to be drawn.
     * @param x
     *            the <i>x</i> coordinate.
     * @param y
     *            the <i>y</i> coordinate.
     * @see java.awt.Graphics#drawBytes
     * @see java.awt.Graphics#drawChars
     * @since JDK1.0
     */
    public void drawString(String str, int x, int y) {
    }

    /**
     * Draws the text given by the specified iterator, using this graphics
     * context's current color. The iterator has to specify a font for each
     * character. The baseline of the leftmost character is at position (
     * <i>x</i>,&nbsp;<i>y</i>) in this graphics context's coordinate system.
     * 
     * @param iterator
     *            the iterator whose text is to be drawn
     * @param x
     *            the <i>x</i> coordinate.
     * @param y
     *            the <i>y</i> coordinate.
     * @see java.awt.Graphics#drawBytes
     * @see java.awt.Graphics#drawChars
     */
    public void drawString(AttributedCharacterIterator iterator, int x, int y) {
    }

    /**
     * Draws the text given by the specified character array, using this
     * graphics context's current font and color. The baseline of the first
     * character is at position (<i>x</i>,&nbsp;<i>y</i>) in this graphics
     * context's coordinate system.
     * 
     * @param data
     *            the array of characters to be drawn
     * @param offset
     *            the start offset in the data
     * @param length
     *            the number of characters to be drawn
     * @param x
     *            the <i>x</i> coordinate of the baseline of the text
     * @param y
     *            the <i>y</i> coordinate of the baseline of the text
     * @see java.awt.Graphics#drawBytes
     * @see java.awt.Graphics#drawString
     * @since JDK1.0
     */
    public void drawChars(char data[], int offset, int length, int x, int y) {
    }

    /**
     * Draws the text given by the specified byte array, using this graphics
     * context's current font and color. The baseline of the first character is
     * at position (<i>x</i>,&nbsp;<i>y</i>) in this graphics context's
     * coordinate system.
     * 
     * @param data
     *            the data to be drawn
     * @param offset
     *            the start offset in the data
     * @param length
     *            the number of bytes that are drawn
     * @param x
     *            the <i>x</i> coordinate of the baseline of the text
     * @param y
     *            the <i>y</i> coordinate of the baseline of the text
     * @see java.awt.Graphics#drawChars
     * @see java.awt.Graphics#drawString
     * @since JDK1.0
     */
    public void drawBytes(byte data[], int offset, int length, int x, int y) {
    }

    /**
     * Draws as much of the specified image as is currently available. The image
     * is drawn with its top-left corner at (<i>x</i>,&nbsp;<i>y</i>) in this
     * graphics context's coordinate space. Transparent pixels in the image do
     * not affect whatever pixels are already there.
     * <p>
     * This method returns immediately in all cases, even if the complete image
     * has not yet been loaded, and it has not been dithered and converted for
     * the current output device.
     * <p>
     * If the image has not yet been completely loaded, then
     * <code>drawImage</code> returns <code>false</code>. As more of the image
     * becomes available, the process that draws the image notifies the
     * specified image observer.
     * 
     * @param img
     *            the specified image to be drawn.
     * @param x
     *            the <i>x</i> coordinate.
     * @param y
     *            the <i>y</i> coordinate.
     * @param observer
     *            object to be notified as more of the image is converted.
     * @see java.awt.Image
     * @see java.awt.image.ImageObserver
     * @see java.awt.image.ImageObserver#imageUpdate(java.awt.Image, int, int,
     *      int, int, int)
     * @since JDK1.0
     */
    public boolean drawImage(Image img, int x, int y, ImageObserver observer) {
        return false;
    }

    /**
     * Draws as much of the specified image as has already been scaled to fit
     * inside the specified rectangle.
     * <p>
     * The image is drawn inside the specified rectangle of this graphics
     * context's coordinate space, and is scaled if necessary. Transparent
     * pixels do not affect whatever pixels are already there.
     * <p>
     * This method returns immediately in all cases, even if the entire image
     * has not yet been scaled, dithered, and converted for the current output
     * device. If the current output representation is not yet complete, then
     * <code>drawImage</code> returns <code>false</code>. As more of the image
     * becomes available, the process that draws the image notifies the image
     * observer by calling its <code>imageUpdate</code> method.
     * <p>
     * A scaled version of an image will not necessarily be available
     * immediately just because an unscaled version of the image has been
     * constructed for this output device. Each size of the image may be cached
     * separately and generated from the original data in a separate image
     * production sequence.
     * 
     * @param img
     *            the specified image to be drawn.
     * @param x
     *            the <i>x</i> coordinate.
     * @param y
     *            the <i>y</i> coordinate.
     * @param width
     *            the width of the rectangle.
     * @param height
     *            the height of the rectangle.
     * @param observer
     *            object to be notified as more of the image is converted.
     * @see java.awt.Image
     * @see java.awt.image.ImageObserver
     * @see java.awt.image.ImageObserver#imageUpdate(java.awt.Image, int, int,
     *      int, int, int)
     * @since JDK1.0
     */
    public boolean drawImage(Image img, int x, int y, int width, int height, ImageObserver observer) {
        return false;
    }

    /**
     * Draws as much of the specified image as is currently available. The image
     * is drawn with its top-left corner at (<i>x</i>,&nbsp;<i>y</i>) in this
     * graphics context's coordinate space. Transparent pixels are drawn in the
     * specified background color.
     * <p>
     * This operation is equivalent to filling a rectangle of the width and
     * height of the specified image with the given color and then drawing the
     * image on top of it, but possibly more efficient.
     * <p>
     * This method returns immediately in all cases, even if the complete image
     * has not yet been loaded, and it has not been dithered and converted for
     * the current output device.
     * <p>
     * If the image has not yet been completely loaded, then
     * <code>drawImage</code> returns <code>false</code>. As more of the image
     * becomes available, the process that draws the image notifies the
     * specified image observer.
     * 
     * @param img
     *            the specified image to be drawn.
     * @param x
     *            the <i>x</i> coordinate.
     * @param y
     *            the <i>y</i> coordinate.
     * @param bgcolor
     *            the background color to paint under the non-opaque portions of
     *            the image.
     * @param observer
     *            object to be notified as more of the image is converted.
     * @see java.awt.Image
     * @see java.awt.image.ImageObserver
     * @see java.awt.image.ImageObserver#imageUpdate(java.awt.Image, int, int,
     *      int, int, int)
     * @since JDK1.0
     */
    public boolean drawImage(Image img, int x, int y, Color bgcolor, ImageObserver observer) {
        return false;
    }

    /**
     * Draws as much of the specified image as has already been scaled to fit
     * inside the specified rectangle.
     * <p>
     * The image is drawn inside the specified rectangle of this graphics
     * context's coordinate space, and is scaled if necessary. Transparent
     * pixels are drawn in the specified background color. This operation is
     * equivalent to filling a rectangle of the width and height of the
     * specified image with the given color and then drawing the image on top of
     * it, but possibly more efficient.
     * <p>
     * This method returns immediately in all cases, even if the entire image
     * has not yet been scaled, dithered, and converted for the current output
     * device. If the current output representation is not yet complete then
     * <code>drawImage</code> returns <code>false</code>. As more of the image
     * becomes available, the process that draws the image notifies the
     * specified image observer.
     * <p>
     * A scaled version of an image will not necessarily be available
     * immediately just because an unscaled version of the image has been
     * constructed for this output device. Each size of the image may be cached
     * separately and generated from the original data in a separate image
     * production sequence.
     * 
     * @param img
     *            the specified image to be drawn.
     * @param x
     *            the <i>x</i> coordinate.
     * @param y
     *            the <i>y</i> coordinate.
     * @param width
     *            the width of the rectangle.
     * @param height
     *            the height of the rectangle.
     * @param bgcolor
     *            the background color to paint under the non-opaque portions of
     *            the image.
     * @param observer
     *            object to be notified as more of the image is converted.
     * @see java.awt.Image
     * @see java.awt.image.ImageObserver
     * @see java.awt.image.ImageObserver#imageUpdate(java.awt.Image, int, int,
     *      int, int, int)
     * @since JDK1.0
     */
    public boolean drawImage(Image img, int x, int y, int width, int height, Color bgcolor, ImageObserver observer) {
        return false;
    }

    /**
     * Draws as much of the specified area of the specified image as is
     * currently available, scaling it on the fly to fit inside the specified
     * area of the destination drawable surface. Transparent pixels do not
     * affect whatever pixels are already there.
     * <p>
     * This method returns immediately in all cases, even if the image area to
     * be drawn has not yet been scaled, dithered, and converted for the current
     * output device. If the current output representation is not yet complete
     * then <code>drawImage</code> returns <code>false</code>. As more of the
     * image becomes available, the process that draws the image notifies the
     * specified image observer.
     * <p>
     * This method always uses the unscaled version of the image to render the
     * scaled rectangle and performs the required scaling on the fly. It does
     * not use a cached, scaled version of the image for this operation. Scaling
     * of the image from source to destination is performed such that the first
     * coordinate of the source rectangle is mapped to the first coordinate of
     * the destination rectangle, and the second source coordinate is mapped to
     * the second destination coordinate. The subimage is scaled and flipped as
     * needed to preserve those mappings.
     * 
     * @param img
     *            the specified image to be drawn
     * @param dx1
     *            the <i>x</i> coordinate of the first corner of the destination
     *            rectangle.
     * @param dy1
     *            the <i>y</i> coordinate of the first corner of the destination
     *            rectangle.
     * @param dx2
     *            the <i>x</i> coordinate of the second corner of the
     *            destination rectangle.
     * @param dy2
     *            the <i>y</i> coordinate of the second corner of the
     *            destination rectangle.
     * @param sx1
     *            the <i>x</i> coordinate of the first corner of the source
     *            rectangle.
     * @param sy1
     *            the <i>y</i> coordinate of the first corner of the source
     *            rectangle.
     * @param sx2
     *            the <i>x</i> coordinate of the second corner of the source
     *            rectangle.
     * @param sy2
     *            the <i>y</i> coordinate of the second corner of the source
     *            rectangle.
     * @param observer
     *            object to be notified as more of the image is scaled and
     *            converted.
     * @see java.awt.Image
     * @see java.awt.image.ImageObserver
     * @see java.awt.image.ImageObserver#imageUpdate(java.awt.Image, int, int,
     *      int, int, int)
     * @since JDK1.1
     */
    public boolean drawImage(Image img, int dx1, int dy1, int dx2, int dy2, int sx1, int sy1, int sx2, int sy2,
            ImageObserver observer) {
        return false;
    }

    /**
     * Draws as much of the specified area of the specified image as is
     * currently available, scaling it on the fly to fit inside the specified
     * area of the destination drawable surface.
     * <p>
     * Transparent pixels are drawn in the specified background color. This
     * operation is equivalent to filling a rectangle of the width and height of
     * the specified image with the given color and then drawing the image on
     * top of it, but possibly more efficient.
     * <p>
     * This method returns immediately in all cases, even if the image area to
     * be drawn has not yet been scaled, dithered, and converted for the current
     * output device. If the current output representation is not yet complete
     * then <code>drawImage</code> returns <code>false</code>. As more of the
     * image becomes available, the process that draws the image notifies the
     * specified image observer.
     * <p>
     * This method always uses the unscaled version of the image to render the
     * scaled rectangle and performs the required scaling on the fly. It does
     * not use a cached, scaled version of the image for this operation. Scaling
     * of the image from source to destination is performed such that the first
     * coordinate of the source rectangle is mapped to the first coordinate of
     * the destination rectangle, and the second source coordinate is mapped to
     * the second destination coordinate. The subimage is scaled and flipped as
     * needed to preserve those mappings.
     * 
     * @param img
     *            the specified image to be drawn
     * @param dx1
     *            the <i>x</i> coordinate of the first corner of the destination
     *            rectangle.
     * @param dy1
     *            the <i>y</i> coordinate of the first corner of the destination
     *            rectangle.
     * @param dx2
     *            the <i>x</i> coordinate of the second corner of the
     *            destination rectangle.
     * @param dy2
     *            the <i>y</i> coordinate of the second corner of the
     *            destination rectangle.
     * @param sx1
     *            the <i>x</i> coordinate of the first corner of the source
     *            rectangle.
     * @param sy1
     *            the <i>y</i> coordinate of the first corner of the source
     *            rectangle.
     * @param sx2
     *            the <i>x</i> coordinate of the second corner of the source
     *            rectangle.
     * @param sy2
     *            the <i>y</i> coordinate of the second corner of the source
     *            rectangle.
     * @param bgcolor
     *            the background color to paint under the non-opaque portions of
     *            the image.
     * @param observer
     *            object to be notified as more of the image is scaled and
     *            converted.
     * @see java.awt.Image
     * @see java.awt.image.ImageObserver
     * @see java.awt.image.ImageObserver#imageUpdate(java.awt.Image, int, int,
     *      int, int, int)
     * @since JDK1.1
     */
    public boolean drawImage(Image img, int dx1, int dy1, int dx2, int dy2, int sx1, int sy1, int sx2, int sy2,
            Color bgcolor, ImageObserver observer) {
        return false;
    }

    /**
     * Disposes of this graphics context and releases any system resources that
     * it is using. A <code>Graphics</code> object cannot be used after
     * <code>dispose</code>has been called.
     * <p>
     * When a Java program runs, a large number of <code>Graphics</code> objects
     * can be created within a short time frame. Although the finalization
     * process of the garbage collector also disposes of the same system
     * resources, it is preferable to manually free the associated resources by
     * calling this method rather than to rely on a finalization process which
     * may not run to completion for a long period of time.
     * <p>
     * Graphics objects which are provided as arguments to the
     * <code>paint</code> and <code>update</code> methods of components are
     * automatically released by the system when those methods return. For
     * efficiency, programmers should call <code>dispose</code> when finished
     * using a <code>Graphics</code> object only if it was created directly from
     * a component or another <code>Graphics</code> object.
     * 
     * @see java.awt.Graphics#finalize
     * @see java.awt.Component#paint
     * @see java.awt.Component#update
     * @see java.awt.Component#getGraphics
     * @see java.awt.Graphics#create
     * @since JDK1.0
     */
    public void dispose() {
    }

    public int[] getAvailableCompositeRules() {
        return new int[] { DVBAlphaComposite.CLEAR, DVBAlphaComposite.SRC, DVBAlphaComposite.SRC_OVER };
    }

    public DVBAlphaComposite getDVBComposite() {
        return DVBAlphaComposite.getInstance(composite.getRule());
    }

    public void setDVBComposite(DVBAlphaComposite comp) throws UnsupportedDrawingOperationException {
        try {
            setComposite(AlphaComposite.getInstance(comp.getRule(), comp.getAlpha()));
        } catch (AWTError e) {
            throw new UnsupportedDrawingOperationException(e.getMessage());
        }
    }

    public Composite getComposite() {
        return composite;
    }

    public GraphicsConfiguration getDeviceConfiguration() {
        return graphicsConfiguration;
    }

    public void setComposite(Composite comp) {
        if ((comp != null) && (comp != this.composite)) {
            if (!(comp instanceof AlphaComposite)) {
                throw new IllegalArgumentException("Only AlphaComposite is supported");
            }
            this.composite = (AlphaComposite) comp;
        }
    }
}
