/*
 * @(#)InputEvent.java	1.28 05/01/04
 *
 * Copyright  1990-2008 Sun Microsystems, Inc. All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 only, as published by the Free Software Foundation. 
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License version 2 for more details (a copy is
 * included at /legal/license.txt). 
 * 
 * You should have received a copy of the GNU General Public License
 * version 2 along with this work; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA 
 * 
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa
 * Clara, CA 95054 or visit www.sun.com if you need additional
 * information or have any questions. 
 *
 */

package jail.java.awt.event;

import jail.java.awt.Component;

/**
 * The root event class for all component-level input events.
 *
 * Input events are delivered to listeners before they are processed normally by
 * the source where they originated. This allows listeners and component
 * subclasses to "consume" the event so that the source will not process them in
 * their default manner. For example, consuming mousePressed events on a Button
 * component will prevent the Button from being activated.
 *
 * @author Carl Quinn
 * @version 1.22, 08/19/02
 *
 * @see KeyEvent
 * @see KeyAdapter
 * @see MouseEvent
 * @see MouseAdapter
 * @see MouseMotionAdapter
 *
 * @since 1.1
 */
public abstract class InputEvent extends ComponentEvent {
    /**
     * The shift key modifier constant. It is recommended that SHIFT_DOWN_MASK
     * to be used instead.
     */
    public static final int SHIFT_MASK = 1 << 0;
    /**
     * The control key modifier constant. It is recommended that CTRL_DOWN_MASK
     * to be used instead.
     */
    public static final int CTRL_MASK = 1 << 1;
    /**
     * The meta key modifier constant. It is recommended that META_DOWN_MASK to
     * be used instead.
     */
    public static final int META_MASK = 1 << 2;
    /**
     * The alt key modifier constant. It is recommended that ALT_DOWN_MASK to be
     * used instead.
     */
    public static final int ALT_MASK = 1 << 3;
    /**
     * The alt-graph key modifier constant.
     */
    public static final int ALT_GRAPH_MASK = 1 << 5;
    /**
     * The mouse button1 modifier constant. It is recommended that
     * BUTTON1_DOWN_MASK to be used instead.
     */
    public static final int BUTTON1_MASK = 1 << 4;
    /**
     * The mouse button2 modifier constant. It is recommended that
     * BUTTON2_DOWN_MASK to be used instead.
     */
    public static final int BUTTON2_MASK = ALT_MASK;
    /**
     * The mouse button3 modifier constant. It is recommended that
     * BUTTON3_DOWN_MASK to be used instead.
     */
    public static final int BUTTON3_MASK = META_MASK;

    /**
     * The input events Time stamp. The time stamp is in UTC format that
     * indicates when the input event was created.
     *
     * @serial
     * @see getWhen()
     */
    long when;
    /**
     * The state of the modifier key at the time the input event was fired.
     *
     * @serial
     * @see getModifiers()
     * @see java.awt.event.MouseEvent
     */
    int modifiers;

    /**
     * Constructs an InputEvent object with the specified source component,
     * modifiers, and type.
     * 
     * @param source
     *            the object where the event originated
     * @id the event type
     * @when the time the event occurred
     * @modifiers the modifier keys down while event occurred
     */
    InputEvent(Component source, int id, long when, int modifiers) {
        super(source, id);
        this.when = when;
        this.modifiers = modifiers;
    }

    /**
     * Returns whether or not the Shift modifier is down on this event.
     */
    public boolean isShiftDown() {
        return (modifiers & SHIFT_MASK) != 0;
    }

    /**
     * Returns whether or not the Control modifier is down on this event.
     */
    public boolean isControlDown() {
        return (modifiers & CTRL_MASK) != 0;
    }

    /**
     * Returns whether or not the Meta modifier is down on this event.
     */
    public boolean isMetaDown() {
        return (modifiers & META_MASK) != 0;
    }

    /**
     * Returns whether or not the Alt modifier is down on this event.
     */
    public boolean isAltDown() {
        return (modifiers & ALT_MASK) != 0;
    }

    /**
     * Returns whether or not the Alt-Graph modifier is down on this event.
     */
    public boolean isAltGraphDown() {
        return (modifiers & ALT_GRAPH_MASK) != 0;
    }

    /**
     * Returns the timestamp of when this event occurred.
     */
    public long getWhen() {
        return when;
    }

    /**
     * Returns the modifiers flag for this event.
     */
    public int getModifiers() {
        return modifiers;
    }

    /**
     * Consumes this event so that it will not be processed in the default
     * manner by the source which originated it.
     */
    public void consume() {
        consumed = true;
    }

    /**
     * Returns whether or not this event has been consumed.
     * 
     * @see #consume
     */
    public boolean isConsumed() {
        return consumed;
    }

    // state serialization compatibility with JDK 1.1
    static final long serialVersionUID = -2482525981698309786L;
}
