﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/invoicing/InvoicingRequest.h>
#include <aws/invoicing/Invoicing_EXPORTS.h>
#include <aws/invoicing/model/InvoiceUnitRule.h>

#include <utility>

namespace Aws {
namespace Invoicing {
namespace Model {

/**
 */
class UpdateInvoiceUnitRequest : public InvoicingRequest {
 public:
  AWS_INVOICING_API UpdateInvoiceUnitRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateInvoiceUnit"; }

  AWS_INVOICING_API Aws::String SerializePayload() const override;

  AWS_INVOICING_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ARN to identify an invoice unit. This information can't be modified or
   * deleted. </p>
   */
  inline const Aws::String& GetInvoiceUnitArn() const { return m_invoiceUnitArn; }
  inline bool InvoiceUnitArnHasBeenSet() const { return m_invoiceUnitArnHasBeenSet; }
  template <typename InvoiceUnitArnT = Aws::String>
  void SetInvoiceUnitArn(InvoiceUnitArnT&& value) {
    m_invoiceUnitArnHasBeenSet = true;
    m_invoiceUnitArn = std::forward<InvoiceUnitArnT>(value);
  }
  template <typename InvoiceUnitArnT = Aws::String>
  UpdateInvoiceUnitRequest& WithInvoiceUnitArn(InvoiceUnitArnT&& value) {
    SetInvoiceUnitArn(std::forward<InvoiceUnitArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The assigned description for an invoice unit. This information can't be
   * modified or deleted. </p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateInvoiceUnitRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether the invoice unit based tax inheritance is/ should be enabled or
   * disabled. </p>
   */
  inline bool GetTaxInheritanceDisabled() const { return m_taxInheritanceDisabled; }
  inline bool TaxInheritanceDisabledHasBeenSet() const { return m_taxInheritanceDisabledHasBeenSet; }
  inline void SetTaxInheritanceDisabled(bool value) {
    m_taxInheritanceDisabledHasBeenSet = true;
    m_taxInheritanceDisabled = value;
  }
  inline UpdateInvoiceUnitRequest& WithTaxInheritanceDisabled(bool value) {
    SetTaxInheritanceDisabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>InvoiceUnitRule</code> object used to update invoice units. </p>
   */
  inline const InvoiceUnitRule& GetRule() const { return m_rule; }
  inline bool RuleHasBeenSet() const { return m_ruleHasBeenSet; }
  template <typename RuleT = InvoiceUnitRule>
  void SetRule(RuleT&& value) {
    m_ruleHasBeenSet = true;
    m_rule = std::forward<RuleT>(value);
  }
  template <typename RuleT = InvoiceUnitRule>
  UpdateInvoiceUnitRequest& WithRule(RuleT&& value) {
    SetRule(std::forward<RuleT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_invoiceUnitArn;
  bool m_invoiceUnitArnHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  bool m_taxInheritanceDisabled{false};
  bool m_taxInheritanceDisabledHasBeenSet = false;

  InvoiceUnitRule m_rule;
  bool m_ruleHasBeenSet = false;
};

}  // namespace Model
}  // namespace Invoicing
}  // namespace Aws
