package ebpf

import (
	"context"
	"io"
	"log/slog"

	"github.com/cilium/ebpf"

	ebpfcommon "github.com/grafana/beyla/v2/pkg/internal/ebpf/common"
	"github.com/grafana/beyla/v2/pkg/internal/exec"
	"github.com/grafana/beyla/v2/pkg/internal/goexec"
	"github.com/grafana/beyla/v2/pkg/internal/request"
	"github.com/grafana/beyla/v2/pkg/internal/svc"
	"github.com/grafana/beyla/v2/pkg/pipe/msg"
)

type Instrumentable struct {
	Type                 svc.InstrumentableType
	InstrumentationError error

	// in some runtimes, like python gunicorn, we need to allow
	// tracing both the parent pid and all of its children pid
	ChildPids []uint32

	FileInfo *exec.FileInfo
	Offsets  *goexec.Offsets
	Tracer   *ProcessTracer
}

func (ie *Instrumentable) CopyToServiceAttributes() {
	// If the user does not override the service name via configuration
	// the service name is the name of the found executable
	if ie.FileInfo.Service.UID.Name == "" {
		ie.FileInfo.Service.UID.Name = ie.FileInfo.ExecutableName()
		// we mark the service ID as automatically named in case we want to look,
		// in later stages of the pipeline, for better automatic service name
		ie.FileInfo.Service.SetAutoName()
	}

	ie.FileInfo.Service.SDKLanguage = ie.Type
}

type PIDsAccounter interface {
	// AllowPID notifies the tracer to accept traces from the process with the
	// provided PID. The Tracer should discard
	// traces from processes whose PID has not been allowed before
	// We must use a pointer for svc.Attrs so that all child processes share the same
	// object. This is important when we tag a service as exporting traces or metrics.
	AllowPID(uint32, uint32, *svc.Attrs)
	// BlockPID notifies the tracer to stop accepting traces from the process
	// with the provided PID. After receiving them via ringbuffer, it should
	// discard them.
	BlockPID(uint32, uint32)
}

type CommonTracer interface {
	// Load the bpf object that is generated by the bpf2go compiler
	Load() (*ebpf.CollectionSpec, error)
	// AddCloser adds io.Closer instances that need to be invoked when the
	// Run function ends.
	AddCloser(c ...io.Closer)
	// BpfObjects that are created by the bpf2go compiler
	BpfObjects() any
	// Sets up any tail call tables if the BPF program has it
	SetupTailCalls()
}

type KprobesTracer interface {
	CommonTracer
	// KProbes returns a map with the name of the kernel probes that need to be
	// tapped into. Start matches kprobe, End matches kretprobe
	KProbes() map[string]ebpfcommon.ProbeDesc
	Tracepoints() map[string]ebpfcommon.ProbeDesc
}

// Tracer is an individual eBPF program (e.g. the net/http or the grpc tracers)
type Tracer interface {
	PIDsAccounter
	KprobesTracer
	// Constants returns a map of constants to be overridden into the eBPF program.
	// The key is the constant name and the value is the value to overwrite.
	Constants() map[string]any
	// GoProbes returns a slice with the name of Go functions that need to be inspected
	// in the executable, as well as the eBPF programs that optionally need to be
	// inserted as the Go function start and end probes
	GoProbes() map[string][]*ebpfcommon.ProbeDesc
	// UProbes returns a map with the module name mapping to the uprobes that need to be
	// tapped into. Start matches uprobe, End matches uretprobe
	UProbes() map[string]map[string][]*ebpfcommon.ProbeDesc
	// SocketFilters  returns a list of programs that need to be loaded as a
	// generic eBPF socket filter
	SocketFilters() []*ebpf.Program
	// SockMsgs returns a list of programs that need to be loaded as a
	// BPF_PROG_TYPE_SK_MSG eBPF programs
	SockMsgs() []ebpfcommon.SockMsg
	// SockOps returns a list of programs that need to be loaded as a
	// BPF_PROG_TYPE_SOCK_OPS eBPF programs
	SockOps() []ebpfcommon.SockOps
	// Probes can potentially instrument a shared library among multiple executables
	// These two functions allow programs to remember this and avoid duplicated instrumentations
	// The argument is the OS file id
	// Closers are the associated closable resources to this lib, that may be
	// closed when UnlinkInstrumentedLib() is called
	RecordInstrumentedLib(uint64, []io.Closer)
	AddInstrumentedLibRef(uint64)
	AlreadyInstrumentedLib(uint64) bool
	UnlinkInstrumentedLib(uint64)
	RegisterOffsets(*exec.FileInfo, *goexec.Offsets)
	ProcessBinary(*exec.FileInfo)
	Required() bool
	// Run will do the action of listening for eBPF traces and forward them
	// periodically to the output channel.
	Run(context.Context, *ebpfcommon.EBPFEventContext, *msg.Queue[[]request.Span])
}

// Subset of the above interface, which supports loading eBPF programs which
// are not tied to service monitoring
type UtilityTracer interface {
	KprobesTracer
	Run(context.Context)
}

type ProcessTracerType int

const (
	Go = ProcessTracerType(iota)
	Generic
)

// ProcessTracer instruments an executable with eBPF and provides the eBPF readers
// that will forward the traces to later stages in the pipeline
// TODO: We need to pass the ELFInfo from this ProcessTracker to inside a Tracer
// so that the GPU kernel event listener can find symbols names from addresses
// in the ELF file.
type ProcessTracer struct {
	log      *slog.Logger //nolint:unused
	Programs []Tracer

	Type            ProcessTracerType
	Instrumentables map[uint64]*instrumenter
}

func (pt *ProcessTracer) AllowPID(pid, ns uint32, svc *svc.Attrs) {
	for i := range pt.Programs {
		pt.Programs[i].AllowPID(pid, ns, svc)
	}
}

func (pt *ProcessTracer) BlockPID(pid, ns uint32) {
	for i := range pt.Programs {
		pt.Programs[i].BlockPID(pid, ns)
	}
}
