/*
 * Copyright (c) 1995, 2006, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package glue.java.lang;

import jail.java.io.File;
import jail.java.io.FileDescriptor;
import jail.java.io.FilePermission;

public class SecurityManager {

    /**
     * Throws a <code>SecurityException</code> if the calling thread is not
     * allowed to create a subprocess.
     * <p>
     * This method is invoked for the current security manager by the
     * <code>exec</code> methods of class <code>Runtime</code>.
     * <p>
     * This method calls <code>checkPermission</code> with the
     * <code>FilePermission(cmd,"execute")</code> permission if cmd is an
     * absolute path, otherwise it calls <code>checkPermission</code> with
     * <code>FilePermission("&lt;&lt;ALL FILES&gt;&gt;","execute")</code>.
     * <p>
     * If you override this method, then you should make a call to
     * <code>super.checkExec</code> at the point the overridden method would
     * normally throw an exception.
     *
     * @param cmd
     *            the specified system command.
     * @exception SecurityException
     *                if the calling thread does not have permission to create a
     *                subprocess.
     * @exception NullPointerException
     *                if the <code>cmd</code> argument is <code>null</code>.
     * @see java.lang.Runtime#exec(java.lang.String)
     * @see java.lang.Runtime#exec(java.lang.String, java.lang.String[])
     * @see java.lang.Runtime#exec(java.lang.String[])
     * @see java.lang.Runtime#exec(java.lang.String[], java.lang.String[])
     * @see #checkPermission(java.security.Permission) checkPermission
     */
    public static void checkExec(Object othis, String cmd) {
        java.lang.SecurityManager that = (java.lang.SecurityManager) othis;
        File f = new File(cmd);
        if (f.isAbsolute()) {
            that.checkPermission(new FilePermission(cmd, "execute"));
        } else {
            that.checkPermission(new FilePermission("<<ALL FILES>>", "execute"));
        }
    }

    /**
     * Throws a <code>SecurityException</code> if the calling thread is not
     * allowed to read from the specified file descriptor.
     * <p>
     * This method calls <code>checkPermission</code> with the
     * <code>RuntimePermission("readFileDescriptor")</code> permission.
     * <p>
     * If you override this method, then you should make a call to
     * <code>super.checkRead</code> at the point the overridden method would
     * normally throw an exception.
     *
     * @param fd
     *            the system-dependent file descriptor.
     * @exception SecurityException
     *                if the calling thread does not have permission to access
     *                the specified file descriptor.
     * @exception NullPointerException
     *                if the file descriptor argument is <code>null</code>.
     * @see java.io.FileDescriptor
     * @see #checkPermission(java.security.Permission) checkPermission
     */
    public static void checkRead(Object othis, FileDescriptor fd) {
        java.lang.SecurityManager that = (java.lang.SecurityManager) othis;
        if (fd == null) {
            throw new NullPointerException("file descriptor can't be null");
        }
        that.checkPermission(new RuntimePermission("readFileDescriptor"));
    }

    /**
     * Throws a <code>SecurityException</code> if the calling thread is not
     * allowed to read the file specified by the string argument.
     * <p>
     * This method calls <code>checkPermission</code> with the
     * <code>FilePermission(file,"read")</code> permission.
     * <p>
     * If you override this method, then you should make a call to
     * <code>super.checkRead</code> at the point the overridden method would
     * normally throw an exception.
     *
     * @param file
     *            the system-dependent file name.
     * @exception SecurityException
     *                if the calling thread does not have permission to access
     *                the specified file.
     * @exception NullPointerException
     *                if the <code>file</code> argument is <code>null</code>.
     * @see #checkPermission(java.security.Permission) checkPermission
     */
    public static void checkRead(Object othis, String file) {
        java.lang.SecurityManager that = (java.lang.SecurityManager) othis;
        that.checkPermission(new FilePermission(file, "read"));
    }

    /**
     * Throws a <code>SecurityException</code> if the specified security context
     * is not allowed to read the file specified by the string argument. The
     * context must be a security context returned by a previous call to
     * <code>getSecurityContext</code>.
     * <p>
     * If <code>context</code> is an instance of
     * <code>AccessControlContext</code> then the
     * <code>AccessControlContext.checkPermission</code> method will be invoked
     * with the <code>FilePermission(file,"read")</code> permission.
     * <p>
     * If <code>context</code> is not an instance of
     * <code>AccessControlContext</code> then a <code>SecurityException</code>
     * is thrown.
     * <p>
     * If you override this method, then you should make a call to
     * <code>super.checkRead</code> at the point the overridden method would
     * normally throw an exception.
     *
     * @param file
     *            the system-dependent filename.
     * @param context
     *            a system-dependent security context.
     * @exception SecurityException
     *                if the specified security context is not an instance of
     *                <code>AccessControlContext</code> (e.g., is
     *                <code>null</code>), or does not have permission to read
     *                the specified file.
     * @exception NullPointerException
     *                if the <code>file</code> argument is <code>null</code>.
     * @see java.lang.SecurityManager#getSecurityContext()
     * @see java.security.AccessControlContext#checkPermission(java.security.Permission)
     */
    public static void checkRead(Object othis, String file, Object context) {
        java.lang.SecurityManager that = (java.lang.SecurityManager) othis;
        that.checkPermission(new FilePermission(file, "read"), context);
    }

    /**
     * Throws a <code>SecurityException</code> if the calling thread is not
     * allowed to write to the specified file descriptor.
     * <p>
     * This method calls <code>checkPermission</code> with the
     * <code>RuntimePermission("writeFileDescriptor")</code> permission.
     * <p>
     * If you override this method, then you should make a call to
     * <code>super.checkWrite</code> at the point the overridden method would
     * normally throw an exception.
     *
     * @param fd
     *            the system-dependent file descriptor.
     * @exception SecurityException
     *                if the calling thread does not have permission to access
     *                the specified file descriptor.
     * @exception NullPointerException
     *                if the file descriptor argument is <code>null</code>.
     * @see java.io.FileDescriptor
     * @see #checkPermission(java.security.Permission) checkPermission
     */
    public static void checkWrite(Object othis, FileDescriptor fd) {
        java.lang.SecurityManager that = (java.lang.SecurityManager) othis;
        if (fd == null) {
            throw new NullPointerException("file descriptor can't be null");
        }
        that.checkPermission(new RuntimePermission("writeFileDescriptor"));

    }

    /**
     * Throws a <code>SecurityException</code> if the calling thread is not
     * allowed to write to the file specified by the string argument.
     * <p>
     * This method calls <code>checkPermission</code> with the
     * <code>FilePermission(file,"write")</code> permission.
     * <p>
     * If you override this method, then you should make a call to
     * <code>super.checkWrite</code> at the point the overridden method would
     * normally throw an exception.
     *
     * @param file
     *            the system-dependent filename.
     * @exception SecurityException
     *                if the calling thread does not have permission to access
     *                the specified file.
     * @exception NullPointerException
     *                if the <code>file</code> argument is <code>null</code>.
     * @see #checkPermission(java.security.Permission) checkPermission
     */
    public static void checkWrite(Object othis, String file) {
        java.lang.SecurityManager that = (java.lang.SecurityManager) othis;
        that.checkPermission(new FilePermission(file, "write"));
    }

    /**
     * Throws a <code>SecurityException</code> if the calling thread is not
     * allowed to delete the specified file.
     * <p>
     * This method is invoked for the current security manager by the
     * <code>delete</code> method of class <code>File</code>.
     * <p>
     * This method calls <code>checkPermission</code> with the
     * <code>FilePermission(file,"delete")</code> permission.
     * <p>
     * If you override this method, then you should make a call to
     * <code>super.checkDelete</code> at the point the overridden method would
     * normally throw an exception.
     *
     * @param file
     *            the system-dependent filename.
     * @exception SecurityException
     *                if the calling thread does not have permission to delete
     *                the file.
     * @exception NullPointerException
     *                if the <code>file</code> argument is <code>null</code>.
     * @see java.io.File#delete()
     * @see #checkPermission(java.security.Permission) checkPermission
     */
    public static void checkDelete(Object othis, String file) {
        java.lang.SecurityManager that = (java.lang.SecurityManager) othis;
        that.checkPermission(new FilePermission(file, "delete"));
    }

}
