/*
    BLUES - BD-Java emulation server

    Copyright (C) 2007-2025 GuinpinSoft inc <blues@makemkv.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
package impl.java.io;

import blues.Blob;
import blues.Server;

public class BluesFile extends FsFile {

    static final int CACHE_SIZE = 8192*3;
    static final int CACHE_HINT = 1024;

    private int fd;

    private byte[] cacheData = null;
    private long cacheOffset = 0;
    private int cacheSize = 0;


    public BluesFile(String path, String nativePath) {
        fd = 0;
        setPath(path, nativePath);
    }

    public BluesFile(FsDir parent, String name, String nativePath) {
        fd = 0;
        setPath(parent, name, nativePath);
    }

    @Override
    public int writeBytes(long offset, byte[] b, int off, int len) {
        return -1;
    }

    @Override
    protected void populateAttributes() {

        accessFlags = ACCESS_READ;

        Blob request = new Blob(50);
        request.strings = new String[] { getNativePath() };
        request.args = new int[] { 1 };
        Blob response = Server.callRequest(request);
        if (response == null) {
            fd = 0;
            length = 0;
            return;
        }

        lastModifiedTime = response.getLong(0);
        length = response.getLong(2);
        fd = response.args[4];
    }

    @Override
    public synchronized int readBytes(long offset, byte[] b, int off, int len) {
        if (fd == 0) {
            populateAttributes();
            if (fd == 0) {
                return -3;
            }
        }

        if ( (offset>=cacheOffset) && ((offset+len) <= (cacheOffset+cacheSize)) ) {
            System.arraycopy(cacheData, (int)(offset-cacheOffset), b, off, len);
            return len;
        }

        if ( (len >= CACHE_HINT) || ((offset+len) >= this.length) || (offset>=this.length) ) {
            return callReadBytes(offset,b,off,len);
        }

        if (cacheData==null) {
            cacheData = new byte[CACHE_SIZE];
        }

        cacheOffset = offset;
        cacheSize = callReadBytes(offset,cacheData,0,CACHE_SIZE);
        if (cacheSize<=0) {
            cacheSize = 0;
            return callReadBytes(offset,b,off,len);
        }

        if ( (offset>=cacheOffset) && ((offset+len) <= (cacheOffset+cacheSize)) ) {
            System.arraycopy(cacheData, (int)(offset-cacheOffset), b, off, len);
            return len;
        }

        return callReadBytes(offset,b,off,len);
    }

    private int callReadBytes(long offset, byte[] b, int off, int len) {
        Blob request = new Blob(51);
        request.args = new int[] { fd, len, (int) (offset >> 32), (int) (offset) };
        Blob response = Server.callRequest(request);
        if (response == null)
            return -2;
        if (response.data == null)
            return -1;
        System.arraycopy(response.data, 0, b, off, response.data.length);
        return response.data.length;
    }

    @Override
    protected void lastReferenceClosed() {
        Blob request = new Blob(52);
        request.args = new int[] { fd };
        Server.callRequest(request);
        fd = 0;
        cacheData = null;
        cacheSize = 0;
    }

    @Override
    public void close() {
    }

    @Override
    public int hashCode() {
        return nativePath.hashCode();
    }

    @Override
    public boolean equals(Object obj) {
        if (obj instanceof BluesFile) {
            return this.nativePath.equals(((BluesFile)obj).nativePath);
        }
        return false;
    }


}
