﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/lambda/LambdaRequest.h>
#include <aws/lambda/Lambda_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Lambda {
namespace Model {

/**
 */
class PublishVersionRequest : public LambdaRequest {
 public:
  AWS_LAMBDA_API PublishVersionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PublishVersion"; }

  AWS_LAMBDA_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name or ARN of the Lambda function.</p> <p class="title"> <b>Name
   * formats</b> </p> <ul> <li> <p> <b>Function name</b> -
   * <code>MyFunction</code>.</p> </li> <li> <p> <b>Function ARN</b> -
   * <code>arn:aws:lambda:us-west-2:123456789012:function:MyFunction</code>.</p>
   * </li> <li> <p> <b>Partial ARN</b> -
   * <code>123456789012:function:MyFunction</code>.</p> </li> </ul> <p>The length
   * constraint applies only to the full ARN. If you specify only the function name,
   * it is limited to 64 characters in length.</p>
   */
  inline const Aws::String& GetFunctionName() const { return m_functionName; }
  inline bool FunctionNameHasBeenSet() const { return m_functionNameHasBeenSet; }
  template <typename FunctionNameT = Aws::String>
  void SetFunctionName(FunctionNameT&& value) {
    m_functionNameHasBeenSet = true;
    m_functionName = std::forward<FunctionNameT>(value);
  }
  template <typename FunctionNameT = Aws::String>
  PublishVersionRequest& WithFunctionName(FunctionNameT&& value) {
    SetFunctionName(std::forward<FunctionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Only publish a version if the hash value matches the value that's specified.
   * Use this option to avoid publishing a version if the function code has changed
   * since you last updated it. You can get the hash for the version that you
   * uploaded from the output of <a>UpdateFunctionCode</a>.</p>
   */
  inline const Aws::String& GetCodeSha256() const { return m_codeSha256; }
  inline bool CodeSha256HasBeenSet() const { return m_codeSha256HasBeenSet; }
  template <typename CodeSha256T = Aws::String>
  void SetCodeSha256(CodeSha256T&& value) {
    m_codeSha256HasBeenSet = true;
    m_codeSha256 = std::forward<CodeSha256T>(value);
  }
  template <typename CodeSha256T = Aws::String>
  PublishVersionRequest& WithCodeSha256(CodeSha256T&& value) {
    SetCodeSha256(std::forward<CodeSha256T>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description for the version to override the description in the function
   * configuration.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  PublishVersionRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Only update the function if the revision ID matches the ID that's specified.
   * Use this option to avoid publishing a version if the function configuration has
   * changed since you last updated it.</p>
   */
  inline const Aws::String& GetRevisionId() const { return m_revisionId; }
  inline bool RevisionIdHasBeenSet() const { return m_revisionIdHasBeenSet; }
  template <typename RevisionIdT = Aws::String>
  void SetRevisionId(RevisionIdT&& value) {
    m_revisionIdHasBeenSet = true;
    m_revisionId = std::forward<RevisionIdT>(value);
  }
  template <typename RevisionIdT = Aws::String>
  PublishVersionRequest& WithRevisionId(RevisionIdT&& value) {
    SetRevisionId(std::forward<RevisionIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_functionName;
  bool m_functionNameHasBeenSet = false;

  Aws::String m_codeSha256;
  bool m_codeSha256HasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::String m_revisionId;
  bool m_revisionIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace Lambda
}  // namespace Aws
