/*
 * Copyright (c) 1995, 2015, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package jail.java.net;

import java.net.Inet4Address;
import java.net.UnknownHostException;
import java.util.ArrayList;

import blues.Log;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectInputStream.GetField;
import java.io.ObjectOutputStream;
import java.io.ObjectOutputStream.PutField;
import java.io.ObjectStreamField;

/**
 * This class represents an Internet Protocol (IP) address.
 *
 * <p>
 * An IP address is either a 32-bit or 128-bit unsigned number used by IP, a
 * lower-level protocol on which protocols like UDP and TCP are built. The IP
 * address architecture is defined by
 * <a href="http://www.ietf.org/rfc/rfc790.txt"><i>RFC&nbsp;790: Assigned
 * Numbers</i></a>,
 * <a href="http://www.ietf.org/rfc/rfc1918.txt"> <i>RFC&nbsp;1918: Address
 * Allocation for Private Internets</i></a>,
 * <a href="http://www.ietf.org/rfc/rfc2365.txt"><i>RFC&nbsp;2365:
 * Administratively Scoped IP Multicast</i></a>, and
 * <a href="http://www.ietf.org/rfc/rfc2373.txt"><i>RFC&nbsp;2373: IP Version 6
 * Addressing Architecture</i></a>. An instance of an InetAddress consists of an
 * IP address and possibly its corresponding host name (depending on whether it
 * is constructed with a host name or whether it has already done reverse host
 * name resolution).
 *
 * <h4>Address types</h4>
 *
 * <blockquote>
 * <table cellspacing=2 summary=
 * "Description of unicast and multicast address types">
 * <tr>
 * <th valign=top><i>unicast</i></th>
 * <td>An identifier for a single interface. A packet sent to a unicast address
 * is delivered to the interface identified by that address.
 *
 * <p>
 * The Unspecified Address -- Also called anylocal or wildcard address. It must
 * never be assigned to any node. It indicates the absence of an address. One
 * example of its use is as the target of bind, which allows a server to accept
 * a client connection on any interface, in case the server host has multiple
 * interfaces.
 *
 * <p>
 * The <i>unspecified</i> address must not be used as the destination address of
 * an IP packet.
 *
 * <p>
 * The <i>Loopback</i> Addresses -- This is the address assigned to the loopback
 * interface. Anything sent to this IP address loops around and becomes IP input
 * on the local host. This address is often used when testing a client.</td>
 * </tr>
 * <tr>
 * <th valign=top><i>multicast</i></th>
 * <td>An identifier for a set of interfaces (typically belonging to different
 * nodes). A packet sent to a multicast address is delivered to all interfaces
 * identified by that address.</td>
 * </tr>
 * </table>
 * </blockquote>
 *
 * <h4>IP address scope</h4>
 *
 * <p>
 * <i>Link-local</i> addresses are designed to be used for addressing on a
 * single link for purposes such as auto-address configuration, neighbor
 * discovery, or when no routers are present.
 *
 * <p>
 * <i>Site-local</i> addresses are designed to be used for addressing inside of
 * a site without the need for a global prefix.
 *
 * <p>
 * <i>Global</i> addresses are unique across the internet.
 *
 * <h4>Textual representation of IP addresses</h4>
 *
 * The textual representation of an IP address is address family specific.
 *
 * <p>
 *
 * For IPv4 address format, please refer to
 * <A HREF="Inet4Address.html#format">Inet4Address#format</A>; For IPv6 address
 * format, please refer to
 * <A HREF="Inet6Address.html#format">Inet6Address#format</A>.
 *
 * <h4>Host Name Resolution</h4>
 *
 * Host name-to-IP address <i>resolution</i> is accomplished through the use of
 * a combination of local machine configuration information and network naming
 * services such as the Domain Name System (DNS) and Network Information
 * Service(NIS). The particular naming services(s) being used is by default the
 * local machine configured one. For any host name, its corresponding IP address
 * is returned.
 *
 * <p>
 * <i>Reverse name resolution</i> means that for any IP address, the host
 * associated with the IP address is returned.
 *
 * <p>
 * The InetAddress class provides methods to resolve host names to their IP
 * addresses and vice versa.
 *
 * <h4>InetAddress Caching</h4>
 *
 * The InetAddress class has a cache to store successful as well as unsuccessful
 * host name resolutions.
 *
 * <p>
 * By default, when a security manager is installed, in order to protect against
 * DNS spoofing attacks, the result of positive host name resolutions are cached
 * forever. When a security manager is not installed, the default behavior is to
 * cache entries for a finite (implementation dependent) period of time. The
 * result of unsuccessful host name resolution is cached for a very short period
 * of time (10 seconds) to improve performance.
 *
 * <p>
 * If the default behavior is not desired, then a Java security property can be
 * set to a different Time-to-live (TTL) value for positive caching. Likewise, a
 * system admin can configure a different negative caching TTL value when
 * needed.
 *
 * <p>
 * Two Java security properties control the TTL values used for positive and
 * negative host name resolution caching:
 *
 * <blockquote>
 * <dl>
 * <dt><b>networkaddress.cache.ttl</b></dt>
 * <dd>Indicates the caching policy for successful name lookups from the name
 * service. The value is specified as as integer to indicate the number of
 * seconds to cache the successful lookup. The default setting is to cache for
 * an implementation specific period of time.
 * <p>
 * A value of -1 indicates "cache forever".</dd>
 * <p>
 * <dt><b>networkaddress.cache.negative.ttl</b> (default: 10)</dt>
 * <dd>Indicates the caching policy for un-successful name lookups from the name
 * service. The value is specified as as integer to indicate the number of
 * seconds to cache the failure for un-successful lookups.
 * <p>
 * A value of 0 indicates "never cache". A value of -1 indicates "cache forever"
 * .</dd>
 * </dl>
 * </blockquote>
 *
 * @author Chris Warth
 * @see java.net.InetAddress#getByAddress(byte[])
 * @see java.net.InetAddress#getByAddress(java.lang.String, byte[])
 * @see java.net.InetAddress#getAllByName(java.lang.String)
 * @see java.net.InetAddress#getByName(java.lang.String)
 * @see java.net.InetAddress#getLocalHost()
 * @since JDK1.0
 */
public class InetAddress implements java.io.Serializable {

    transient Inet4Address peer;
    transient String writeName;
    transient int writeAddress;

    /**
     * use serialVersionUID from InetAddress, but Inet4Address instance is
     * always replaced by an InetAddress instance before being serialized
     */
    private static final long serialVersionUID = 3286316764910316507L;

    private InetAddress(Inet4Address peer) {
        this.peer = peer;
    }

    private InetAddress(String hostName, int address) {
        peer = createPeer(hostName, address);
    }

    private static Inet4Address createPeer(String hostName, int address) {
        byte[] bytes = new byte[4];
        bytes[0] = (byte) (address >> 24);
        bytes[1] = (byte) (address >> 16);
        bytes[2] = (byte) (address >> 8);
        bytes[3] = (byte) (address);
        Inet4Address peer;
        try {
            peer = (Inet4Address) java.net.InetAddress.getByAddress(hostName, bytes);
        } catch (UnknownHostException e) {
            peer = null;
        }
        return peer;
    }

    static InetAddress fromPeer(java.net.Inet4Address peer) {
        return new InetAddress(peer);
    }
    
    static InetAddress fromPeer(java.net.InetAddress peer) {
        if (peer instanceof java.net.Inet4Address) {
            return new InetAddress((java.net.Inet4Address)peer);
        }
        Log.log(Log.LOG_NET, "Peer is not ipv4 address");
        return null;
    }
    
    
    static InetAddress getByAddress(byte[] ip) throws UnknownHostException {
        java.net.Inet4Address addr = null;
        if ( (ip!=null) && (ip.length == impl.java.net.Inet4Address.INADDRSZ) ) {
             try {
                addr = (java.net.Inet4Address)java.net.InetAddress.getByAddress(ip);
            } catch (UnknownHostException e) {
                addr = null;
            }
            if ( (addr != null) && (addr instanceof java.net.Inet4Address) ) {
                return new InetAddress((java.net.Inet4Address)addr);
            }
        }
        throw new UnknownHostException();
    }

    /**
     * Utility routine to check if the InetAddress is an IP multicast address.
     * 
     * @return a <code>boolean</code> indicating if the InetAddress is an IP
     *         multicast address
     * @since JDK1.1
     */
    public boolean isMulticastAddress() {
        return peer.isMulticastAddress();
    }

    /**
     * Gets the host name for this IP address.
     *
     * <p>
     * If this InetAddress was created with a host name, this host name will be
     * remembered and returned; otherwise, a reverse name lookup will be
     * performed and the result will be returned based on the system configured
     * name lookup service. If a lookup of the name service is required, call
     * {@link #getCanonicalHostName() getCanonicalHostName}.
     *
     * <p>
     * If there is a security manager, its <code>checkConnect</code> method is
     * first called with the hostname and <code>-1</code> as its arguments to
     * see if the operation is allowed. If the operation is not allowed, it will
     * return the textual representation of the IP address.
     *
     * @return the host name for this IP address, or if the operation is not
     *         allowed by the security check, the textual representation of the
     *         IP address.
     *
     * @see InetAddress#getCanonicalHostName
     * @see SecurityManager#checkConnect
     */
    public String getHostName() {
        return peer.getHostName();
    }

    /**
     * Returns the raw IP address of this <code>InetAddress</code> object. The
     * result is in network byte order: the highest order byte of the address is
     * in <code>getAddress()[0]</code>.
     *
     * @return the raw IP address of this object.
     */
    public byte[] getAddress() {
        return peer.getAddress();
    }

    /**
     * Returns the IP address string in textual presentation.
     *
     * @return the raw IP address in a string format.
     * @since JDK1.0.2
     */
    public String getHostAddress() {
        return peer.getHostAddress();
    }

    /**
     * Returns a hashcode for this IP address.
     *
     * @return a hash code value for this IP address.
     */
    public int hashCode() {
        return peer.hashCode();
    }

    /**
     * Compares this object against the specified object. The result is
     * <code>true</code> if and only if the argument is not <code>null</code>
     * and it represents the same IP address as this object.
     * <p>
     * Two instances of <code>InetAddress</code> represent the same IP address
     * if the length of the byte arrays returned by <code>getAddress</code> is
     * the same for both, and each of the array components is the same for the
     * byte arrays.
     *
     * @param obj
     *            the object to compare against.
     * @return <code>true</code> if the objects are the same; <code>false</code>
     *         otherwise.
     * @see java.net.InetAddress#getAddress()
     */
    public boolean equals(Object obj) {
        if (null == obj)
            return false;
        if (obj instanceof InetAddress) {
            return peer.equals(((InetAddress) obj).peer);
        }
        return false;
    }

    /**
     * Converts this IP address to a <code>String</code>. The string returned is
     * of the form: hostname / literal IP address.
     *
     * If the host name is unresolved, no reverse name service lookup is
     * performed. The hostname part will be represented by an empty string.
     *
     * @return a string representation of this IP address.
     */
    public String toString() {
        String hostName = peer.getHostName();
        return ((hostName != null) ? hostName : "") + "/" + peer.getHostAddress();
    }

    /**
     * Determines the IP address of a host, given the host's name.
     *
     * <p>
     * The host name can either be a machine name, such as
     * "<code>java.sun.com</code>", or a textual representation of its IP
     * address. If a literal IP address is supplied, only the validity of the
     * address format is checked.
     *
     * <p>
     * For <code>host</code> specified in literal IPv6 address, either the form
     * defined in RFC 2732 or the literal IPv6 address format defined in RFC
     * 2373 is accepted. IPv6 scoped addresses are also supported. See
     * <a href="Inet6Address.html#scoped">here</a> for a description of IPv6
     * scoped addresses.
     *
     * <p>
     * If the host is <tt>null</tt> then an <tt>InetAddress</tt> representing an
     * address of the loopback interface is returned. See
     * <a href="http://www.ietf.org/rfc/rfc3330.txt">RFC&nbsp;3330</a>
     * section&nbsp;2 and
     * <a href="http://www.ietf.org/rfc/rfc2373.txt">RFC&nbsp;2373</a>
     * section&nbsp;2.5.3.
     * </p>
     *
     * @param host
     *            the specified host, or <code>null</code>.
     * @return an IP address for the given host name.
     * @exception UnknownHostException
     *                if no IP address for the <code>host</code> could be found,
     *                or if a scope_id was specified for a global IPv6 address.
     * @exception SecurityException
     *                if a security manager exists and its checkConnect method
     *                doesn't allow the operation
     */
    public static InetAddress getByName(String host) throws UnknownHostException {
        return InetAddress.getAllByName(host)[0];
    }

    /**
     * Given the name of a host, returns an array of its IP addresses, based on
     * the configured name service on the system.
     *
     * <p>
     * The host name can either be a machine name, such as
     * "<code>java.sun.com</code>", or a textual representation of its IP
     * address. If a literal IP address is supplied, only the validity of the
     * address format is checked.
     *
     * <p>
     * For <code>host</code> specified in <i>literal IPv6 address</i>, either
     * the form defined in RFC 2732 or the literal IPv6 address format defined
     * in RFC 2373 is accepted. A literal IPv6 address may also be qualified by
     * appending a scoped zone identifier or scope_id. The syntax and usage of
     * scope_ids is described <a href="Inet6Address.html#scoped">here</a>.
     * <p>
     * If the host is <tt>null</tt> then an <tt>InetAddress</tt> representing an
     * address of the loopback interface is returned. See
     * <a href="http://www.ietf.org/rfc/rfc3330.txt">RFC&nbsp;3330</a>
     * section&nbsp;2 and
     * <a href="http://www.ietf.org/rfc/rfc2373.txt">RFC&nbsp;2373</a>
     * section&nbsp;2.5.3.
     * </p>
     *
     * <p>
     * If there is a security manager and <code>host</code> is not null and
     * <code>host.length() </code> is not equal to zero, the security manager's
     * <code>checkConnect</code> method is called with the hostname and
     * <code>-1</code> as its arguments to see if the operation is allowed.
     *
     * @param host
     *            the name of the host, or <code>null</code>.
     * @return an array of all the IP addresses for a given host name.
     *
     * @exception UnknownHostException
     *                if no IP address for the <code>host</code> could be found,
     *                or if a scope_id was specified for a global IPv6 address.
     * @exception SecurityException
     *                if a security manager exists and its
     *                <code>checkConnect</code> method doesn't allow the
     *                operation.
     *
     * @see SecurityManager#checkConnect
     */
    public static InetAddress[] getAllByName(String host) throws UnknownHostException {
        if (host == null || host.length() == 0) {
            return new InetAddress[] { new InetAddress("localhost", 0x7f000001) };
        }
        if ((host.charAt(0) == '[') || (host.charAt(0) == ':')) {
            throw new UnknownHostException(host);
        }

        java.net.InetAddress[] r = java.net.InetAddress.getAllByName(host);
        ArrayList<InetAddress> t = new ArrayList(r.length);
        for (int i = 0; i < r.length; i++) {
            if (r[i] instanceof Inet4Address) {
                t.add(new InetAddress((Inet4Address) r[i]));
            }
        }
        if (t.isEmpty())
            throw new UnknownHostException(host);

        InetAddress[] a = new InetAddress[t.size()];
        System.arraycopy(t.toArray(), 0, a, 0, a.length);
        return a;
    }

    /**
     * Returns the address of the local host. This is achieved by retrieving the
     * name of the host from the system, then resolving that name into an
     * <code>InetAddress</code>.
     *
     * <P>
     * Note: The resolved address may be cached for a short period of time.
     * </P>
     *
     * <p>
     * If there is a security manager, its <code>checkConnect</code> method is
     * called with the local host name and <code>-1</code> as its arguments to
     * see if the operation is allowed. If the operation is not allowed, an
     * InetAddress representing the loopback address is returned.
     *
     * @return the address of the local host.
     *
     * @exception UnknownHostException
     *                if the local host name could not be resolved into an
     *                address.
     *
     * @see SecurityManager#checkConnect
     * @see java.net.InetAddress#getByName(java.lang.String)
     */
    public static InetAddress getLocalHost() throws UnknownHostException {

        java.net.InetAddress a = java.net.InetAddress.getLocalHost();
        if (a instanceof Inet4Address) {
            return new InetAddress((Inet4Address) a);
        }
        return new InetAddress("localhost", 0x7f000001);
    }
    static InetAddress getAnyLocalAddress() {

        return new InetAddress("*",0);
    }

    private void readObject(ObjectInputStream s) throws IOException, ClassNotFoundException {
        if (getClass().getClassLoader() != null) {
            throw new SecurityException("invalid address type");
        }
        GetField gf = s.readFields();
        String host = (String) gf.get("hostName", null);
        int address = gf.get("address", 0);
        int family = gf.get("family", 0);
        if (family != 2)
            throw new IOException();
        peer = createPeer(host, address);
    }

    /* needed because the serializable fields no longer exist */

    /**
     * @serialField hostName
     *                  String
     * @serialField address
     *                  int
     * @serialField family
     *                  int
     */
    private static final ObjectStreamField[] serialPersistentFields = { new ObjectStreamField("hostName", String.class),
            new ObjectStreamField("address", int.class), new ObjectStreamField("family", int.class), };

    private void writeObject(ObjectOutputStream s) throws IOException {
        if (getClass().getClassLoader() != null) {
            throw new SecurityException("invalid address type");
        }
        int addr = 0;
        byte[] ab = peer.getAddress();
        addr |= (((int) ab[0]) & 0xff);
        addr <<= 8;
        addr |= (((int) ab[1]) & 0xff);
        addr <<= 8;
        addr |= (((int) ab[2]) & 0xff);
        addr <<= 8;
        addr |= (((int) ab[3]) & 0xff);
        addr <<= 8;
        PutField pf = s.putFields();
        pf.put("hostName", peer.getHostName());
        pf.put("address", addr);
        pf.put("family", 2);
        s.writeFields();
    }
}
