// Copyright The OpenTelemetry Authors
// SPDX-License-Identifier: Apache-2.0

package ebpfcommon

import (
	"bytes"
	"net"
	"strconv"
	"strings"

	"go.opentelemetry.io/obi/pkg/app/request"
	"go.opentelemetry.io/obi/pkg/components/ebpf/ringbuf"
)

// misses serviceID
func httpInfoToSpan(info *HTTPInfo) request.Span {
	scheme := "http"
	if info.Ssl == 1 {
		scheme = "https"
	}

	return request.Span{
		Type:           request.EventType(info.Type),
		Method:         info.Method,
		Path:           removeQuery(info.URL),
		Peer:           info.Peer,
		PeerPort:       int(info.ConnInfo.S_port),
		Host:           info.Host,
		HostPort:       int(info.ConnInfo.D_port),
		ContentLength:  int64(info.Len),
		ResponseLength: int64(info.RespLen),
		RequestStart:   int64(info.StartMonotimeNs),
		Start:          int64(info.StartMonotimeNs),
		End:            int64(info.EndMonotimeNs),
		Status:         int(info.Status),
		TraceID:        info.Tp.TraceId,
		SpanID:         info.Tp.SpanId,
		ParentSpanID:   info.Tp.ParentId,
		TraceFlags:     info.Tp.Flags,
		Pid: request.PidInfo{
			HostPID:   info.Pid.HostPid,
			UserPID:   info.Pid.UserPid,
			Namespace: info.Pid.Ns,
		},
		Statement: scheme + request.SchemeHostSeparator + info.HeaderHost,
	}
}

func removeQuery(url string) string {
	idx := strings.IndexByte(url, '?')
	if idx > 0 {
		return url[:idx]
	}
	return url
}

type HTTPInfo struct {
	BPFHTTPInfo
	Method     string
	URL        string
	Host       string
	Peer       string
	HeaderHost string
}

func ReadHTTPInfoIntoSpan(record *ringbuf.Record, filter ServiceFilter) (request.Span, bool, error) {
	event, err := ReinterpretCast[BPFHTTPInfo](record.RawSample)
	if err != nil {
		return request.Span{}, true, err
	}

	// Generated by Go instrumentation
	if !filter.ValidPID(event.Pid.UserPid, event.Pid.Ns, PIDTypeKProbes) {
		return request.Span{}, true, nil
	}

	return HTTPInfoEventToSpan(event)
}

func HTTPInfoEventToSpan(event *BPFHTTPInfo) (request.Span, bool, error) {
	result := HTTPInfo{BPFHTTPInfo: *event}

	var bufHost string
	var bufPort int
	parsedHost := false

	// When we can't find the connection info, we signal that through making the
	// source and destination ports equal to max short. E.g. async SSL
	if event.ConnInfo.S_port != 0 || event.ConnInfo.D_port != 0 {
		source, target := (*BPFConnInfo)(&event.ConnInfo).reqHostInfo()
		result.Host = target
		result.Peer = source
	} else {
		bufHost, bufPort = event.hostFromBuf()
		parsedHost = true

		if bufPort >= 0 {
			result.Host = bufHost
			result.ConnInfo.D_port = uint16(bufPort)
		}
	}
	result.URL = event.url()
	result.Method = event.method()

	if request.EventType(result.Type) == request.EventTypeHTTPClient && !parsedHost {
		bufHost, _ = event.hostFromBuf()
	}

	result.HeaderHost = bufHost

	return httpInfoToSpan(&result), false, nil
}

func (event *BPFHTTPInfo) url() string {
	buf := string(event.Buf[:])
	space := strings.Index(buf, " ")
	if space < 0 {
		return ""
	}

	bufEnd := bytes.IndexByte(event.Buf[:], 0) // We assume the buffer was zero initialized in eBPF
	if bufEnd < 0 {
		bufEnd = len(buf)
	}

	if space+1 > bufEnd {
		return ""
	}

	nextSpace := strings.IndexAny(buf[space+1:bufEnd], " \r\n")
	if nextSpace < 0 {
		return buf[space+1 : bufEnd]
	}

	end := nextSpace + space + 1
	if end > bufEnd {
		end = bufEnd
	}

	return buf[space+1 : end]
}

func (event *BPFHTTPInfo) method() string {
	buf := string(event.Buf[:])
	space := strings.Index(buf, " ")
	if space < 0 {
		return ""
	}

	return buf[:space]
}

func (event *BPFHTTPInfo) hostFromBuf() (string, int) {
	buf := cstr(event.Buf[:])

	host := "Host: "
	idx := strings.Index(buf, host)

	if idx < 0 {
		return "", -1
	}

	buf = buf[idx+len(host):]

	rIdx := strings.Index(buf, "\r")

	if rIdx < 0 {
		rIdx = len(buf)
	}

	host, portStr, err := net.SplitHostPort(buf[:rIdx])
	if err != nil {
		return buf[:rIdx], -1
	}

	port, _ := strconv.Atoi(portStr)

	return host, port
}
