<?php
/*
 * @copyright 2024 Passwords App
 *
 * @author Marius David Wieschollek
 * @license AGPL-3.0
 *
 * This file is part of the Passwords App
 * created by Marius David Wieschollek.
 */

namespace OCA\Passwords\Controller\Api;

use Exception;
use OCA\Passwords\Db\FolderRevision;
use OCA\Passwords\Exception\ApiException;
use OCA\Passwords\Helper\ApiObjects\AbstractObjectHelper;
use OCA\Passwords\Helper\ApiObjects\FolderObjectHelper;
use OCA\Passwords\Services\EncryptionService;
use OCA\Passwords\Services\Object\AbstractModelService;
use OCA\Passwords\Services\Object\AbstractRevisionService;
use OCA\Passwords\Services\Object\FolderRevisionService;
use OCA\Passwords\Services\Object\FolderService;
use OCA\Passwords\Services\ValidationService;
use OCP\AppFramework\Db\DoesNotExistException;
use OCP\AppFramework\Db\MultipleObjectsReturnedException;
use OCP\AppFramework\Http;
use OCP\AppFramework\Http\Attribute\CORS;
use OCP\AppFramework\Http\Attribute\NoAdminRequired;
use OCP\AppFramework\Http\Attribute\NoCSRFRequired;
use OCP\AppFramework\Http\JSONResponse;
use OCP\IRequest;

/**
 * Class FolderApiController
 *
 * @package OCA\Passwords\Controller\Api
 */
class FolderApiController extends AbstractObjectApiController {

    /**
     * @var FolderService|AbstractModelService
     */
    protected AbstractModelService $modelService;

    /**
     * @var FolderObjectHelper|AbstractObjectHelper
     */
    protected AbstractObjectHelper $objectHelper;

    /**
     * @var FolderRevisionService|AbstractRevisionService
     */
    protected AbstractRevisionService $revisionService;

    /**
     * @var array
     */
    protected array $allowedFilterFields = ['created', 'updated', 'edited', 'cseType', 'sseType', 'trashed', 'favorite', 'parent'];

    /**
     * FolderApiController constructor.
     *
     * @param IRequest              $request
     * @param FolderService         $modelService
     * @param FolderObjectHelper    $objectHelper
     * @param ValidationService     $validationService
     * @param FolderRevisionService $revisionService
     */
    public function __construct(
        IRequest $request,
        FolderService $modelService,
        FolderObjectHelper $objectHelper,
        ValidationService $validationService,
        FolderRevisionService $revisionService
    ) {
        parent::__construct($request, $modelService, $objectHelper, $validationService, $revisionService);
    }

    /**
     * @param string $label
     * @param string $parent
     * @param string $cseKey
     * @param string $cseType
     * @param int    $edited
     * @param bool   $hidden
     * @param bool   $favorite
     *
     * @return JSONResponse
     * @throws ApiException
     * @throws Exception
     */
    #[CORS]
    #[NoCSRFRequired]
    #[NoAdminRequired]
    public function create(
        string $label = '',
        string $parent = FolderService::BASE_FOLDER_UUID,
        string $cseKey = '',
        string $cseType = EncryptionService::DEFAULT_CSE_ENCRYPTION,
        int $edited = 0,
        bool $hidden = false,
        bool $favorite = false
    ): JSONResponse {
        if($edited < 1) $edited = time();

        $model    = $this->modelService->create();
        $revision = $this->revisionService->create(
            $model->getUuid(), $label, $parent, $cseKey, $cseType, $edited, $hidden, false, $favorite
        );

        $this->revisionService->save($revision);
        $this->modelService->setRevision($model, $revision);

        return $this->createJsonResponse(
            ['id' => $model->getUuid(), 'revision' => $revision->getUuid()],
            Http::STATUS_CREATED
        );
    }

    /**
     * @param string $id
     * @param string $label
     * @param string|null $revision
     * @param string $parent
     * @param string $cseKey
     * @param string $cseType
     * @param int    $edited
     * @param bool   $hidden
     * @param bool   $favorite
     *
     * @return JSONResponse
     * @throws ApiException
     * @throws DoesNotExistException
     * @throws MultipleObjectsReturnedException
     * @throws Exception
     */
    #[CORS]
    #[NoCSRFRequired]
    #[NoAdminRequired]
    public function update(
        string $id,
        string $label,
        ?string $revision = null,
        string $parent = FolderService::BASE_FOLDER_UUID,
        string $cseKey = '',
        string $cseType = EncryptionService::DEFAULT_CSE_ENCRYPTION,
        int $edited = 0,
        bool $hidden = false,
        bool $favorite = false
    ): JSONResponse {
        if($id === $this->modelService::BASE_FOLDER_UUID) throw new ApiException('Can not edit base folder', Http::STATUS_UNPROCESSABLE_ENTITY);

        $model = $this->modelService->findByUuid($id);
        if($revision !== null && $revision !== $model->getRevision()) {
            throw new ApiException('Outdated revision id', Http::STATUS_CONFLICT);
        }

        /** @var FolderRevision $oldRevision */
        $oldRevision = $this->revisionService->findByUuid($model->getRevision());

        if($edited < 0) $edited = $oldRevision->getEdited();
        $revision = $this->revisionService->create(
            $model->getUuid(), $label, $parent, $cseKey, $cseType, $edited, $hidden, $oldRevision->isTrashed(), $favorite
        );

        $this->revisionService->save($revision);
        $this->modelService->setRevision($model, $revision);

        return $this->createJsonResponse(['id' => $model->getUuid(), 'revision' => $revision->getUuid()]);
    }

    /**
     * @param string      $id
     * @param string|null $revision
     *
     * @return JSONResponse
     * @throws ApiException
     * @throws DoesNotExistException
     * @throws MultipleObjectsReturnedException
     */
    #[CORS]
    #[NoCSRFRequired]
    #[NoAdminRequired]
    public function delete(string $id, ?string $revision = null): JSONResponse {
        if($id === $this->modelService::BASE_FOLDER_UUID) {
            throw new ApiException('Can not edit base folder', Http::STATUS_UNPROCESSABLE_ENTITY);
        }

        return parent::delete($id, $revision);
    }

    /**
     * @param string $id
     * @param null   $revision
     *
     * @return JSONResponse
     * @throws ApiException
     * @throws Exception
     * @throws DoesNotExistException
     * @throws MultipleObjectsReturnedException
     */
    #[CORS]
    #[NoCSRFRequired]
    #[NoAdminRequired]
    public function restore(string $id, $revision = null): JSONResponse {
        if($id === $this->modelService::BASE_FOLDER_UUID || $revision === $this->revisionService::BASE_REVISION_UUID) {
            throw new ApiException('Can not edit base folder', Http::STATUS_UNPROCESSABLE_ENTITY);
        }

        return parent::restore($id, $revision);
    }
}
