/*
    BLUES - BD-Java emulation server

    Copyright (C) 2007-2024 GuinpinSoft inc <blues@makemkv.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
package blues;

import java.lang.reflect.Method;
import java.util.HashMap;

public class Reflection {

    static HashMap<Method, Method> trapMethods = new HashMap<Method, Method>();

    public static synchronized Method getTrapMethod(Method method) {
        Method r = trapMethods.get(method);
        if (r != null) {
            return (r != method) ? r : null;
        }

        ClassInfo classInfo = getClassInfo(method.getDeclaringClass());
        if (classInfo == null) {
            trapMethods.put(method, method);
            return null;
        }

        String name = getMethodName(method);

        int flags = classInfo.find(name);

        if (flags == 0) {
            trapMethods.put(method, method);
            return null;
        }

        if ((flags & ClassInfo.TRAP_SPECIAL) != 0) {
            Class glueClass;
            try {
                glueClass = Class.forName("glue." + method.getDeclaringClass().getName());
            } catch (ClassNotFoundException e) {
                glueClass = null;
                Log.log(Log.LOG_ERROR, "Glue class not found for ", method.getDeclaringClass().getName());
                return null;
            }
            Class[] paramGlue;
            Class[] paramOrig = method.getParameterTypes();
            if ((flags & ClassInfo.STATIC) == 0) {
                paramGlue = new Class[paramOrig.length + 1];
                paramGlue[0] = Object.class;
                System.arraycopy(paramOrig, 0, paramGlue, 1, paramOrig.length);
            } else {
                paramGlue = paramOrig;
            }
            try {
                r = glueClass.getMethod(method.getName(), paramGlue);
            } catch (Exception e) {
                Log.log(Log.LOG_ERROR, "Glue method not found for ", method.toString());
                return null;
            }
            trapMethods.put(method, r);
            return r;
        }

        return r;
    }

    private static ClassInfo getClassInfo(Class clazz) {
        ClassMap map = Jail.map();
        String className = clazz.getName();
        return map.findEntry(className);
    }

    private static String getMethodName(Method method) {
        Class[] param = method.getParameterTypes();
        StringBuilder sb = new StringBuilder(256);
        sb.append(method.getName());
        sb.append('(');
        for (int i = 0; i < param.length; i++) {
            sb.append(getJvmType(param[i]));
        }
        sb.append(')');
        sb.append(getJvmType(method.getReturnType()));
        return sb.toString();
    }

    private static String getJvmType(Class clazz) {
        if (clazz.isPrimitive()) {
            if (clazz.equals(Byte.TYPE))
                return "B";
            if (clazz.equals(Character.TYPE))
                return "C";
            if (clazz.equals(Double.TYPE))
                return "D";
            if (clazz.equals(Float.TYPE))
                return "F";
            if (clazz.equals(Integer.TYPE))
                return "I";
            if (clazz.equals(Long.TYPE))
                return "J";
            if (clazz.equals(Short.TYPE))
                return "S";
            if (clazz.equals(Boolean.TYPE))
                return "Z";
        }
        if (clazz.isArray()) {
            return clazz.getName().replace('.', '/');
        }
        return "L" + Jail.classNameAddJail(clazz.getName()).replace('.', '/') + ";";
    }

}
